<?php
/**
 * CSSIgniter theme base functions and definitions
 */

if ( ! function_exists( 'andros_base_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function andros_base_setup() {
	// Make theme available for translation.
	load_theme_textdomain( 'andros', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	// Let WordPress manage the document title.
	add_theme_support( 'title-tag' );

	// Enable support for Post Thumbnails on posts and pages.
	add_theme_support( 'post-thumbnails' );

	register_nav_menus( array(
		'menu-1' => esc_html__( 'Main Menu', 'andros' ),
		'menu-2' => esc_html__( 'Main Menu - Right', 'andros' ),
	) );

	// Switch default core markup for search form, comment form, and comments to output valid HTML5.
	add_theme_support( 'html5', apply_filters( 'andros_add_theme_support_html5', array(
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) ) );

	// Add theme support for custom logos.
	add_theme_support( 'custom-logo', apply_filters( 'andros_add_theme_support_custom_logo', array() ) );

	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'andros_custom_background_args', array() ) );

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );
}
endif;

/**
 * Register scripts and styles unconditionally.
 */
function andros_base_register_scripts() {
	if ( ! wp_script_is( 'alpha-color-picker', 'enqueued' ) && ! wp_script_is( 'alpha-color-picker', 'registered' ) ) {
		wp_register_style( 'alpha-color-picker', get_theme_file_uri( '/base/assets/vendor/alpha-color-picker/alpha-color-picker.css' ), array(
			'wp-color-picker',
		), andros_asset_version( '1.0.0' ) );
		wp_register_script( 'alpha-color-picker', get_theme_file_uri( '/base/assets/vendor/alpha-color-picker/alpha-color-picker.js' ), array(
			'jquery',
			'wp-color-picker',
		), andros_asset_version( '1.0.0' ), true );
	}

	if ( ! wp_script_is( 'slick', 'enqueued' ) && ! wp_script_is( 'slick', 'registered' ) ) {
		wp_register_style( 'slick', get_theme_file_uri( '/base/assets/vendor/slick/slick.css' ), array(), andros_asset_version( '1.6.0' ) );
		wp_register_script( 'slick', get_theme_file_uri( '/base/assets/vendor/slick/slick.js' ), array(
			'jquery',
		), andros_asset_version( '1.6.0' ), true );
	}

	if ( ! wp_script_is( 'andros-plugin-post-meta', 'enqueued' ) && ! wp_script_is( 'andros-plugin-post-meta', 'registered' ) ) {
		wp_register_style( 'andros-plugin-post-meta', get_theme_file_uri( '/base/assets/css/post-meta.css' ), array(
			'alpha-color-picker',
		), andros_asset_version() );
		wp_register_script( 'andros-plugin-post-meta', get_theme_file_uri( '/base/assets/js/post-meta.js' ), array(
			'media-editor',
			'jquery',
			'jquery-ui-sortable',
			'alpha-color-picker',
		), andros_asset_version(), true );

		$settings = array(
			'ajaxurl'             => admin_url( 'admin-ajax.php' ),
			'tSelectFile'         => esc_html__( 'Select file', 'andros' ),
			'tSelectFiles'        => esc_html__( 'Select files', 'andros' ),
			'tUseThisFile'        => esc_html__( 'Use this file', 'andros' ),
			'tUseTheseFiles'      => esc_html__( 'Use these files', 'andros' ),
			'tUpdateGallery'      => esc_html__( 'Update gallery', 'andros' ),
			'tLoading'            => esc_html__( 'Loading...', 'andros' ),
			'tPreviewUnavailable' => esc_html__( 'Gallery preview not available.', 'andros' ),
			'tRemoveImage'        => esc_html__( 'Remove image', 'andros' ),
			'tRemoveFromGallery'  => esc_html__( 'Remove from gallery', 'andros' ),
		);
		wp_localize_script( 'andros-plugin-post-meta', 'andros_plugin_PostMeta', $settings );
	}

	wp_register_style( 'andros-repeating-fields', get_theme_file_uri( '/base/assets/repeating-fields/repeating-fields.css' ), array(), andros_asset_version() );
	wp_register_script( 'andros-repeating-fields', get_theme_file_uri( '/base/assets/repeating-fields/repeating-fields.js' ), array(
		'jquery',
		'jquery-ui-sortable',
	), andros_asset_version(), true );

	wp_register_style( 'font-awesome-5', get_theme_file_uri( '/base/assets/vendor/fontawesome/css/all.css' ), array(), andros_asset_version( '5.7.2' ) );

	wp_register_script( 'imagesLoaded', get_theme_file_uri( '/base/assets/js/imagesloaded.pkgd.min.js' ), array( 'jquery' ), andros_asset_version( '4.1.3' ), true );
	wp_register_script( 'anim-on-scroll', get_theme_file_uri( '/base/assets/js/anim-on-scroll.js' ), array(
		'jquery',
		'imagesLoaded',
	), andros_asset_version( '1.0.1' ), true );

	wp_register_script( 'andros-base-front-scripts', get_theme_file_uri( '/base/assets/js/scripts.js' ), array(
		'jquery',
	), andros_asset_version(), true );

	wp_register_style( 'jquery-magnific-popup', get_theme_file_uri( '/base/assets/vendor/magnific-popup/magnific.css' ), array(), andros_asset_version( '1.0.0' ) );
	wp_register_script( 'jquery-magnific-popup', get_theme_file_uri( '/base/assets/vendor/magnific-popup/jquery.magnific-popup.js' ), array( 'jquery' ), andros_asset_version( '1.0.0' ), true );
	wp_register_script( 'andros-magnific-init', get_theme_file_uri( '/base/assets/js/magnific-init.js' ), array( 'jquery' ), andros_asset_version(), true );

	wp_register_style( 'andros-base-style', '', apply_filters( 'andros_base_style_depends', array( 'andros-style' ) ), andros_asset_version() );

}
add_action( 'init', 'andros_base_register_scripts' );


/**
 * Enqueue scripts and styles.
 */
function andros_base_enqueue_scripts() {

	// Load the user-selected google fonts, if any.
	andros_enqueue_google_fonts();

	wp_enqueue_style( 'andros-base-style' );
	wp_add_inline_style( 'andros-base-style', andros_get_all_customizer_css() );

	wp_enqueue_script( 'andros-base-front-scripts' );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	if ( get_theme_mod( 'theme_lightbox', 1 ) ) {
		wp_enqueue_style( 'jquery-magnific-popup' );
		wp_enqueue_script( 'jquery-magnific-popup' );
		wp_enqueue_script( 'andros-magnific-init' );
	}

}
add_action( 'wp_enqueue_scripts', 'andros_base_enqueue_scripts' );


function andros_base_admin_enqueue_scripts( $hook ) {
	if ( in_array( $hook, array( 'widgets.php', 'customize.php' ), true ) ) {
		wp_enqueue_style( 'andros-repeating-fields' );
		wp_enqueue_script( 'andros-repeating-fields' );
	}
}
add_action( 'admin_enqueue_scripts', 'andros_base_admin_enqueue_scripts' );


function andros_base_register_sidebars() {
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 1st column', 'andros' ),
		'id'            => 'footer-1',
		'description'   => esc_html__( 'Widgets added here will appear on the first footer column.', 'andros' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 2nd column', 'andros' ),
		'id'            => 'footer-2',
		'description'   => esc_html__( 'Widgets added here will appear on the second footer column.', 'andros' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 3rd column', 'andros' ),
		'id'            => 'footer-3',
		'description'   => esc_html__( 'Widgets added here will appear on the third footer column.', 'andros' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 4th column', 'andros' ),
		'id'            => 'footer-4',
		'description'   => esc_html__( 'Widgets added here will appear on the fourth footer column.', 'andros' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
}
add_action( 'widgets_init', 'andros_base_register_sidebars' );

function andros_base_load_widgets() {
	require_once get_theme_file_path( '/base/widgets/socials.php' );
	require_once get_theme_file_path( '/base/widgets/contact.php' );
	require_once get_theme_file_path( '/base/widgets/schedule.php' );
	require_once get_theme_file_path( '/base/widgets/latest-post-type.php' );

	register_widget( 'CI_Widget_Socials' );
	register_widget( 'CI_Widget_Contact' );
	register_widget( 'CI_Widget_Schedule' );
	register_widget( 'CI_Widget_Latest_Post_Type' );
}
add_action( 'widgets_init', 'andros_base_load_widgets' );


if ( ! function_exists( 'andros_get_all_customizer_css' ) ) :
	function andros_get_all_customizer_css() {
		$styles = array(
			'base-customizer' => andros_base_get_customizer_css(),
			'customizer'      => andros_get_customizer_css(),
			'customizer-part' => andros_get_part_customizer_css(),
			'base-hero'       => andros_base_get_hero_styles(),
		);

		$styles = apply_filters( 'andros_all_customizer_css', $styles );

		if ( is_customize_preview() ) {
			$styles[] = '/* Placeholder for preview. */';
		}

		return implode( PHP_EOL, $styles );
	}
endif;

if ( ! function_exists( 'andros_has_sidebar' ) ) :
/**
 * Determine if a sidebar is being displayed.
 */
function andros_has_sidebar() {
	$has_sidebar = false;

	if ( is_home() || is_archive() ) {
		if ( get_theme_mod( 'archive_sidebar', 1 ) && is_active_sidebar( 'sidebar-1' ) ) {
			$has_sidebar = true;
		}
	} elseif ( ! is_page() && is_active_sidebar( 'sidebar-1' ) ) {
		$has_sidebar = true;
	} elseif ( is_page() && is_active_sidebar( 'sidebar-2' ) ) {
		$has_sidebar = true;
	}

	return apply_filters( 'andros_has_sidebar', $has_sidebar );
}
endif;

if ( ! function_exists( 'andros_get_layout_info' ) ) :
/**
 * Return appropriate layout information.
 */
function andros_get_layout_info() {
	$has_sidebar = andros_has_sidebar();

	$classes = array(
		'container_classes' => $has_sidebar ? 'col-lg-8 col-12' : 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
		'sidebar_classes'   => $has_sidebar ? 'col-xl-3 offset-xl-1 col-lg-4 col-12' : '',
		'content_width'     => 750,
		'has_sidebar'       => $has_sidebar,
	);

	if ( is_singular() ) {
		if ( 'left' === get_post_meta( get_the_ID(), 'andros_sidebar', true ) ) {
			$classes = array(
				'container_classes' => $has_sidebar ? 'col-lg-8 push-lg-4 col-12' : 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
				'sidebar_classes'   => $has_sidebar ? 'col-xl-3 offset-xl-1 pull-xl-9 col-lg-4 pull-lg-8 col-12' : '',
				'content_width'     => 750,
				'has_sidebar'       => $has_sidebar,
			);
		} elseif ( 'none' === get_post_meta( get_the_ID(), 'andros_sidebar', true ) ) {
			$classes = array(
				'container_classes' => 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
				'sidebar_classes'   => '',
				'content_width'     => 750,
				'has_sidebar'       => false,
			);
		}
	} elseif ( is_home() || is_archive() ) {
		// 1 will get default narrow fullwidth classes. 2 and 3 will get fullwidth.
		if ( 1 !== (int) get_theme_mod( 'archive_layout', andros_archive_layout_default() ) ) {
			if ( ! $has_sidebar ) {
				$classes = array(
					'container_classes' => 'col-12',
					'sidebar_classes'   => '',
					'content_width'     => 1140,
					'has_sidebar'       => false,
				);
			}
		}
	}

	return apply_filters( 'andros_layout_info', $classes, $has_sidebar );
}
endif;

/**
 * Echoes container classes based on whether
 * the current template has a visible sidebar or not
 */
function andros_the_container_classes() {
	$info = andros_get_layout_info();
	echo esc_attr( $info['container_classes'] );
}

/**
 * Echoes container classes based on whether
 * the current template has a visible sidebar or not
 */
function andros_the_sidebar_classes() {
	$info = andros_get_layout_info();
	echo esc_attr( $info['sidebar_classes'] );
}

/**
 * Returns an array of the supported social networks and their properties.
 *
 * @return array
 */
function andros_get_social_networks() {
	return apply_filters( 'andros_social_networks', array(
		array(
			'name'  => 'facebook',
			'label' => esc_html__( 'Facebook', 'andros' ),
			'icon'  => 'fab fa-facebook',
		),
		array(
			'name'  => 'twitter',
			'label' => esc_html__( 'Twitter', 'andros' ),
			'icon'  => 'fab fa-twitter',
		),
		array(
			'name'  => 'instagram',
			'label' => esc_html__( 'Instagram', 'andros' ),
			'icon'  => 'fab fa-instagram',
		),
		array(
			'name'  => 'pinterest',
			'label' => esc_html__( 'Pinterest', 'andros' ),
			'icon'  => 'fab fa-pinterest',
		),
		array(
			'name'  => 'snapchat',
			'label' => esc_html__( 'Snapchat', 'andros' ),
			'icon'  => 'fab fa-snapchat',
		),
		array(
			'name'  => 'behance',
			'label' => esc_html__( 'Behance', 'andros' ),
			'icon'  => 'fab fa-behance',
		),
		array(
			'name'  => 'dribbble',
			'label' => esc_html__( 'Dribbble', 'andros' ),
			'icon'  => 'fab fa-dribbble',
		),
		array(
			'name'  => 'youtube',
			'label' => esc_html__( 'YouTube', 'andros' ),
			'icon'  => 'fab fa-youtube',
		),
		array(
			'name'  => 'etsy',
			'label' => esc_html__( 'Etsy', 'andros' ),
			'icon'  => 'fab fa-etsy',
		),
		array(
			'name'  => 'flickr',
			'label' => esc_html__( 'Flickr', 'andros' ),
			'icon'  => 'fab fa-flickr',
		),
		array(
			'name'  => 'github',
			'label' => esc_html__( 'GitHub', 'andros' ),
			'icon'  => 'fab fa-github',
		),
		array(
			'name'  => 'linkedin',
			'label' => esc_html__( 'LinkedIn', 'andros' ),
			'icon'  => 'fab fa-linkedin',
		),
		array(
			'name'  => 'medium',
			'label' => esc_html__( 'Medium', 'andros' ),
			'icon'  => 'fab fa-medium',
		),
		array(
			'name'  => 'mixcloud',
			'label' => esc_html__( 'Mixcloud', 'andros' ),
			'icon'  => 'fab fa-mixcloud',
		),
		array(
			'name'  => 'paypal',
			'label' => esc_html__( 'PayPal', 'andros' ),
			'icon'  => 'fab fa-paypal',
		),
		array(
			'name'  => 'skype',
			'label' => esc_html__( 'Skype', 'andros' ),
			'icon'  => 'fab fa-skype',
		),
		array(
			'name'  => 'slack',
			'label' => esc_html__( 'Slack', 'andros' ),
			'icon'  => 'fab fa-slack',
		),
		array(
			'name'  => 'soundcloud',
			'label' => esc_html__( 'Soundcloud', 'andros' ),
			'icon'  => 'fab fa-soundcloud',
		),
		array(
			'name'  => 'spotify',
			'label' => esc_html__( 'Spotify', 'andros' ),
			'icon'  => 'fab fa-spotify',
		),
		array(
			'name'  => 'vimeo',
			'label' => esc_html__( 'Vimeo', 'andros' ),
			'icon'  => 'fab fa-vimeo',
		),
		array(
			'name'  => 'wordpress',
			'label' => esc_html__( 'WordPress', 'andros' ),
			'icon'  => 'fab fa-wordpress',
		),
		array(
			'name'  => 'xbox',
			'label' => esc_html__( 'Xbox Live', 'andros' ),
			'icon'  => 'fab fa-xbox',
		),
		array(
			'name'  => 'playstation',
			'label' => esc_html__( 'PlayStation Network', 'andros' ),
			'icon'  => 'fab fa-playstation',
		),
		array(
			'name'  => 'bloglovin',
			'label' => esc_html__( 'Bloglovin', 'andros' ),
			'icon'  => 'fas fa-heart',
		),
		array(
			'name'  => 'tumblr',
			'label' => esc_html__( 'Tumblr', 'andros' ),
			'icon'  => 'fab fa-tumblr',
		),
		array(
			'name'  => '500px',
			'label' => esc_html__( '500px', 'andros' ),
			'icon'  => 'fab fa-500px',
		),
		array(
			'name'  => 'tripadvisor',
			'label' => esc_html__( 'Trip Advisor', 'andros' ),
			'icon'  => 'fab fa-tripadvisor',
		),
		array(
			'name'  => 'telegram',
			'label' => esc_html__( 'Telegram', 'andros' ),
			'icon'  => 'fab fa-telegram',
		),
	) );
}


add_filter( 'tiny_mce_before_init', 'andros_base_insert_wp_editor_formats' );
function andros_base_insert_wp_editor_formats( $init_array ) {
	$style_formats = array(
		array(
			'title'   => esc_html__( 'Intro text (big text)', 'andros' ),
			'block'   => 'div',
			'classes' => 'entry-content-intro',
			'wrapper' => true,
		),
		array(
			'title'   => esc_html__( '2 Column Text', 'andros' ),
			'block'   => 'div',
			'classes' => 'entry-content-column-split',
			'wrapper' => true,
		),
	);

	$init_array['style_formats'] = wp_json_encode( $style_formats );

	return $init_array;
}

add_filter( 'mce_buttons_2', 'andros_base_mce_buttons_2' );
function andros_base_mce_buttons_2( $buttons ) {
	array_unshift( $buttons, 'styleselect' );

	return $buttons;
}


if ( ! function_exists( 'andros_color_luminance' ) ) :
	/**
	 * Lightens/darkens a given colour (hex format), returning the altered colour in hex format.
	 *
	 * @see https://gist.github.com/stephenharris/5532899
	 *
	 * @param string $color Hexadecimal color value. May be 3 or 6 digits, with an optional leading # sign.
	 * @param float $percent Decimal (0.2 = lighten by 20%, -0.4 = darken by 40%)
	 *
	 * @return string Lightened/Darkened colour as hexadecimal (with hash)
	 */
	function andros_color_luminance( $color, $percent ) {
		// Remove # if provided
		if ( '#' === $color[0] ) {
			$color = substr( $color, 1 );
		}

		// Validate hex string.
		$hex     = preg_replace( '/[^0-9a-f]/i', '', $color );
		$new_hex = '#';

		$percent = floatval( $percent );

		if ( strlen( $hex ) < 6 ) {
			$hex = $hex[0] + $hex[0] + $hex[1] + $hex[1] + $hex[2] + $hex[2];
		}

		// Convert to decimal and change luminosity.
		for ( $i = 0; $i < 3; $i ++ ) {
			$dec      = hexdec( substr( $hex, $i * 2, 2 ) );
			$dec      = min( max( 0, $dec + $dec * $percent ), 255 );
			$new_hex .= str_pad( dechex( $dec ), 2, 0, STR_PAD_LEFT );
		}

		return $new_hex;
	}
endif;

if ( ! function_exists( 'andros_hex2rgba' ) ) :
	/**
	 * Converts hexadecimal color value to rgb(a) format.
	 *
	 * @param string $color Hexadecimal color value. May be 3 or 6 digits, with an optional leading # sign.
	 * @param float|bool $opacity Opacity level 0-1 (decimal) or false to disable.
	 *
	 * @return string
	 */
	function andros_hex2rgba( $color, $opacity = false ) {

		$default = 'rgb(0,0,0)';

		// Return default if no color provided
		if ( empty( $color ) ) {
			return $default;
		}

		// Remove # if provided
		$color = ltrim( $color, '#' );

		// Check if color has 6 or 3 characters and get values
		if ( strlen( $color ) === 6 ) {
			$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
		} elseif ( strlen( $color ) === 3 ) {
			$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
		} else {
			return $default;
		}

		$rgb = array_map( 'hexdec', $hex );

		if ( false !== $opacity ) {
			$opacity = abs( floatval( $opacity ) );
			if ( $opacity > 1 ) {
				$opacity = 1.0;
			}
			$output = 'rgba(' . implode( ',', $rgb ) . ',' . $opacity . ')';
		} else {
			$output = 'rgb(' . implode( ',', $rgb ) . ')';
		}

		return $output;
	}
endif;


/**
 * Return default args for add_theme_support( 'andros-hero' )
 *
 * Used when declaring support for theme hero section, so that unchanged args can be omitted. E.g.:
 *
 *  	add_theme_support( 'andros-hero', apply_filters( 'andros_theme_support_hero_defaults', wp_parse_args( array(
 *  		'required' => true,
 *  	), andros_base_theme_support_hero_defaults() ) ) );
 *
 * @return array
 */
function andros_base_theme_support_hero_defaults() {
	return apply_filters( 'andros_base_theme_support_hero_defaults', array(
		'required'              => false, // When true, there will be no option to hide the hero section.
		'show-default'          => false, // The default state of the 'hero_show' option.
		'image-size'            => 'andros_hero', // The default image size for the background image.
		'front-page-template'   => 'templates/front-page.php', // The front page template slug. Set to false if theme doesn't have a front page template.
		'front-page-classes'    => '', // Extra hero classes for the front page.
		'front-page-image-size' => false, // The image size for the front page, if different. False means same as 'image-size'.
		'text-align'            => 'left', // The default text-align for the hero text. One of: 'left', 'center', 'right'.
	) );
}

function andros_the_hero_classes( $echo = true ) {
	$classes = array( 'page-hero' );

	$hero_support = get_theme_support( 'andros-hero' );
	$hero_support = $hero_support[0];
	if ( $hero_support['front-page-template'] && is_page_template( $hero_support['front-page-template'] ) ) {
		$classes[] = $hero_support['front-page-classes'];
	}

	$classes = apply_filters( 'andros_hero_classes', $classes );
	$classes = array_filter( $classes );
	if ( $echo ) {
		echo esc_attr( implode( ' ', $classes ) );
	} else {
		return $classes;
	}
}

function andros_get_hero_data( $post_id = false ) {
	if ( is_singular() && false === $post_id ) {
		$post_id = get_the_ID();
	}

	if ( ! current_theme_supports( 'andros-hero' ) ) {
		return array(
			'show'            => 0,
			'page_title_hide' => 0,
		);
	}

	$support = get_theme_support( 'andros-hero' );
	$support = $support[0];

	$video_url = get_theme_mod( 'hero_video_url' );

	$title    = '';
	$subtitle = '';

	$image_size = $support['image-size'];
	if ( $support['front-page-image-size'] && is_page_template( $support['front-page-template'] ) ) {
		$image_size = $support['front-page-image-size'];
	}

	if ( is_home() ) {
		$title = get_theme_mod( 'title_blog', __( 'From the blog', 'andros' ) );
	} elseif ( is_search() ) {
		$title = get_theme_mod( 'title_search', __( 'Search results', 'andros' ) );

		global $wp_query;
		$found = intval( $wp_query->found_posts );
		/* translators: %d is the number of search results. */
		$subtitle = esc_html( sprintf( _n( '%d result found.', '%d results found.', $found, 'andros' ), $found ) );

	} elseif ( is_404() ) {
		$title = get_theme_mod( 'title_404', __( 'Page not found', 'andros' ) );
	} elseif ( is_category() || is_tag() || is_tax() ) {
		$title = single_term_title( '', false );
	} elseif ( is_post_type_archive() ) {
		$title = post_type_archive_title( '', false );
	} elseif ( is_archive() ) {
		$title = get_the_archive_title();
	}

	$generic_data = array(
		'show'             => get_theme_mod( 'hero_show', $support['show-default'] ),
		'title'            => $title,
		'subtitle'         => $subtitle,
		'text_align'       => $support['text-align'],
		'page_title_hide'  => 0,
		'bg_color'         => get_theme_mod( 'hero_bg_color' ),
		'text_color'       => get_theme_mod( 'hero_text_color' ),
		'overlay_color'    => get_theme_mod( 'hero_overlay_color' ),
		'image_id'         => '',
		'image'            => get_theme_mod( 'hero_image' ),
		'image_repeat'     => get_theme_mod( 'hero_image_repeat', 'no-repeat' ),
		'image_position_x' => get_theme_mod( 'hero_image_position_x', 'center' ),
		'image_position_y' => get_theme_mod( 'hero_image_position_y', 'center' ),
		'image_attachment' => get_theme_mod( 'hero_image_attachment', 'scroll' ),
		'image_cover'      => get_theme_mod( 'hero_image_cover', 1 ),
		'video_url'        => $video_url,
		'video_info'       => andros_get_video_url_info( $video_url ),
	);

	$data = $generic_data;

	$single_data = array();

	if ( is_singular() || false !== $post_id ) {
		$image_id  = get_post_meta( $post_id, 'hero_image_id', true );
		$video_url = get_post_meta( $post_id, 'hero_video_url', true );

		$single_data = array(
			'show'             => get_post_meta( $post_id, 'hero_show', true ),
			'title'            => get_the_title(), // May be custom title from hooked andros_base_replace_the_title()
			'subtitle'         => get_post_meta( $post_id, 'subtitle', true ),
			'text_align'       => get_post_meta( $post_id, 'hero_text_align', true ),
			'page_title_hide'  => get_post_meta( $post_id, 'page_title_hide', true ),
			'bg_color'         => get_post_meta( $post_id, 'hero_bg_color', true ),
			'text_color'       => get_post_meta( $post_id, 'hero_text_color', true ),
			'overlay_color'    => get_post_meta( $post_id, 'hero_overlay_color', true ),
			'video_url'        => $video_url,
			'video_info'       => andros_get_video_url_info( $video_url ),
			'image_id'         => $image_id,
			'image'            => $image_id ? wp_get_attachment_image_url( $image_id, $image_size ) : '',
			'image_repeat'     => get_post_meta( $post_id, 'hero_image_repeat', true ),
			'image_position_x' => get_post_meta( $post_id, 'hero_image_position_x', true ),
			'image_position_y' => get_post_meta( $post_id, 'hero_image_position_y', true ),
			'image_attachment' => get_post_meta( $post_id, 'hero_image_attachment', true ),
			'image_cover'      => get_post_meta( $post_id, 'hero_image_cover', true ),
		);

		if ( ! $single_data['page_title_hide'] ) {
			if ( is_singular( 'post' ) || ( false === $post_id && 'post' === get_post_type( $post_id ) ) ) {
				$single_data['title'] = __( 'From the blog', 'andros' );
			}
		}

		$data = $single_data;

		// Hero is required, so lets inherit some values for best appearance.
		if ( $support['required'] ) {
			if ( empty( $data['text_align'] ) ) {
				$data['text_align'] = $generic_data['text_align'];
			}
			if ( empty( $data['bg_color'] ) ) {
				$data['bg_color'] = $generic_data['bg_color'];
			}
			if ( empty( $data['text_color'] ) ) {
				$data['text_color'] = $generic_data['text_color'];
			}
			if ( empty( $data['overlay_color'] ) ) {
				$data['overlay_color'] = $generic_data['overlay_color'];
			}
			if ( empty( $data['video_url'] ) ) {
				$data['video_url']  = $generic_data['video_url'];
				$data['video_info'] = $generic_data['video_info'];
			}
			if ( empty( $data['image_id'] ) ) {
				$data['image_id']         = $generic_data['image_id'];
				$data['image']            = $generic_data['image'];
				$data['image_repeat']     = $generic_data['image_repeat'];
				$data['image_position_x'] = $generic_data['image_position_x'];
				$data['image_position_y'] = $generic_data['image_position_y'];
				$data['image_attachment'] = $generic_data['image_attachment'];
				$data['image_cover']      = $generic_data['image_cover'];
			}
		}
	}

	if ( $support['required'] ) {
		$data['show'] = 1;
	}

	return apply_filters( 'andros_hero_data', $data, $post_id, $generic_data, $single_data );
}


function andros_get_video_url_info( $url ) {
	$is_vimeo   = preg_match( '#(?:https?://)?(?:www\.)?vimeo\.com/([A-Za-z0-9\-_]+)#', $url, $vimeo_id );
	$is_youtube = preg_match( '~
		# Match non-linked youtube URL in the wild. (Rev:20111012)
		https?://         # Required scheme. Either http or https.
		(?:[0-9A-Z-]+\.)? # Optional subdomain.
		(?:               # Group host alternatives.
		  youtu\.be/      # Either youtu.be,
		| youtube\.com    # or youtube.com followed by
		  \S*             # Allow anything up to VIDEO_ID,
		  [^\w\-\s]       # but char before ID is non-ID char.
		)                 # End host alternatives.
		([\w\-]{11})      # $1: VIDEO_ID is exactly 11 chars.
		(?=[^\w\-]|$)     # Assert next char is non-ID or EOS.
		(?!               # Assert URL is not pre-linked.
		  [?=&+%\w]*      # Allow URL (query) remainder.
		  (?:             # Group pre-linked alternatives.
			[\'"][^<>]*>  # Either inside a start tag,
		  | </a>          # or inside <a> element text contents.
		  )               # End recognized pre-linked alts.
		)                 # End negative lookahead assertion.
		[?=&+%\w-]*        # Consume any URL (query) remainder.
		~ix',
	$url, $youtube_id );

	$info = array(
		'supported' => false,
		'provider'  => '',
		'video_id'  => '',
	);

	if ( $is_youtube ) {
		$info['supported'] = true;
		$info['provider']  = 'youtube';
		$info['video_id']  = $youtube_id[1];
	} elseif ( $is_vimeo ) {
		$info['supported'] = true;
		$info['provider']  = 'vimeo';
		$info['video_id']  = $vimeo_id[1];
	}

	return $info;
}


/**
 * Returns a set of related posts, or the arguments needed for such a query.
 *
 * @uses wp_parse_args()
 * @uses get_post_type()
 * @uses get_post()
 * @uses get_object_taxonomies()
 * @uses get_the_terms()
 * @uses wp_list_pluck()
 *
 * @param int $post_id A post ID to get related posts for.
 * @param int $related_count The number of related posts to return.
 * @param array $args Array of arguments to change the default behavior.
 * @return object|array A WP_Query object with the results, or an array with the query arguments.
 */
function andros_get_related_posts( $post_id, $related_count, $args = array() ) {
	$args = wp_parse_args( (array) $args, array(
		'orderby' => 'rand',
		'return'  => 'query', // Valid values are: 'query' (WP_Query object), 'array' (the arguments array)
	) );

	$post_type = get_post_type( $post_id );
	$post      = get_post( $post_id );

	$tax_query  = array();
	$taxonomies = get_object_taxonomies( $post, 'names' );

	foreach ( $taxonomies as $taxonomy ) {
		$terms = get_the_terms( $post_id, $taxonomy );
		if ( is_array( $terms ) && count( $terms ) > 0 ) {
			$term_list = wp_list_pluck( $terms, 'slug' );
			$term_list = array_values( $term_list );
			if ( ! empty( $term_list ) ) {
				$tax_query['tax_query'][] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $term_list,
				);
			}
		}
	}

	if ( count( $taxonomies ) > 1 ) {
		$tax_query['tax_query']['relation'] = 'OR';
	}

	$query_args = array(
		'post_type'      => $post_type,
		'posts_per_page' => $related_count,
		'post_status'    => 'publish',
		'post__not_in'   => array( $post_id ),
		'orderby'        => $args['orderby'],
	);

	if ( 'query' === $args['return'] ) {
		return new WP_Query( array_merge( $query_args, $tax_query ) );
	} else {
		return array_merge( $query_args, $tax_query );
	}
}


/**
 * Returns the appropriate page(d) query variable to use in custom loops (needed for pagination).
 *
 * @uses get_query_var()
 *
 * @param int $default_return The default page number to return, if no query vars are set.
 * @return int The appropriate paged value if found, else 0.
 */
function andros_get_page_var( $default_return = 0 ) {
	$paged = get_query_var( 'paged', false );
	$page  = get_query_var( 'page', false );

	if ( false === $paged && false === $page ) {
		return $default_return;
	}

	return max( $paged, $page );
}


/**
 * Retrieve or display list of posts as a dropdown (select list).
 *
 * @since 2.1.0
 *
 * @param array|string $args Optional. Override default arguments.
 * @param string $name Optional. Name of the select box.
 * @return string HTML content, if not displaying.
 */
function andros_dropdown_posts( $args = '', $name = 'post_id' ) {
	$defaults = array(
		'depth'                 => 0,
		'post_parent'           => 0,
		'selected'              => 0,
		'echo'                  => 1,
		//'name'                  => 'page_id', // With this line, get_posts() doesn't work properly.
		'id'                    => '',
		'class'                 => '',
		'show_option_none'      => '',
		'show_option_no_change' => '',
		'option_none_value'     => '',
		'post_type'             => 'post',
		'post_status'           => 'publish',
		'suppress_filters'      => false,
		'numberposts'           => -1,
		'select_even_if_empty'  => false, // If no posts are found, an empty <select> will be returned/echoed.
	);

	$r = wp_parse_args( $args, $defaults );
	extract( $r, EXTR_SKIP );

	$hierarchical_post_types = get_post_types( array( 'hierarchical' => true ) );
	if ( in_array( $r['post_type'], $hierarchical_post_types ) ) {
		$pages = get_pages($r);
	} else {
		$pages = get_posts($r);
	}

	$output = '';
	// Back-compat with old system where both id and name were based on $name argument
	if ( empty($id) )
		$id = $name;

	if ( ! empty($pages) || $select_even_if_empty == true ) {
		$output = "<select name='" . esc_attr( $name ) . "' id='" . esc_attr( $id ) . "' class='" . esc_attr( $class ) . "'>\n";
		if ( $show_option_no_change ) {
			$output .= "\t<option value=\"-1\">$show_option_no_change</option>";
		}
		if ( $show_option_none ) {
			$output .= "\t<option value=\"" . esc_attr( $option_none_value ) . "\">$show_option_none</option>\n";
		}
		if ( ! empty($pages) ) {
			$output .= walk_page_dropdown_tree($pages, $depth, $r);
		}
		$output .= "</select>\n";
	}

	$output = apply_filters( 'andros_dropdown_posts', $output, $name, $r );

	if ( $echo )
		echo $output;

	return $output;
}


add_action( 'admin_init', 'andros_admin_setup_hide_single_featured' );
function andros_admin_setup_hide_single_featured() {
	if ( current_theme_supports( 'andros-hide-single-featured' ) ) {
		$hide_featured_support = get_theme_support( 'andros-hide-single-featured' );
		$hide_featured_support = $hide_featured_support[0];

		foreach ( $hide_featured_support as $supported_post_type ) {
			add_meta_box( 'andros-single-featured-visibility', esc_html__( 'Featured Image Visibility', 'andros' ), 'andros_single_featured_visibility_metabox', $supported_post_type, 'side', 'default' );
		}
	}

	add_action( 'save_post', 'andros_hide_single_featured_save_post' );
}

add_action( 'init', 'andros_setup_hide_single_featured' );
function andros_setup_hide_single_featured() {
	if ( current_theme_supports( 'andros-hide-single-featured' ) ) {
		add_filter( 'get_post_metadata', 'andros_hide_single_featured_get_post_metadata', 10, 4 );
	}
}

function andros_single_featured_visibility_metabox( $object, $box ) {
	$fieldname = 'andros_hide_single_featured';
	$checked   = get_post_meta( $object->ID, $fieldname, true );

	?>
		<input type="checkbox" id="<?php echo esc_attr( $fieldname ); ?>" class="check" name="<?php echo esc_attr( $fieldname ); ?>" value="1" <?php checked( $checked, 1 ); ?> />
		<label for="<?php echo esc_attr( $fieldname ); ?>"><?php esc_html_e( "Hide when viewing this post's page", 'andros' ); ?></label>
	<?php
	wp_nonce_field( 'andros_hide_single_featured_nonce', '_andros_hide_single_featured_meta_box_nonce' );
}

function andros_hide_single_featured_get_post_metadata( $value, $post_id, $meta_key, $single ) {
	$hide_featured_support = get_theme_support( 'andros-hide-single-featured' );
	$hide_featured_support = $hide_featured_support[0];

	if ( ! in_array( get_post_type( $post_id ), $hide_featured_support, true ) ) {
		return $value;
	}

	if ( '_thumbnail_id' === $meta_key && ( is_single( $post_id ) || is_page( $post_id ) ) && get_post_meta( $post_id, 'andros_hide_single_featured', true ) ) {
		return false;
	}

	return $value;
}

function andros_hide_single_featured_save_post( $post_id ) {
	$hide_featured_support = get_theme_support( 'andros-hide-single-featured' );
	$hide_featured_support = $hide_featured_support[0];

	if ( ! in_array( get_post_type( $post_id ), $hide_featured_support, true ) ) {
		return;
	}

	if ( isset( $_POST['_andros_hide_single_featured_meta_box_nonce'] ) && wp_verify_nonce( $_POST['_andros_hide_single_featured_meta_box_nonce'], 'andros_hide_single_featured_nonce' ) ) { // Input vars okay.
		update_post_meta( $post_id, 'andros_hide_single_featured', isset( $_POST['andros_hide_single_featured'] ) ); // Input var okay.
	}
}



function andros_archive_layout_choices() {
	/* translators: %d is a number of columns. */
	$nooped = _n_noop( '%d Column', '%d Columns', 'andros' );

	return apply_filters( 'andros_archive_layout_choices', array(
		1 => sprintf( translate_nooped_plural( $nooped, 1, 'andros' ), 1 ),
		2 => sprintf( translate_nooped_plural( $nooped, 2, 'andros' ), 2 ),
		3 => sprintf( translate_nooped_plural( $nooped, 3, 'andros' ), 3 ),
	) );
}

function andros_archive_layout_default() {
	return apply_filters( 'andros_archive_layout_default', 1 );
}

function andros_sanitize_archive_layout( $value ) {
	$choices = andros_archive_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return andros_archive_layout_default();
}

function andros_header_layout_choices() {
	return apply_filters( 'andros_header_layout_choices', array(
		'right' => esc_html__( 'Left logo - Right menu', 'andros' ),
		'split' => esc_html__( 'Centered logo - Split menu', 'andros' ),
		'full'  => esc_html__( 'Top logo - Bottom menu', 'andros' ),
	) );
}

function andros_header_layout_default() {
	return apply_filters( 'andros_header_layout_default', 'right' );
}

function andros_sanitize_header_layout( $value ) {
	$choices = andros_header_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return andros_header_layout_default();
}

function andros_header_logo_alignment_choices() {
	return apply_filters( 'andros_header_logo_alignment_choices', array(
		'left'   => esc_html__( 'Left', 'andros' ),
		'center' => esc_html__( 'Center', 'andros' ),
	) );
}

function andros_header_logo_alignment_default() {
	return apply_filters( 'andros_header_logo_alignment_default', 'center' );
}

function andros_sanitize_header_logo_alignment( $value ) {
	$choices = andros_header_logo_alignment_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return andros_header_logo_alignment_default();
}


function andros_footer_layout_choices() {
	/* translators: %d is a number of columns. */
	$nooped = _n_noop( '%d Column', '%d Columns', 'andros' );

	return apply_filters( 'andros_footer_layout_choices', array(
		'4-col' => sprintf( translate_nooped_plural( $nooped, 4, 'andros' ), 4 ),
		'3-col' => sprintf( translate_nooped_plural( $nooped, 3, 'andros' ), 3 ),
		'2-col' => sprintf( translate_nooped_plural( $nooped, 2, 'andros' ), 2 ),
		'1-col' => sprintf( translate_nooped_plural( $nooped, 1, 'andros' ), 1 ),
		'1-3'   => esc_html__( '1/4 - 3/4 Columns', 'andros' ),
		'3-1'   => esc_html__( '3/4 - 1/4 Columns', 'andros' ),
		'1-1-2' => esc_html__( '1/4 - 1/4 - 1/2 Columns', 'andros' ),
		'2-1-1' => esc_html__( '1/2 - 1/4 - 1/4 Columns', 'andros' ),
	) );
}

function andros_footer_layout_default() {
	return apply_filters( 'andros_footer_layout_default', '4-col' );
}

function andros_sanitize_footer_layout( $value ) {
	$choices = andros_footer_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return andros_footer_layout_default();
}

/**
 * Sanitizes the pagination method option.
 *
 * @param string $option Value to sanitize. Either 'numbers' or 'text'.
 * @return string
 */
function andros_sanitize_pagination_method( $option ) {
	if ( in_array( $option, array( 'numbers', 'text' ), true ) ) {
		return $option;
	}

	return andros_pagination_method_default();
}

function andros_pagination_method_default() {
	return apply_filters( 'andros_pagination_method_default', 'numbers' );
}

function andros_footer_widget_area_classes( $layout ) {
	switch ( $layout ) {
		case '3-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '2-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-12',
				),
				'footer-2' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-3':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-9 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '3-1':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-9 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-1-2':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-6 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '2-1-1':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '4-col':
		default:
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-4' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
			);
	}

	return apply_filters( 'andros_footer_widget_area_classes', $classes, $layout );
}

/**
 * Conditionally returns a Javascript/CSS asset's version number.
 *
 * When the site is in debug mode, the normal asset's version is returned.
 * When it's not in debug mode, the theme's version is returned, so that caches can be invalidated on theme updates.
 *
 * @param bool $version The version string of the asset.
 *
 * @return false|string Theme version if SCRIPT_DEBUG or WP_DEBUG are enabled. Otherwise, $version is returned.
 */
function andros_asset_version( $version = false ) {
	static $theme_version = false;

	if ( ! $theme_version ) {
		$theme = wp_get_theme();

		if ( is_child_theme() ) {
			$theme_version = $theme->parent()->get( 'Version' ) . '-' . $theme->get( 'Version' );
		} else {
			$theme_version = $theme->get( 'Version' );
		}
	}

	if ( $version ) {
		if ( ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ||
			( defined( 'WP_DEBUG' ) && WP_DEBUG )
		) {
			return $version;
		}
	}

	return $theme_version;
}

/**
 * Generic fallback callback for the main menu.
 *
 * Displays a few useful links, in contrast to the default wp_page_menu() which may flood the menu area.
 *
 * @param array $args
 *
 * @return void|string
 */
function andros_main_menu_fallback( $args ) {
	$id    = ! empty( $args['menu_id'] ) ? $args['menu_id'] : 'menu-' . $args['theme_location'];
	$class = $args['menu_class'];

	$items = apply_filters( 'andros_main_menu_fallback_items', array(
		home_url( '/' )              => __( 'Home', 'andros' ),
		admin_url( 'nav-menus.php' ) => __( 'Set primary menu', 'andros' ),
	) );

	$items_html = '';
	if ( $items ) {
		foreach ( $items as $item_url => $item_text ) {
			$items_html .= sprintf( '<li><a href="%1$s">%2$s</a></li>', esc_url( $item_url ), esc_html( $item_text ) );
		}
	}

	if ( empty( $items_html ) ) {
		return false;
	}

	$menu_html = sprintf( $args['items_wrap'], esc_attr( $id ), esc_attr( $class ), $items_html );

	if ( $args['echo'] ) {
		echo wp_kses_post( $menu_html );
	} else {
		return $menu_html;
	}
}

/**
 * Outputs a preconnect link that helps performance of Google Fonts.
 * Hooked conditionally on 'wp_head'.
 */
function andros_head_preconnect_google_fonts() {
	?><link rel="preconnect" href="https://fonts.gstatic.com/" crossorigin><?php
}

/**
 * Standard template tags.
 */
require_once get_theme_file_path( '/base/template-tags.php' );

/**
 * Standard sanitization functions.
 */
require_once get_theme_file_path( '/base/sanitization.php' );

/**
 * Standard customizer controls.
 */
require_once get_theme_file_path( '/base/customizer.php' );

/**
 * Standard customizer partial callbacks.
 */
require_once get_theme_file_path( '/base/customizer-partial-callbacks.php' );

/**
 * Standard customizer styles.
 */
require_once get_theme_file_path( '/base/customizer-styles.php' );

/**
 * Standard hooks.
 */
require_once get_theme_file_path( '/base/default-hooks.php' );

