<?php 
/**
* @version		$Id: client.class.php 1486 2020-08-17 18:45:42Z IOS $
* @package		Component IOS Reservations
* @author		Ioannis Sannos ( https://www.isopensource.com )
* @copyright	Copyright (c) 2008-2020 Is Open Source (https://www.isopensource.com). All rights reserved.
* @license		Commercial
* @description 	Online booking system
**************************************/

/**
=== COMMERCIAL LICENSE ===

IT IS EXTREMELY FORBIDDEN TO MODIFY THIS FILE
IOS RENTALS IS COPYRIGHTED BY IOANNIS SANNOS
HTTPS://WWW.ISOPENSOURCE.COM
*/

defined('_IOSRTAPICLIENT_') or die ('Direct access to this location is not allowed');


class iosrtApiClient {

	private $path = '';
	private $uri = '';
	private $host = '';
	private $options = array(
		'serverurl' => '',
		'clienturl' => '',
		'apiid' => '',
		'prkey' => '',
		'platform' => 'standalone', //elxis, joomla, wordpress, standalone
		'sethead' => 1,
		'guilang' => '', //prefered language for User Interface: EMPTY (default), or language 2 letters ISO code
		'lang' => '', //prefered language for Server: EMPTY (server default), or language 2 letters ISO code
		'currency' => '',//prefered currency EMPTY (default), or 3 letters currency code
		'type' => 'XML', //ALWAYS XML!
		'cachetime' => 21600,//6 hours
		'loadfont' => '',
		'vtype' => '',
		'modelfeaticons' => 6, //how many feature icons to display on models
		'modelprice' => 1, //0:in search results display rental fee as price, 1: display total price (inc. pickup fee, etc), 2: display rental fee and rest charges
		'applyvat' => 1, //0: display prices without VAT, 1: display prices with VAT (* APPLIES IF VAT > 0)
		'vatnote' => 1, //0: show VAT YES/NO (VAT 24% included, or prices include all taxes, or prices without VAT)
		'showerrorcodes' => 0,
		'formaction' => '',
		'unifiedmoto' => 0,
		'dropsameaspick' => 0,
		'sidecolumn' => '',
		'formtext1' => '',
		'formtext2' => '',
		'showcoupon' => 1,
		'showinvoice' => 1,
		'rentalduration' => 0,//0:real days/hours + charge days/hours, 1:real days/hours, 2:charge days/hours
		'imagewidth' => 240,
		'pickupday' => -1,//Pickup only on specific day (php/js day number). 0:Sunday, 1: Monday, ... 6: Saturday. "-1" to disable
		'dropoffday' => -1,//Dropoff only on specific day (php/js day number). 0:Sunday, 1: Monday, ... 6: Saturday. "-1" to disable
		'pickuptime' => '',//fixed pickup time that cannot be changed, eg "10:00", empty to disable (default)
		'dropofftime' => '',//fixed dropoff time that cannot be changed, eg "09:00", empty to disable (default)
		'hidedrop' => 0,//if "1" drop-off location is hidden and is always the same as pick-up
		'titlepos' => 0,//0: top, 1: side, 2: bottom
		'modellocs' => 0,//Display the locations a model is available at. 0: Dont display, 1: Display only if not avail. in all locs, 2: Always display
		'lids' => '',//comma separated location IDs (lid) to display. If empty show all
		'fleetpage' => '',//fleet page URL, if empty: current page. Elxis URIs are supported
		'roundprices' => 1//0: No, 1: yes, if price is integer, 2: yes, to nearest integer, 3: yes to nearest integer with decimals
	);
	private $lang = null;
	private $errormsg = '';
	private $helper = null;
	private $hasgallery = false;
	private $clientversion = 20;//introduced in IOS Rentals v1.6, previously did not existed. Useful to get responses by IOS Rentals compatible to the client's version


	/*********************/
	/* MAGIC CONSTRUCTOR */
	/*********************/
	public function __construct($apiid, $cfgoptions=array()) {
		$this->path = str_replace(DIRECTORY_SEPARATOR, '/', dirname(__FILE__));
		$this->uri = $this->getURI();
		$this->host = $this->getHost();

		if (!file_exists($this->path.'/apiconfig.php')) {
			$this->errormsg = 'API configuration file not found!';
			return;
		}

		if ($apiid == '') {
			$this->errormsg = 'Invalid request (No API)!';
			return;
		}

		include($this->path.'/apiconfig.php');
		if (!isset($apiconfig) || !is_array($apiconfig) || (count($apiconfig) == 0)) {
			$this->errormsg = 'API configuration file has no API data!';
			return;
		}

		$found = false;
		foreach ($apiconfig as $cfg) {
			if ($cfg['apiid'] != $apiid) { continue; }
			$found = true;
			$this->options = $cfg;
			$this->options['type'] = 'XML';
			break;
		}

		if (!$found) {
			$this->errormsg = 'API '.$apiid.' not found in configuration file!';
			return;
		}

		$this->options['serverurl'] = rtrim($this->options['serverurl'], '/');
		$this->options['clienturl'] = rtrim($this->options['clienturl'], '/');
		$this->options['cachetime'] = (int)$this->options['cachetime'];

		if ($cfgoptions) {
			foreach ($cfgoptions as $k => $v) {
				switch ($k) {
					case 'guilang': case 'lang': case 'formaction': case 'sidecolumn': case 'formtext1': case 'formtext2': case 'pickuptime': case 'dropofftime': case 'lids': case 'fleetpage':
						if ($v != '') { $this->options[$k] = $v; }
					break;
					case 'vtype': case 'currency': $this->options[$k] = $v; break;
					case 'modelfeaticons': case 'modelprice': case 'showerrorcodes': case 'unifiedmoto': case 'dropsameaspick': 
					case 'applyvat': case 'showvat': case 'showcoupon': case 'showinvoice': case 'rentalduration': case 'imagewidth': 
					case 'pickupday': case 'dropoffday': case 'hidedrop': case 'titlepos': case 'modellocs': case 'roundprices':
						$this->options[$k] = (int)$v;
					break;
					default: break;
				}
			}
		}

		if (!isset($this->options['imagewidth'])) { $this->options['imagewidth'] = 240; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['pickupday'])) { $this->options['pickupday'] = -1; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['dropoffday'])) { $this->options['dropoffday'] = -1; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['pickuptime'])) { $this->options['pickuptime'] = ''; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['dropofftime'])) { $this->options['dropofftime'] = ''; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['hidedrop'])) { $this->options['hidedrop'] = 0; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['titlepos'])) { $this->options['titlepos'] = 0; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['modellocs'])) { $this->options['modellocs'] = 0; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['lids'])) { $this->options['lids'] = ''; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['fleetpage'])) { $this->options['fleetpage'] = ''; }//compatibility with previous than v1.4 configuration files
		if (!isset($this->options['roundprices'])) { $this->options['roundprices'] = 1; }//compatibility with previous than v1.4 configuration files
		if ($this->options['hidedrop'] == 1) { $this->options['dropsameaspick'] = 0; }

		require_once($this->path.'/lang.class.php');
		$this->lang = new iosrtApiLang();
		$this->lang->loadLang($this->path, $this->options['guilang']);
		$this->options['guilang'] = $this->lang->guilang;

		require_once($this->path.'/helper.class.php');
		$this->helper = new iosrtApiHelper($this->options, $this->path, $this->clientversion);

		$this->basicValidation();
	}


	/***********************/
	/* VALIDATE BASIC DATA */
	/***********************/
	private function basicValidation() {
		if (($this->options['serverurl'] == '') || !preg_match('#^http(s)?://[a-z0-9-]+(.[a-z0-9-]+)*(:[0-9]+)?(/.*)?$#i', $this->options['serverurl'])) {
			$this->errormsg = 'Empty or invalid API server URL!';
			return false;
		}
		if (($this->options['clienturl'] == '') || !preg_match('#^http(s)?://[a-z0-9-]+(.[a-z0-9-]+)*(:[0-9]+)?(/.*)?$#i', $this->options['clienturl'])) {
			$this->errormsg = 'Empty or invalid client base URL!';
			return false;
		}
		if (($this->options['apiid'] == '') || (strlen($this->options['apiid']) < 8)) {
			$this->errormsg = 'Empty or invalid API ID!';
			return false;
		}
		if (($this->options['prkey'] == '') || (strlen($this->options['prkey']) < 16)) {
			$this->errormsg = 'Empty or invalid private key for the API!';
			return false;
		}
		return true;
	}


	/*********************************************/
	/* ECHO HEAD DATA FOR STANDALONE INTEGRATION */
	/*********************************************/
	public function standaloneHead() {
		if ($this->errormsg != '') {
			if ($this->options['clienturl'] == '') { return; }
			$cssfile = 'client.css';
		} else {
			$eLang = $this->lang;
			$cssfile = 'client.css';
			if ($eLang->get('DIR') == 'rtl') { $cssfile = 'client-rtl.css'; }
		}

		if ($this->options['loadfont'] != '') {
			echo '<link rel="stylesheet" href="'.$this->options['clienturl'].'/css/'.$this->options['loadfont'].'.css" type="text/css" />'."\n";
		}
		echo '<link rel="stylesheet" href="'.$this->options['clienturl'].'/css/'.$cssfile.'" type="text/css" />'."\n";
		echo '<link rel="stylesheet" href="'.$this->options['clienturl'].'/css/flatpickr.min.css" type="text/css" />'."\n";
		echo '<link rel="stylesheet" href="'.$this->options['clienturl'].'/css/swiper.min.css" type="text/css" />'."\n";
		echo '<script src="'.$this->options['clienturl'].'/js/tippy.min.js"></script>'."\n";
		echo '<script src="'.$this->options['clienturl'].'/js/swiper.min.js"></script>'."\n";
		$step = (int)$this->getFromRequest('step', 0);
		if (($step == 3) || ($step == 4)) {//thanks page, success / payment success
			echo '<link rel="stylesheet" href="'.$this->options['clienturl'].'/css/print.css" type="text/css" />'."\n";
		}
		echo '<script src="'.$this->options['clienturl'].'/js/flatpickr.min.js"></script>'."\n";
		$lng = strtolower($this->options['guilang']);
		if (file_exists($this->path.'/js/flatpickri18n/'.$lng.'.js')) {
			echo '<script src="'.$this->options['clienturl'].'/js/flatpickri18n/'.$lng.'.js"></script>'."\n";
		}
		echo '<script src="'.$this->options['clienturl'].'/js/client.js"></script>'."\n";

		$calendar = $this->calendarData();

		echo '<script>'."\n";
		if ($calendar['disranges']) {
			echo 'var iosRTCalDisRanges = '.json_encode($calendar['disranges']).';'."\n";
		} else {
			echo 'var iosRTCalDisRanges = {};'."\n";
		}
		if ($calendar['nwranges']) {
			echo 'var iosRTCalHourRanges = '.json_encode($calendar['nwranges']).';'."\n";
		} else {
			echo 'var iosRTCalHourRanges = {};'."\n";
		}
		if ($calendar['rentdranges']) {
			echo 'var iosRTCalRentdRanges = '.json_encode($calendar['rentdranges']).';'."\n";
		} else {
			echo 'var iosRTCalRentdRanges = {};'."\n";
		}
		echo 'var iosRTCalminrentdays = '.$calendar['minrentdays'].';'."\n";
		echo 'var iosRTCalmaxrentdays = '.$calendar['maxrentdays'].';'."\n";
		echo 'var iosRTCalpickupdayofweek = '.$calendar['pickupdayofweek'].';'."\n";
		echo 'var iosRTCaldropoffdayofweek = '.$calendar['dropoffdayofweek'].';'."\n";
		echo 'var iosRThidedrop = '.$calendar['hidedrop'].';'."\n";
		echo 'var iosRTpickplushours = '.$calendar['pickplushours'].';'."\n";
		echo "</script>\n";
	}


	/*****************/
	/* SET HEAD DATA */
	/*****************/
	private function setHead() {
		if ((int)$this->options['sethead'] != 1) { return; }

		$eLang = $this->lang;

		$cssfile = 'client.css';
		if ($eLang->get('DIR') == 'rtl') { $cssfile = 'client-rtl.css'; }

		switch ($this->options['platform']) {
			case 'elxis'://4.x
				$eDoc = eFactory::getDocument();
				if ($this->options['loadfont'] != '') {
					$eDoc->addStyleLink($this->options['clienturl'].'/css/'.$this->options['loadfont'].'.css');
				}
				$eDoc->addStyleLink($this->options['clienturl'].'/css/'.$cssfile);
				$eDoc->addStyleLink($this->options['clienturl'].'/css/flatpickr.min.css');
				$eDoc->addStyleLink($this->options['clienturl'].'/css/swiper.min.css');
				$step = (int)$this->getFromRequest('step', 0);
				if (($step == 3) || ($step == 4)) {//thanks page, success / payment success
					$eDoc->addStyleLink($this->options['clienturl'].'/css/print.css');
				}
				$eDoc->addScriptLink($this->options['clienturl'].'/js/tippy.min.js');
				$eDoc->addScriptLink($this->options['clienturl'].'/js/flatpickr.min.js');
				$eDoc->addScriptLink($this->options['clienturl'].'/js/swiper.min.js');
				$lng = strtolower($this->options['guilang']);
				if (file_exists($this->path.'/js/flatpickri18n/'.$lng.'.js')) {
					$eDoc->addScriptLink($this->options['clienturl'].'/js/flatpickri18n/'.$lng.'.js');
				}
				$eDoc->addScriptLink($this->options['clienturl'].'/js/client.js');
				$calendar = $this->calendarData();
				if ($calendar['disranges']) {
					$js = 'var iosRTCalDisRanges = '.json_encode($calendar['disranges']).';'."\n";
				} else {
					$js = 'var iosRTCalDisRanges = {};'."\n";
				}
				if ($calendar['nwranges']) {
					$js .= 'var iosRTCalHourRanges = '.json_encode($calendar['nwranges']).';'."\n";
				} else {
					$js .= 'var iosRTCalHourRanges = {};'."\n";
				}
				if ($calendar['rentdranges']) {
					$js .= 'var iosRTCalRentdRanges = '.json_encode($calendar['rentdranges']).';'."\n";
				} else {
					$js .= 'var iosRTCalRentdRanges = {};'."\n";
				}
				$js .= 'var iosRTCalminrentdays = '.$calendar['minrentdays'].';'."\n";
				$js .= 'var iosRTCalmaxrentdays = '.$calendar['maxrentdays'].';'."\n";
				$js .= 'var iosRTCalpickupdayofweek = '.$calendar['pickupdayofweek'].';'."\n";
				$js .= 'var iosRTCaldropoffdayofweek = '.$calendar['dropoffdayofweek'].';'."\n";
				$js .= 'var iosRThidedrop = '.$calendar['hidedrop'].';'."\n";
				$js .= 'var iosRTpickplushours = '.$calendar['pickplushours'].';'."\n";
				$eDoc->addScript($js);
			break;
			case 'joomla'://3.x, NOT TESTED!
				$document = JFactory::getDocument();
				if ($this->options['loadfont'] != '') {
					$document->addStyleSheet($this->options['clienturl'].'/css/'.$this->options['loadfont'].'.css');
				}
				$document->addStyleSheet($this->options['clienturl'].'/css/'.$cssfile);
				$document->addStyleSheet($this->options['clienturl'].'/css/flatpickr.min.css');
				$document->addStyleSheet($this->options['clienturl'].'/css/swiper.min.css');
				$step = (int)$this->getFromRequest('step', 0);
				if (($step == 3) || ($step == 4)) {//thanks page, success / payment success
					$document->addStyleLink($this->options['clienturl'].'/css/print.css');
				}
				$document->addScript($this->options['clienturl'].'/js/client.js');
				$document->addScript($this->options['clienturl'].'/js/flatpickr.min.js');
				$lng = strtolower($this->options['guilang']);
				if (file_exists($this->path.'/js/flatpickri18n/'.$lng.'.js')) {
					$document->addScript($this->options['clienturl'].'/js/flatpickri18n/'.$lng.'.js');
				}
				$document->addScript($this->options['clienturl'].'/js/tippy.min.js');
				$document->addScript($this->options['clienturl'].'/js/swiper.min.js');
				$calendar = $this->calendarData();
				if ($calendar['disranges']) {
					$js = 'var iosRTCalDisRanges = '.json_encode($calendar['disranges']).';'."\n";
				} else {
					$js = 'var iosRTCalDisRanges = {};'."\n";
				}
				if ($calendar['nwranges']) {
					$js .= 'var iosRTCalHourRanges = '.json_encode($calendar['nwranges']).';'."\n";
				} else {
					$js .= 'var iosRTCalHourRanges = {};'."\n";
				}
				if ($calendar['rentdranges']) {
					$js .= 'var iosRTCalRentdRanges = '.json_encode($calendar['rentdranges']).';'."\n";
				} else {
					$js .= 'var iosRTCalRentdRanges = {};'."\n";
				}
				$js .= 'var iosRTCalminrentdays = '.$calendar['minrentdays'].';'."\n";
				$js .= 'var iosRTCalmaxrentdays = '.$calendar['maxrentdays'].';'."\n";
				$js .= 'var iosRTCalpickupdayofweek = '.$calendar['pickupdayofweek'].';'."\n";
				$js .= 'var iosRTCaldropoffdayofweek = '.$calendar['dropoffdayofweek'].';'."\n";
				$js .= 'var iosRThidedrop = '.$calendar['hidedrop'].';'."\n";
				$js .= 'var iosRTpickplushours = '.$calendar['pickplushours'].';'."\n";
				$document->addScriptDeclaration($js);
			break;
			case 'wordpress': break;//controlled in plugin file
			default:
				$this->options['platform'] = 'standalone';
			break;
		}
	}


	/*****************************************/
	/* INITIALIZE/ROUTER (BASIC INTEGRATION) */
	/*****************************************/
	public function init() {
		if ($this->errormsg != '') {
			if ($this->options['clienturl'] != '') { $this->setHead(); }
			return '<div class="iosrt_error">'.$this->errormsg."</div>\n";
		}

		$step = (int)$this->getFromRequest('step', 0);
		if ($step == 0) {
			$html = $this->availabilityForm();
			return $html;
		} else if ($step == 1) {
			$html = $this->listAvailableVehicles();
			return $html;
		} else if ($step == 2) {
			$html = $this->bookForm();
			return $html;
		} else if ($step == 3) {//thanks page, success
			$html = $this->finalPage(3);
			return $html;
		} else if ($step == 4) {//thanks page, payment success
			$html = $this->finalPage(4);
			return $html;
		} else if ($step == 5) {//thanks page, payment failure
			$html = $this->finalPage(5);
			return $html;
		} else if ($step == 6) {//thanks page, payment success for Alpha bank e-Commerce
			$html = $this->finalPageAlpha();
			return $html;
		} else if ($step == 7) {//thanks page, payment success for Piraeus bank ePOS Paycenter
			$html = $this->finalPageePOS();
			return $html;
		} else if ($step == 8) {//thanks page, payment success for PayUBiz (PayU India)
			$html = $this->finalPagePayUBiz();
			return $html;
		} else if ($step == 9) {//thanks page, payment success for Viva Wallet
			$html = $this->finalPageVivaWallet();
			return $html;
		} else {
			if ($this->options['clienturl'] != '') { $this->setHead(); }
			return '<div class="iosrt_error">Invalid request!'."</div>\n";
		}
	}


	/*****************************************/
	/* INITIALIZE/ROUTER (FLEET INTEGRATION) */
	/*****************************************/
	public function fleetinit() {
		if ($this->errormsg != '') {
			if ($this->options['clienturl'] != '') { $this->setHead(); }
			return '<div class="iosrt_error">'.$this->errormsg."</div>\n";
		}

		$mid = (int)$this->getFromRequest('mid', 0);
		if ($mid < 1) {
			$html = $this->listFleet();
			return $html;
		} else {
			$html = $this->fleetModel();
			return $html;
		}
	}


	/************************************/
	/* INITIALIZE/ROUTER FOR AJAX TASKS */
	/************************************/
	public function initAjax() {
		$act = isset($_POST['act']) ? trim($_POST['act']) : '';
		if ($act == 'rc') {
			$this->initRC();
			return;
		}

		if ($this->errormsg != '') {
			$this->pageHeaders('text/plain');
			echo $this->errormsg;
			exit;
		}
		if ($act == 'checkcoupon') {
			$this->checkCoupon();
			exit;
		} else if ($act == 'book') {
			$this->book();
			exit;
		} else if ($act == 'change') {
			$this->changeReservation();
			exit;
		} else {
			$this->pageHeaders('text/plain');
			echo 'Invalid request';
			exit;
		}
	}


	/************************************************************/
	/* INITIALIZE/ROUTER FOR REMOTE CALLS (ajax.php / act = rc) */
	/************************************************************/
	private function initRC() {
		$type = isset($_POST['type']) ? trim($_POST['type']) : '';
		if ($type != 'XML') { $type = 'JSON'; }

		if ($this->errormsg != '') {
			$response = array('success' => 0, 'message' => $this->errormsg);
			$this->responseRC($type, $response);
		}

		$ts = isset($_POST['ts']) ? (int)$_POST['ts'] : 0;
		$rctask = trim(filter_input(INPUT_POST, 'rctask', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH));
		$enc = trim(filter_input(INPUT_POST, 'enc', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH));

		if ($ts < 100000) {
			$this->errormsg = 'Invalid request! (ERROR RC01)';
		} else if ($rctask == '') {
			$this->errormsg = 'Invalid request! (ERROR RC02)';
		} else if (($enc == '') || (strlen($enc) < 16)) {
			$this->errormsg = 'Authorization failed! (ERROR RC03)';
		} else {
			$client_enc = sha1($this->options['apiid'].','.$ts.','.$type.','.$this->options['prkey']);
			if ($client_enc != $enc) {
				$this->errormsg = 'Authentication failed! (ERROR RC04)';
			}
		}

		if ($this->errormsg != '') {
			$response = array('success' => 0, 'message' => $this->errormsg);
			$this->responseRC($type, $response);
		}

		if ($rctask == 'clearcache') {
			$this->rcClearCache($type);
		}

		$response = array('success' => 0, 'message' => 'Not supported task request! (ERROR RC05)');
		$this->responseRC($type, $response);
	}


	/*********************************/
	/* GENERIC GET/PUT WITH RAW DATA */
	/*********************************/
	public function request($endpoint, $type='XML', $custom_options=array()) {
		if ($type != 'JSON') { $type = 'XML'; }

		$cfgtype = $this->options['type'];
		$this->options['type'] = $type;

		$response = $this->helper->genericRequest($endpoint, $type, $custom_options);
		$this->options['type'] = $cfgtype;

		if ($response === false) {
			$this->errormsg = $this->helper->getError();
			if ($this->errormsg == '') { $this->errormsg = 'Request failed - Unknown error message'; }
			return false;
		}
		return $response;
	}


	/*************************************/
	/* STEP 0 :: AVAILABILITY CHECK FORM */
	/*************************************/
	private function availabilityForm($errormsg='', $sidecolumn=false, $initformdata=array(), $vehdata=array()) {
		$eLang = $this->lang;

		if (!$sidecolumn) {
			$this->setHead();
		}

		$company = $this->helper->basicRequest('company');
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}

		$places = $this->helper->basicRequest('places');

		$vtypes = $this->helper->searchableVehicleTypes($company);
		if (!$vtypes) {
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL')."<br />There are no valid vehicle types!</p></div>\n";
		}

		$action = '';
		if (trim($this->options['formaction']) != '') {
			if ($this->options['platform'] == 'elxis') {
				if (stripos($this->options['formaction'], 'http') === false) {//Elxis URI
					$action = eFactory::getElxis()->makeURL($this->options['formaction']);
				} else {
					$action = $this->options['formaction'];
				}
			} else {
				$action = $this->options['formaction'];
			}
			if (!filter_var($action, FILTER_VALIDATE_URL)) { $action = ''; }
		}
		if ($action == '') { $action = $this->makeFormURL(); }

		if ($vehdata) {//fleet vehicle page
			$html = '';
		} else if (!$sidecolumn) {
			$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";
		} else {
			$html = '';
		}
		
		if ($errormsg != '') {
			$html .= '<div class="iosrt_error">'.$errormsg."</div>\n";
		}

		if ($initformdata) {//for sidecolumn
			$initial_values = $initformdata;
		} else {
			$initial_values = array(
				'picklid' => 0, 
				'droplid' => 0, 
				'pickdate' => '',
				'pickhour' => 10, 
				'pickmin' => '00', 
				'dropdate' => '',
				'drophour' => 10, 
				'dropmin' => '00', 
				'vtype' => ''
			);
			if (isset($_GET['picklid']) && isset($_GET['droplid']) && isset($_GET['pickup']) && isset($_GET['dropoff']) && isset($_GET['vtype']) && isset($_GET['enc'])) {
				$udata = array(
					'picklid' => (int)$_GET['picklid'],
					'droplid' => (int)$_GET['droplid'],
					'pickup' => urldecode($_GET['pickup']),
					'dropoff' => urldecode($_GET['dropoff']),
					'vtype' => $_GET['vtype']
				);
				$enc = md5($udata['picklid'].$udata['droplid'].$udata['pickup'].$udata['dropoff'].$udata['vtype'].$this->options['prkey']);
				if ($enc == urldecode($_GET['enc'])) {
					$initial_values['picklid'] = $udata['picklid'];
					$initial_values['droplid'] = $udata['droplid'];
					$initial_values['pickdate'] = substr($udata['pickup'], 8, 2).'-'.substr($udata['pickup'], 5, 2).'-'.substr($udata['pickup'], 0, 4);//DD-MM-YYYY
					$initial_values['pickhour'] = intval(substr($udata['pickup'], 11, 2));
					$initial_values['pickmin'] = substr($udata['pickup'], 14, 2);
					$initial_values['dropdate'] = substr($udata['dropoff'], 8, 2).'-'.substr($udata['dropoff'], 5, 2).'-'.substr($udata['dropoff'], 0, 4);//DD-MM-YYYY
					$initial_values['drophour'] = intval(substr($udata['dropoff'], 11, 2));
					$initial_values['dropmin'] = substr($udata['dropoff'], 14, 2);
					$vt = $udata['vtype'];
					if ($udata['vtype'] != '') {
						if (strpos($udata['vtype'], ',') !== false) {
							$parts = explode(',', $udata['vtype']);
							$vt = $parts[0];
						}
						if ($this->options['unifiedmoto'] == 1) {
							if (($vt == 'moto') || ($vt == 'scooter') || ($vt == 'atv') || ($vt == 'buggy')) { $vt = 'moto'; }
						}
					}
					$initial_values['vtype'] = $vt;
				}//valid data
				unset($udata);
			}
		}

		if ($vehdata) {//fleet vehicle page
			$class = 'iosrt_formvehwrap';
		} else if ($sidecolumn) {
			$class = 'iosrt_formsidewrap';
		} else {
			$class = 'iosrt_formwrap';
		}

		$html .= '<div class="'.$class.'" id="iosrt_form_search">'."\n";
		$html .= '<form name="fmpickdates" id="iosrt_fmpickdates" method="get" action="'.$action.'" class="iosrt_form" data-aurl="'.$this->options['clienturl'].'" data-apiid="'.$this->options['apiid'].'" data-lang="'.$this->options['lang'].'" data-currency="'.$this->options['currency'].'" data-guilang="'.$this->options['guilang'].'" >'."\n";
		if ($this->options['hidedrop'] == 0) {
			$html .= '<div class="iosrt_2colflexwrap">'."\n";
			$html .= '<div class="iosrt_2colflexbox iosrt_spad">'."\n";
		} else {
			$html .= '<div class="iosrt_spad">'."\n";
		}
		$html .= '<label class="iosrt_labelblock" for="iosrt_picklid"><span class="iosrt_minicircle_block">1</span>'.$eLang->get('PICKUP_LOC')."</label>\n";

		$onchange2 = ($this->options['hidedrop'] == 1) ? ' iosRTSetLocation(\'drop\');' : '';
		$html .= '<select name="picklid" id="iosrt_picklid" class="iosrt_select" onchange="iosRTSetLocation(\'pick\');'.$onchange2.'">'."\n";
		$sel = ($initial_values['picklid'] == 0) ? ' selected="selected"' : '';
		$html .= '<option value="0"'.$sel.'>--- '.$eLang->get('SELECT_LOCATION')." ---</option>\n";

		if ($places) {
			$multi_countries = (count($places) > 1) ? true : false;
			$onlylids = ($this->options['lids'] != '') ? explode(',', $this->options['lids']) : array();
			foreach ($places as $code => $countrydata) {
				if (!$countrydata['areas']) { continue; }
				if ($multi_countries) {
					$html .= '<optgroup label="'.$countrydata['name'].'">'."\n";
				}
				$multi_areas = (count($countrydata['areas']) > 1) ? true : false;
				foreach ($countrydata['areas'] as $aid => $area) {
					if (!$area['locations']) { continue; }
					if ($onlylids) {
						$has_visible_locs = false;
						foreach ($area['locations'] as $lid => $loc) {
							if (in_array($lid, $onlylids)) { $has_visible_locs = true; break; }
						}
						if (!$has_visible_locs) { continue; }
					}

					if ($multi_areas) {
						if ($multi_countries) {
							$html .= '<option value="-1" disabled="disabled">--- '.$area['title']." ---</option>\n";
						} else {
							$html .= '<optgroup label="'.$area['title'].'">'."\n";
						}
					}
					foreach ($area['locations'] as $lid => $loc) {
						if ($onlylids) {
							if (!in_array($lid, $onlylids)) { continue; }
						}
						$sel = ($initial_values['picklid'] == $lid) ? ' selected="selected"' : '';
						$html .= '<option value="'.$lid.'"'.$sel.'>'.$loc['title']."</option>\n";
					}
					if ($multi_areas && !$multi_countries) { $html .= "</optgroup>\n"; }
				}
				if ($multi_countries) { $html .= "</optgroup>\n"; }
			}
		}
		$html .= "</select>\n";
		$html .= "</div>\n";//iosrt_2colflexbox or iosrt_spad

		if ($this->options['hidedrop'] == 0) {
			$html .= '<div class="iosrt_2colflexbox iosrt_spad">'."\n";
		} else {
			$html .= '<div class="iosrt_spad">'."\n";
		}
		$html .= '<label class="iosrt_labelblock" for="iosrt_pickdate"><span class="iosrt_minicircle_block">2</span>'.$eLang->get('PICKUP_DATE')."</label>\n";
		$html .= '<div class="iosrt_datetimewrap">';

		if ($this->options['pickuptime'] != '') {
			$parts = explode(':', $this->options['pickuptime']);
			$min = isset($parts[1]) ? $parts[1] : '00';
			$html .= '<input type="hidden" name="pickhour" id="iosrt_pickhour" value="'.$parts[0].'" />'."\n";
			$html .= '<input type="hidden" name="pickmin" id="iosrt_pickmin" value="'.$min.'" />'."\n";
		} else {
			$html .= '<div class="iosrt_timebox">';
			$html .= '<div class="iosrt_hourbox">';
			$html .= '<select name="pickhour" id="iosrt_pickhour" class="iosrt_select" tabindex="-1" title="'.$eLang->get('PICKUP_HOUR').'">'."\n";
			for ($i = 0; $i < 24; $i++) {
				$sel = ($i == $initial_values['pickhour']) ? ' selected="selected"' : '';
				$v = sprintf("%02d", $i);
				$html .= '<option value="'.$v.'"'.$sel.'>'.$v."</option>\n";
			}
			$html .= "</select>\n";
			$html .= '</div>';//iosrt_hourbox
			$html .= '<div class="iosrt_minbox">';
			$html .= '<select name="pickmin" id="iosrt_pickmin" class="iosrt_select" tabindex="-1" title="'.$eLang->get('PICKUP_MINUTES').'">'."\n";
			$sel = ($initial_values['pickmin'] == '00') ? ' selected="selected"' : '';
			$html .= '<option value="00"'.$sel.'>00</option>'."\n";
			$sel = ($initial_values['pickmin'] == '15') ? ' selected="selected"' : '';
			$html .= '<option value="15"'.$sel.'>15</option>'."\n";
			$sel = ($initial_values['pickmin'] == '30') ? ' selected="selected"' : '';
			$html .= '<option value="30"'.$sel.'>30</option>'."\n";
			$sel = ($initial_values['pickmin'] == '45') ? ' selected="selected"' : '';
			$html .= '<option value="45"'.$sel.'>45</option>'."\n";
			$html .= "</select>\n";
			$html .= '</div>';//iosrt_minbox
			$html .= '</div>';//iosrt_timebox
		}

		$high = ($initial_values['pickdate'] == '') ? ' onclick="iosRTHighlight(\'iosrt_picklid\');"' : '';
		$class = ($this->options['pickuptime'] != '') ? 'iosrt_dateonlybox' : 'iosrt_datebox';

		$html .= '<div class="'.$class.'"><input type="text" name="pickdate" id="iosrt_pickdate" maxlength="10" dir="ltr" value="'.$initial_values['pickdate'].'" class="iosrt_text"'.$high.' placeholder="'.$eLang->get('DATE_PLACEHOLDER').'" pattern="[0-9]{2}-[0-9]{2}-[0-9]{4}" readonly="readonly" /></div>';
		$html .= "</div>\n";//iosrt_datetimewrap

		if ($this->options['hidedrop'] == 0) {
			$html .= "</div>\n";//iosrt_2colflexbox
			$html .= "</div>\n";//iosrt_2colflexwrap
		} else {
			$html .= "</div>\n";//iosrt_spad
		}

		if ($this->options['hidedrop'] == 0) {
			$html .= '<div class="iosrt_2colflexwrap">'."\n";
			$html .= '<div class="iosrt_2colflexbox iosrt_spad">'."\n";

			$html .= '<label class="iosrt_labelblock" for="iosrt_droplid"><span class="iosrt_minicircle_block">3</span>'.$eLang->get('RETURN_LOC')."</label>\n";
			$html .= '<select name="droplid" id="iosrt_droplid" class="iosrt_select" onchange="iosRTSetLocation(\'drop\');">'."\n";
			$sel = ($initial_values['droplid'] == 0) ? ' selected="selected"' : '';
			$html .= '<option value="0"'.$sel.'>--- '.$eLang->get('SELECT_LOCATION')." ---</option>\n";
			if ($places) {
				$multi_countries = (count($places) > 1) ? true : false;
				$onlylids = ($this->options['lids'] != '') ? explode(',', $this->options['lids']) : array();
				foreach ($places as $code => $countrydata) {
					if (!$countrydata['areas']) { continue; }
					if ($multi_countries) {
						$html .= '<optgroup label="'.$countrydata['name'].'">'."\n";
					}
					$multi_areas = (count($countrydata['areas']) > 1) ? true : false;
					foreach ($countrydata['areas'] as $aid => $area) {
						if (!$area['locations']) { continue; }
						if ($onlylids) {
							$has_visible_locs = false;
							foreach ($area['locations'] as $lid => $loc) {
								if (in_array($lid, $onlylids)) { $has_visible_locs = true; break; }
							}
							if (!$has_visible_locs) { continue; }
						}
						if ($multi_areas) {
							if ($multi_countries) {
								$html .= '<option value="-1" disabled="disabled">--- '.$area['title']." ---</option>\n";
							} else {
								$html .= '<optgroup label="'.$area['title'].'">'."\n";
							}
						}
						foreach ($area['locations'] as $lid => $loc) {
							if ($onlylids) {
								if (!in_array($lid, $onlylids)) { continue; }
							}
							$sel = ($initial_values['droplid'] == $lid) ? ' selected="selected"' : '';
							$html .= '<option value="'.$lid.'"'.$sel.'>'.$loc['title']."</option>\n";
						}
						if ($multi_areas && !$multi_countries) { $html .= "</optgroup>\n"; }
					}
					if ($multi_countries) { $html .= "</optgroup>\n"; }
				}
			}
			$html .= "</select>\n";

			if ($this->options['dropsameaspick'] == 1) {
				$html .= '<a href="javascript:void(null);" onclick="iosRTDropSamePick();" class="iosrt_samedrop" tabindex="-1">'.$eLang->get('SAME_AS_PICKLOC').'</a>';
			}
			$html .= "</div>\n";//iosrt_2colflexbox
		} else {
			$html .= '<input type="hidden" name="droplid" id="iosrt_droplid" value="'.$initial_values['droplid'].'" />'."\n";
		}

		if ($this->options['hidedrop'] == 0) {
			$html .= '<div class="iosrt_2colflexbox iosrt_spad">'."\n";
			$num = '4';
		} else {
			$html .= '<div class="iosrt_spad">'."\n";
			$num = '3';
		}
		$html .= '<label class="iosrt_labelblock" for="iosrt_dropdate"><span class="iosrt_minicircle_block">'.$num.'</span>'.$eLang->get('RETURN_DATE')."</label>\n";
		$html .= '<div class="iosrt_datetimewrap">';

		if ($this->options['dropofftime'] != '') {
			$parts = explode(':', $this->options['dropofftime']);
			$min = isset($parts[1]) ? $parts[1] : '00';
			$html .= '<input type="hidden" name="drophour" id="iosrt_drophour" value="'.$parts[0].'" />'."\n";
			$html .= '<input type="hidden" name="dropmin" id="iosrt_dropmin" value="'.$min.'" />'."\n";
		} else {
			$html .= '<div class="iosrt_timebox">';
			$html .= '<div class="iosrt_hourbox">';
			$html .= '<select name="drophour" id="iosrt_drophour" class="iosrt_select" tabindex="-1" title="'.$eLang->get('DROPOFF_HOUR').'">'."\n";
			for ($i = 0; $i < 24; $i++) {
				$sel = ($i == $initial_values['drophour']) ? ' selected="selected"' : '';
				$v = sprintf("%02d", $i);
				$html .= '<option value="'.$v.'"'.$sel.'>'.$v."</option>\n";
			}
			$html .= "</select>\n";
			$html .= '</div>';//iosrt_hourbox
			$html .= '<div class="iosrt_minbox">';
			$html .= '<select name="dropmin" id="iosrt_dropmin" class="iosrt_select" tabindex="-1" title="'.$eLang->get('DROPOFF_MINUTES').'">'."\n";
			$sel = ($initial_values['dropmin'] == '00') ? ' selected="selected"' : '';
			$html .= '<option value="00"'.$sel.'>00</option>'."\n";
			$sel = ($initial_values['dropmin'] == '15') ? ' selected="selected"' : '';
			$html .= '<option value="15"'.$sel.'>15</option>'."\n";
			$sel = ($initial_values['dropmin'] == '30') ? ' selected="selected"' : '';
			$html .= '<option value="30"'.$sel.'>30</option>'."\n";
			$sel = ($initial_values['dropmin'] == '45') ? ' selected="selected"' : '';
			$html .= '<option value="45"'.$sel.'>45</option>'."\n";
			$html .= "</select>\n";
			$html .= '</div>';//iosrt_minbox
			$html .= '</div>';//iosrt_timebox
		}

		$high = ($initial_values['dropdate'] == '') ? ' onclick="iosRTHighlight(\'iosrt_droplid\');"' : '';
		if ($this->options['hidedrop'] == 1) { $high = ''; }
		$class = ($this->options['dropofftime'] != '') ? 'iosrt_dateonlybox' : 'iosrt_datebox';

		$html .= '<div class="'.$class.'"><input type="text" name="dropdate" id="iosrt_dropdate" value="'.$initial_values['dropdate'].'"'.$high.' maxlength="10" dir="ltr" class="iosrt_text" placeholder="'.$eLang->get('DATE_PLACEHOLDER').'" pattern="[0-9]{2}-[0-9]{2}-[0-9]{4}" readonly="readonly" /></div>';
		$html .= "</div>\n";//iosrt_datetimewrap

		$html .= "</div>\n";//iosrt_2colflexbox or "iosrt_spad"
		if ($this->options['hidedrop'] == 0) {
			$html .= "</div>\n";//iosrt_2colflexwrap
		}

		$html .= '<script>'."\n";
		$lng = strtolower($this->options['guilang']);
		if (file_exists($this->path.'/js/flatpickri18n/'.$lng.'.js')) {
			$floc = ($lng == 'el') ? 'gr' : $lng;
			$html .= 'flatpickr.localize(flatpickr.l10ns.'.$floc.');'."\n";
		}
		$html .= 'flatpickr.l10ns.default.firstDayOfWeek = 1;'."\n";//Monday
		$html .= "</script>\n";

		$formvtypes = $vtypes;
		if ($this->options['unifiedmoto'] == 1) {
			if (count($vtypes) > 1) {
				$unified = array('moto', 'scooter', 'atv', 'buggy');
				$formvtypes = array();
				foreach ($vtypes as $vtype) {
					if (($vtype == 'moto') || ($vtype == 'scooter') || ($vtype == 'atv') || ($vtype == 'buggy')) {
						if (!in_array('moto', $formvtypes)) { $formvtypes[] = 'moto'; }
						continue; 
					}
					$formvtypes[] = $vtype;
				}
			}
		}

		if ($vehdata) {//fleet vehicle page
			$html .= '<input type="hidden" name="vtype" id="iosrt_vtype" value="'.$vehdata['vtype'].'" />'."\n";
			$html .= '<input type="hidden" name="mid" id="iosrt_mid" value="'.$vehdata['mid'].'" />'."\n";
			$html .= '<input type="hidden" name="gid" id="iosrt_gid" value="'.$vehdata['gid'].'" />'."\n";
		} else if (count($formvtypes) > 1) {
			if ($initial_values['pickdate'] != '') {
				$presel = $initial_values['vtype'];
			} else {
				$presel = '';
			}

			$html .= '<div class="iosrt_2colflexwrap">'."\n";
			$html .= '<div class="iosrt_2colflexbox iosrt_spad">'."\n";

			$html .= '<label class="iosrt_labelblock" for="iosrt_vtype">'.$eLang->get('VEHICLE_TYPE')."</label>\n";
			$html .= '<select name="vtype" id="iosrt_vtype" class="iosrt_select" tabindex="-1">'."\n";
			$anyv = implode(',', $formvtypes);
			$sel = ($presel == '') ? ' selected="selected"' : '';
			$html .= '<option value="'.$anyv.'"'.$sel.'>--- '.$eLang->get('ANY_VEHICLE')." ---</option>\n";
			foreach ($formvtypes as $vtype) {
				switch ($vtype) {
					case 'car': $title = $eLang->get('CARS'); break;
					case 'moto': $title = $eLang->get('MOTORCYCLES'); break;
					case 'scooter': $title = $eLang->get('SCOOTERS'); break;
					case 'boat': $title = $eLang->get('BOATS'); break;
					case 'atv': $title = $eLang->get('ATVS'); break;
					case 'buggy': $title = $eLang->get('BUGGIES'); break;
					case 'bicycle': $title = $eLang->get('BICYCLES'); break;
					case 'other': $title = $eLang->get('OTHER_VEHICLES'); break;
					default: $title = ucfirst($vtype); break;
				}
				$sel = ($vtype == $presel) ? ' selected="selected"' : '';
				$html .= '<option value="'.$vtype.'"'.$sel.'>'.$title."</option>\n";
			}
			$html .= "</select>\n";
			$html .= "</div>\n";//iosrt_2colflexbox
			$html .= "</div>\n";//iosrt_2colflexwrap
		} else {
			$html .= '<input type="hidden" name="vtype" id="iosrt_vtype" value="'.$formvtypes[0].'" />'."\n";
		}

		if (trim($this->options['formtext1']) != '') {
			$k = $this->options['formtext1'];
			$txt = $eLang->exist($k) ? $eLang->get($k) : $k;
			$html .= '<div class="iosrt_formtext1">'.$txt."</div>\n";
		}

		if (trim($this->options['formtext2']) != '') {
			$k = $this->options['formtext2'];
			$txt = $eLang->exist($k) ? $eLang->get($k) : $k;
			$html .= '<div class="iosrt_formtext2">'.$txt."</div>\n";
		}

		$html .= '<input type="hidden" name="step" value="1" />'."\n";//step : "next step"

		$btnclass = $sidecolumn ? 'iosrt_btn' : 'iosrt_sbtn';
		$html .= '<div class="iosrt_vsspace iosrt_spad"><button type="button" name="sbt" id="iosrt_sbt" class="'.$btnclass.'" onclick="iosRTAVFSubmit();">'.$eLang->get('SEARCH')."</button></div>\n";
		$txt = sprintf($eLang->get('CLOSED_ON_DATE'), 'ZZZZZ');
		$txt1 = sprintf($eLang->get('MINRENT_FORIS'), 'ZZZZZ', 'QQQQQ');
		$txt2 = sprintf($eLang->get('MAXRENT_FORIS'), 'ZZZZZ', 'QQQQQ');
		$txt3 = sprintf($eLang->get('CLOSED_FROMTO'), 'ZZZZZ', 'QQQQQ');
		$txt4 = sprintf($eLang->get('SELPICK_AFTER_DATETIME'), 'ZZZZZ');
		$txt5 = sprintf($eLang->get('SELPICK_AFTER_TIME'), 'ZZZZZ');
		$html .= '<div class="iosrt_invisible" id="iosrt_sfmlang" data-closedon="'.$txt.'" data-minis="'.$txt1.'" data-maxis="'.$txt2.'" data-closedtime="'.$txt3.'" data-afterdatetime="'.$txt4.'" data-aftertime="'.$txt5.'" data-dropafterpick="'.$eLang->get('SELDROP_AFTERPICK').'"></div>'."\n";

		if ($initial_values['pickdate'] != '') {
			$html .= '<script>iosRTInitCalendar();</script>'."\n";
		}

		if ($company->rcancel == 1) {
			$addonclass = $sidecolumn ? ' iosrt_center' : '';
			$html .= '<div id="iosrt_rescancelbox" class="iosrt_vsspace iosrt_spad">';
			$html .= '<a href="javascript:void(null);" class="iosrt_reschange'.$addonclass.'" onclick="iosRTChangeForm();">'.$eLang->get('CANCEL_CHANGE_RESERVATION')."</a>";
			$html .= "</div>\n";
		}

		$html .= "</form>\n";
		$html .= "</div>\n";

		if (!$sidecolumn) {
			$html .= $this->makeFooter($eLang, $company);
		}
		return $html;
	}


	/*************************************/
	/* STEP 1 :: LIST AVAILABLE VEHICLES */
	/*************************************/
	private function listAvailableVehicles() {
		$eLang = $this->lang;

		$fdata = $this->getValAVFdata($eLang, true);
		if ($fdata['ok'] === false) {
			$errormsg = $fdata['errormsg'];
			unset($fdata);
			$html = $this->availabilityForm($errormsg);
			return $html;
		}

		$this->setHead();

		$company = $this->helper->basicRequest('company', true);
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}

		$places = $this->helper->basicRequest('places', true);

		$vtypes = $this->helper->searchableVehicleTypes($company);
		if (!$vtypes) {
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL')."<br />There are no valid vehicle types!</p></div>\n";
		}

		$results = $this->helper->searchRequest($fdata, $company);
		if ($results === false) {
			$noavail_lng = ($fdata['vtype'] == 'boat') ? $eLang->get('NO_AVAILABLE_BOATS') : $eLang->get('NO_AVAILABLE_VEHICLES');
			$this->errormsg = $this->helper->getError();
			$errormsg = ($this->errormsg == '') ? $noavail_lng : $this->errormsg;
			$html = $this->availabilityForm($errormsg);
			return $html;
		}

		if (!$results->groups) {
			$html = $this->noAvailableVehicles($results, $fdata, $eLang, $company);
			return $html;
		}

		require_once($this->path.'/currency.class.php');
		$currency = new iosrtApiCurrency($results->currency, $eLang->get('THOUSANDS_SEP'), $eLang->get('DECIMALS_SEP'), $eLang->get('CURRENCYFRONT'));

		if ($fdata['vtype'] == 'car') {
			$title = $eLang->get('AVAILABLE_CARS');
		} else if ($fdata['vtype'] == 'moto') {
			$title = $eLang->get('AVAILABLE_MOTOS');
		} else if ($fdata['vtype'] == 'boat') {
			$title = $eLang->get('AVAILABLE_BOATS');
		} else {
			$title = $eLang->get('AVAILABLE_VEHICLES');
		}

		$enc = md5($results->picklid.$results->droplid.$results->pickup.$results->dropoff.$fdata['vtype'].$this->options['prkey']);
		$newsearchlink = $this->makeFormURL().'?picklid='.$results->picklid.'&amp;droplid='.$results->droplid.'&amp;pickup='.urlencode($results->pickup).'&amp;dropoff='.urlencode($results->dropoff).'&amp;vtype='.$fdata['vtype'].'&amp;enc='.urlencode($enc);

		$modelsfeatures = $this->helper->basicRequest('modelsfeatures');
		$priceincludes = $this->helper->basicRequest('priceincludes');
		$categories = $this->helper->basicRequest('categories');

		$veh_notaval_msg = '';
		if (isset($fdata['gid']) && isset($fdata['mid'])) {//from fleet page
			if (($fdata['gid'] > 0) && ($fdata['mid'] > 0)) {
				if ($results->groups) {
					$gid = $fdata['gid'];
					$mid = $fdata['mid'];
					if (!isset($results->groups[$gid])) {
						$veh_notaval_msg = $eLang->get('VEHNOTAV_ALTERNATIVES');
					} else {
						if (!isset($results->groups[$gid]['models'][$mid])) {
							$veh_notaval_msg = $eLang->get('VEHNOTAV_ALTERNATIVES');
						}
					}
				}
			}
		}

		$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";
		if ($this->options['sidecolumn'] != '') {
			$html .= '<h1 class="iosrt_h1">'.$title.'</h1>';
			if (strpos($this->options['sidecolumn'], 'search') === false) {
				$html .= '<div class="iosrt_zero"><a href="'.$newsearchlink.'" class="iosrt_newsearch">'.$eLang->get('NEW_SEARCH')."</a></div>\n";
			}
			if ($veh_notaval_msg != '') { $html .= '<div class="iosrt_warning"><span>i</span> '.$veh_notaval_msg."</div>\n"; }
			$html .= '<div class="iosrt_colswrap">'."\n";
			$html .= '<div class="iosrt_sidecol">'."\n";
			$html .= $this->sideColumn($results->groups, $categories, $modelsfeatures, $fdata, $eLang);
			$html .= "</div>\n";//iosrt_sidecol
			$html .= '<div class="iosrt_maincol">'."\n";
		} else {
			$html .= '<h1 class="iosrt_h1">'.$title.'</h1>';
			$html .= '<div class="iosrt_zero"><a href="'.$newsearchlink.'" class="iosrt_newsearch">'.$eLang->get('NEW_SEARCH')."</a></div>\n";
			if ($veh_notaval_msg != '') { $html .= '<div class="iosrt_warning"><span>i</span> '.$veh_notaval_msg."</div>\n"; }
		}

		$bdata = array(
			'pickup' => $results->pickup,
			'dropoff' => $results->dropoff,
			'picklid' => $results->picklid,
			'droplid' => $results->droplid,
			'urlbase' => $this->makeFormURL()
		);

		$num_shown = 0;
		$all_mids = array();
		$rental_duration_txt = $this->avVehiclesDuration($results->days, $results->hours, $results->charge_days, $results->charge_hours, $eLang);
		foreach ($results->groups as $gid => $group) {
			if ($group['models']) {
				foreach ($group['models'] as $mid => $model) {
					$html .= $this->makeModelHTML($gid, $mid, $group, $modelsfeatures, $priceincludes, $categories, $rental_duration_txt, $company, $eLang, $currency, $bdata, $places);
					$num_shown++;
					$all_mids[] = $mid;
				}
			}
		}

		unset($modelsfeatures, $priceincludes, $categories);
		if ($num_shown > 3) {
			if (strpos($this->options['sidecolumn'], 'search') === false) {
				$html .= '<p><a href="'.$this->makeFormURL().'" class="iosrt_newsearch">'.$eLang->get('NEW_SEARCH')."</a></p>\n";
			}
		}

		$html .= '<div class="iosrt_invisible" id="iosrt_novehmatch_filters">'.$eLang->get('NOVEH_MATCH_FILTERS').'</div>'."\n";
		$html .= '<div class="iosrt_invisible" id="iosrt_all_models" data-mids="'.implode(';', $all_mids).'"></div>'."\n";

		if ($this->options['sidecolumn'] != '') {
			$html .= "</div>\n";//iosrt_maincol
			$html .= "</div>\n";//iosrt_colswrap
		}

		$html .= $this->makeFooter($eLang, $company);

		if ($this->hasgallery) {
			$html .= "<script>\n";
			$html .= ' var iosRTSwiper = new Swiper (\'.iosrt_swiper\', {'."\n";
			$html .= 'loop: true, pagination: { el: \'.swiper-pagination\' }, navigation: { nextEl: \'.swiper-button-next\', prevEl: \'.swiper-button-prev\'}'."\n";
			$html .= "})\n";
			$html .= "</script>\n";
		}
		return $html;
	}


	/********************************************************/
	/* RENTAL DURATION FOR SEARCH AVAILABILITY LISTING PAGE */
	/********************************************************/
	private function avVehiclesDuration($days, $hours, $charge_days, $charge_hours, $eLang) {
		if ($this->options['rentalduration'] == 2) {//charge days/hours
			$vd = $charge_days;
			$vh = $charge_hours;
		} else {//real / real+charge (only real)
			$vd = $days;
			$vh = $hours;
		}
		if ($vd == 0) {
			if ($vh == 1) {
				$durtxt = $eLang->get('PRICE_FOR_ONE_HOUR');
			} else {
				$durtxt = sprintf($eLang->get('PRICE_FOR_X_HOURS'), $vh);
			}
			return $durtxt;
		}

		if ($vd == 1) {
			if ($vh == 0) {
				$durtxt = $eLang->get('PRICE_FOR_ONE_DAY');
			} else if ($vh == 1) {
				$durtxt = $eLang->get('PRICE_FOR_1DAY_1HOUR');
			} else {
				$durtxt = sprintf($eLang->get('PRICE_FOR_1DAY_XHOURS'), $vh);
			}
			return $durtxt;
		}

		if ($vh == 0) {
			$durtxt = sprintf($eLang->get('PRICE_FOR_X_DAYS'), $vd);
		} else if ($vh == 1) {
			$durtxt = sprintf($eLang->get('PRICE_FOR_XDAYS_1HOUR'), $vd);
		} else {
			$durtxt = sprintf($eLang->get('PRICE_FOR_XDAYS_XHOURS'), $vd, $vh);
		}

		return $durtxt;
	}


	/*************************************/
	/* STEP 1/B :: NO AVAILABLE VEHICLES */
	/*************************************/
	private function noAvailableVehicles($results, $fdata, $eLang, $company) {
		if ($fdata['vtype'] == 'car') {
			$title = $eLang->get('AVAILABLE_CARS');
		} else if ($fdata['vtype'] == 'moto') {
			$title = $eLang->get('AVAILABLE_MOTOS');
		} else if ($fdata['vtype'] == 'boat') {
			$title = $eLang->get('AVAILABLE_BOATS');
		} else {
			$title = $eLang->get('AVAILABLE_VEHICLES');
		}

		$enc = md5($results->picklid.$results->droplid.$results->pickup.$results->dropoff.$fdata['vtype'].$this->options['prkey']);
		$newsearchlink = $this->makeFormURL().'?picklid='.$results->picklid.'&amp;droplid='.$results->droplid.'&amp;pickup='.urlencode($results->pickup).'&amp;dropoff='.urlencode($results->dropoff).'&amp;vtype='.$fdata['vtype'].'&amp;enc='.urlencode($enc);

		$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";
		$html .= '<h1 class="iosrt_h1">'.$title.'</h1>';
		$html .= '<p><a href="'.$newsearchlink.'" class="iosrt_newsearch">'.$eLang->get('NEW_SEARCH')."</a></p>\n";
		$msg = $eLang->get('NO_AVAILABLE_VEHICLES');
		if (isset($fdata['vtype'])) {
			switch ($fdata['vtype']) {
				case 'car': $msg = $eLang->get('NO_AVAILABLE_CARS'); break;
				case 'moto': $msg = $eLang->get('NO_AVAILABLE_MOTOS'); break;
				case 'boat': $msg = $eLang->get('NO_AVAILABLE_BOATS'); break;
				default: $msg = $eLang->get('NO_AVAILABLE_VEHICLES'); break;
			}
		}
		$html .= '<div class="iosrt_error"><p>'.$msg."</p></div>\n";
		$html .= $this->makeFooter($eLang, $company);
		return $html;
	}


	/***********************/
	/* STEP 2 :: BOOK FORM */
	/***********************/
	private function bookForm($errormsg='') {
		$eLang = $this->lang;

		$fdata = $this->getValModelseldata($eLang);
		if ($fdata['ok'] === false) {
			$errormsg = $fdata['errormsg'];
			unset($fdata);
			$html = $this->availabilityForm($errormsg);
			return $html;
		}

		$this->setHead();

		$company = $this->helper->basicRequest('company');
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}

		$response = $this->helper->beginBookRequest($fdata);
		if ($response === false) {
			$this->errormsg = $this->helper->getError();
			$html = $this->availabilityForm($this->errormsg);
			return $html;
			//return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$this->errormsg."</p></div>\n";
		}

		$categories = $this->helper->basicRequest('categories');
		$priceincludes = $this->helper->basicRequest('priceincludes');

		require_once($this->path.'/currency.class.php');
		$currency = new iosrtApiCurrency($response->currency, $eLang->get('THOUSANDS_SEP'), $eLang->get('DECIMALS_SEP'), $eLang->get('CURRENCYFRONT'));

		$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";
		$html .= '<div class="iosrt_book_flexwrap">'."\n";
		$html .= '<div class="iosrt_book_mainbox">'."\n";

		$img = ($response->vehicle['image'] != '') ? $company->imagesurl.'large/'.$response->vehicle['image'] : $this->options['clienturl'].'/images/noimage.png';

		if ($response->vehicle['vtype'] == 'boat') {
			$html .= '<h2 class="iosrt_block_h2"><span class="circle_block">1</span> '.$eLang->get('BOAT_DETAILS')."</h2>\n";
		} else {
			$html .= '<h2 class="iosrt_block_h2"><span class="circle_block">1</span> '.$eLang->get('VEHICLE_DETAILS')."</h2>\n";
		}

		$html .= '<div class="iosrt_model iosrt_boxbookfix">'."\n";
		$html .= '<div class="iosrt_model_colswrap">'."\n";
		$html .= '<div class="iosrt_model_main">'."\n";
		
		if ($this->options['titlepos'] == 0) {
			if ($response->vehicle['vtype'] == 'boat') {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$response->vehicle['model'].'</span></h2>'."\n";
			} else {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$response->vehicle['model'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span></h2>'."\n";
			}
			$html .= '<div class="iosrt_model_subtitle">'.$this->makeModelSubtitle($response->vehicle['group'], $response->vehicle['vtype'], $response->vehicle['categories'], $categories)."</div>\n";
			switch ($response->vehicle['vtype']) {
				case 'car': $txt = $eLang->get('CHANGE_CAR'); break;
				case 'boat': $txt = $eLang->get('CHANGE_BOAT'); break;
				default: $txt = $eLang->get('CHANGE_VEHICLE'); break;
			}

			$link = $this->makeFormURL().'?picklid='.$response->picklid.'&amp;pickup='.urlencode($response->pickup).'&amp;droplid='.$response->droplid.'&amp;dropoff='.urlencode($response->dropoff).'&amp;vtype='.$response->vehicle['vtype'].'&amp;step=1';
			$html .= '<a href="'.$link.'" class="iosrt_change_veh" title="'.$txt.'">'.$txt."</a>\n";
		}

		switch ($this->options['imagewidth']) {
			case 480: $imgaddon = ' iosrt_model_img480'; $feataddon = ' iosrt_model_feat480'; break;
			case 400: $imgaddon = ' iosrt_model_img400'; $feataddon = ' iosrt_model_feat400'; break;
			case 320: $imgaddon = ' iosrt_model_img320'; $feataddon = ' iosrt_model_feat320'; break;
			case 240: default: $imgaddon = ''; $feataddon = ''; break;
		}

		$html .= '<div class="iosrt_model_imgfeat">'."\n";
		$html .= '<figure class="iosrt_model_img'.$imgaddon.'">';
		$html .= '<img src="'.$img.'" alt="'.$response->vehicle['model'].'" />'."\n";
		$html .= '</figure>'."\n";//iosrt_model_img

		$txt = '';
		if ($this->options['titlepos'] > 0) {
			if ($response->vehicle['vtype'] == 'boat') {
				$txt .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$response->vehicle['model'].'</span></h2>'."\n";
			} else {
				$txt .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$response->vehicle['model'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span></h2>'."\n";
			}
			$txt .= '<div class="iosrt_model_subtitle">'.$this->makeModelSubtitle($response->vehicle['group'], $response->vehicle['vtype'], $response->vehicle['categories'], $categories)."</div>\n";
			switch ($response->vehicle['vtype']) {
				case 'car': $title = $eLang->get('CHANGE_CAR'); break;
				case 'boat': $title = $eLang->get('CHANGE_BOAT'); break;
				default: $title = $eLang->get('CHANGE_VEHICLE'); break;
			}
			$link = $this->makeFormURL().'?picklid='.$response->picklid.'&amp;pickup='.urlencode($response->pickup).'&amp;droplid='.$response->droplid.'&amp;dropoff='.urlencode($response->dropoff).'&amp;vtype='.$response->vehicle['vtype'].'&amp;step=1';
			$txt .= '<a href="'.$link.'" class="iosrt_change_veh iosrt_dspace" title="'.$title.'">'.$title."</a>\n";
		}

		$txt .= $this->makePriceIncludesHTML($response->vehicle['mid'], $response->vehicle['priceincludes'], $priceincludes, $eLang, false);
		if ($txt != '') {
			$html .= '<div class="iosrt_model_feat'.$feataddon.'">'.$txt.'</div>'."\n";
		}
		$html .= "</div>\n";//iosrt_model_imgfeat
		$html .= "</div>\n";//iosrt_model_main
		$html .= "</div>\n";//iosrt_model_colswrap
		$html .= "</div>\n";//iosrt_model

		$html .= "</div>\n";//iosrt_book_mainbox
		$html .= '<div class="iosrt_book_sidebox">'."\n";
		$html .= '<div class="iosrt_pad">'."\n";
		$html .= '<h3 class="iosrt_side_h3">'.$eLang->get('BOOKING_DETAILS')."</h3>\n";
		$html .= '<div class="iosrt_dbox">';
		$html .= '<h4 class="iosrt_dbox_h4">'.$eLang->get('PICKUP')."</h4>\n";
		$html .= '<div class="iosrt_dbox_date">'.$this->formatDate($response->pickup, $eLang->get('DATETIME_FORMAT_1')).'</div>';
		$html .= '<div class="iosrt_dbox_loc"><span>'.$eLang->get('FROM').'</span> '.$response->pickarea.', '.$response->picklocation.'</div>';
		$html .= "</div>\n";
		$html .= '<div class="iosrt_dbox">';
		$html .= '<h4 class="iosrt_dbox_h4">'.$eLang->get('DROPOFF')."</h4>\n";
		$html .= '<div class="iosrt_dbox_date">'.$this->formatDate($response->dropoff, $eLang->get('DATETIME_FORMAT_1')).'</div>';
		$html .= '<div class="iosrt_dbox_loc"><span>'.$eLang->get('AT').'</span> '.$response->droparea.', '.$response->droplocation.'</div>';
		$html .= "</div>\n";
		$parts = array();
		if ($response->days > 0) { $parts[] = ($response->days == 1) ? '1 '.$eLang->get('DAY') : $response->days.' '.$eLang->get('DAYS'); }
		if ($response->hours > 0) { $parts[] = ($response->hours == 1) ? '1 '.$eLang->get('HOUR') : $response->hours.' '.$eLang->get('HOURS'); }
		$txt = implode(' '.$eLang->get('AND').' ', $parts);
		$charge_txt = '';
		if (($response->days != $response->charge_days) || ($response->hours != $response->charge_hours)) {
			$parts = array();
			if ($response->charge_days > 0) { $parts[] = ($response->charge_days == 1) ? '1 '.$eLang->get('DAY') : $response->charge_days.' '.$eLang->get('DAYS'); }
			if ($response->charge_hours > 0) { $parts[] = ($response->charge_hours == 1) ? '1 '.$eLang->get('HOUR') : $response->charge_hours.' '.$eLang->get('HOURS'); }
			$charge_txt = implode(' '.$eLang->get('AND').' ', $parts);
		}
		
		$html .= '<div class="iosrt_dbox">';
		if ($this->options['rentalduration'] == 2) {//charge days/hours
			$html .= '<h4 class="iosrt_dbox_h4">'.$eLang->get('CHARGE_DURATION')."</h4>\n";
			if ($charge_txt != '') {
				$html .= '<div class="iosrt_dbox_dur">'.$charge_txt.'</div>';
			} else {
				$html .= '<div class="iosrt_dbox_dur">'.$txt.'</div>';
			}
		} else if ($this->options['rentalduration'] == 1) {//real days/hours
			$html .= '<h4 class="iosrt_dbox_h4">'.$eLang->get('RENTAL_DURATION')."</h4>\n";
			$html .= '<div class="iosrt_dbox_dur">'.$txt.'</div>';
		} else {//0, default, real days/hours + charge days/hours
			$html .= '<h4 class="iosrt_dbox_h4">'.$eLang->get('RENTAL_DURATION')."</h4>\n";
			$html .= '<div class="iosrt_dbox_dur">'.$txt.'</div>';
			if ($charge_txt != '') { $html .= '<div class="iosrt_dbox_chargedur">('.$eLang->get('CHARGE').' '.$charge_txt.')</div>'; }
		}
		$html .= "</div>\n";
		$html .= "</div>\n";//iosrt_pad
		$html .= "</div>\n";//iosrt_book_sidebox

		$html .= "</div>\n";//iosrt_book_flexwrap

		$nextblock = 2;
		if ($this->options['showcoupon'] == 1) {
			$html .= '<h2 class="iosrt_block_h2"><span class="circle_block">'.$nextblock.'</span> '.$eLang->get('DISCOUNT_COUPON')."</h2>\n";

			$html .= '<div class="iosrt_coupon_box">'."\n";
			$html .= '<form name="fmcoupon" id="iosrt_fmcoupon" method="post" action="'.$this->options['clienturl'].'/ajax.php" class="iosrt_form">'."\n";
			$html .= '<p class="iosrt_coupon_tip">'.$eLang->get('DISCOUNT_COUPON_TIP').'</p>';
			$html .= '<div class="iosrt_couponwrap">'."\n";
			$html .= '<div class="iosrt_couponbtnbox">'."\n";
			$html .= '<button name="couponbtn" id="iosrt_couponbtn" type="button" class="iosrt_couponbtn" onclick="iosrtCheckCoupon();" data-apply="'.$eLang->get('APPLY').'" data-wait="'.$eLang->get('WAIT').'">'.$eLang->get('APPLY')."</button>\n";
			$html .= "</div>\n";
			$html .= '<div class="iosrt_couponcodebox">'."\n";
			$html .= '<input type="text" name="checkcoupon" id="iosrt_checkcoupon" class="iosrt_text" value="" placeholder="'.$eLang->get('COUPON_CODE').'" title="'.$eLang->get('COUPON_CODE').'" required="required" />'."\n";
			$html .= '<input type="hidden" name="guilang" id="iosrt_ccguilang" value="'.$this->options['guilang'].'" />'."\n";
			$html .= '<input type="hidden" name="lang" id="iosrt_cclang" value="'.$this->options['lang'].'" />'."\n";
			$html .= '<input type="hidden" name="currency" id="iosrt_cccurrency" value="'.$this->options['currency'].'" />'."\n";
			$html .= '<input type="hidden" name="apiid" id="iosrt_ccapiid" value="'.$this->options['apiid'].'" />'."\n";
			$html .= "</div>\n";//iosrt_couponcodebox
			$html .= "</div>\n";//iosrt_couponwrap
			$html .= '<div id="iosrt_checkcouponmsg" class="iosrt_invisible"></div>'."\n";
			$html .= "</form>\n";
			$txt = sprintf($eLang->get('COUPON_GOT_DISCOUNT'), '<strong>AAAAA</strong>', '<strong>BBBBB</strong>');
			$html .= '<p id="iosrt_couponwonmsg" class="iosrt_invisible">'.$txt.'</p>';
			$html .= "</div>\n";//iosrt_coupon_box

			$nextblock++;
		} else {
			$html .= '<input type="hidden" name="guilang" id="iosrt_ccguilang" value="'.$this->options['guilang'].'" />'."\n";
			$html .= '<input type="hidden" name="lang" id="iosrt_cclang" value="'.$this->options['lang'].'" />'."\n";
			$html .= '<input type="hidden" name="currency" id="iosrt_cccurrency" value="'.$this->options['currency'].'" />'."\n";
			$html .= '<input type="hidden" name="apiid" id="iosrt_ccapiid" value="'.$this->options['apiid'].'" />'."\n";
		}

		$price_data = array('subtotal' => 0.00, 'tax' => 0.00, 'totalprice' => 0.00, 'feeamount' => 0.00, 'feeamount_note' => '', 'use_fullamount' => false);
		if (isset($_GET['fullamount'])) {
			if (intval($_GET['fullamount']) == 1) {
				if (($response->fullamountpc > 0) && ($company->surfee != '100pc')) { $price_data['use_fullamount'] = true; }
			}
		}

		$cur_pid = 0;
		$cur_paymod = '';
		$cur_paymod_surfee = -1;
		if ($response->paymods) {
			foreach ($response->paymods as $pid => $paymod) {
				if (($cur_paymod == 'cashondelivery') && ($price_data['use_fullamount'] == true)) { continue; }//donot use "cashondelivery" with fullamount discount
				$cur_pid = $pid;
				$cur_paymod = $paymod['name'];
				if ($cur_paymod == 'cashondelivery') { $cur_paymod_surfee = 0; }
				break;
			}
		}

		$earlybook = ($response->earlybookpc > 0) ? ($response->rentalfee * $response->earlybookpc) / 100 : 0.00;
		$fullamount = $price_data['use_fullamount'] ? ($response->rentalfee * $response->fullamountpc) / 100 : 0.00;

		$price_data['subtotal'] = $response->rentalfee;
		$price_data['subtotal'] += $response->extrahoursfee;
		$price_data['subtotal'] += $response->afterhourspick;
		$price_data['subtotal'] += $response->afterhoursdrop;
		$price_data['subtotal'] += $response->pickupfee;
		$price_data['subtotal'] += $response->dropofffee;
		$price_data['subtotal'] += $response->preselextras;
		$price_data['subtotal'] -= $earlybook;
		$price_data['subtotal'] -= $fullamount;
		if ($company->tax > 0) {
			if ($company->taxispc == 1) {
				$price_data['tax'] = ($price_data['subtotal'] * $company->tax) / 100;
			} else {
				$price_data['tax'] = $company->tax;
			}
		}
		$price_data['totalprice'] = $price_data['subtotal'] + $price_data['tax'];

		if ($fullamount > 0) {
			$price_data['feeamount'] = $price_data['totalprice'];
			$price_data['feeamount_note'] = ' <span class="iosrt_sfeenote" id="iosrt_sfeenote">(100%)</span>';
		} else if ($cur_paymod_surfee > -1) {
			$price_data['feeamount'] = ($price_data['totalprice'] * $cur_paymod_surfee) / 100;
			if ($cur_paymod_surfee > 0) {
				$price_data['feeamount_note'] = ' <span class="iosrt_sfeenote" id="iosrt_sfeenote">('.$cur_paymod_surfee.'%)</span>';
			} else {
				$price_data['feeamount_note'] = ' <span class="iosrt_invisible" id="iosrt_sfeenote">('.$cur_paymod_surfee.'%)</span>';
			}
		} else if (strpos($company->surfee, 'pc') === false) {
			$price_data['feeamount'] = (int)$company->surfee;
			$price_data['feeamount_note'] = ' <span class="iosrt_invisible" id="iosrt_sfeenote">(0%)</span>';
		} else {
			$feepc = intval(str_replace('pc', '', $company->surfee));
			$price_data['feeamount'] = ($price_data['totalprice'] * $feepc) / 100;
			if ($feepc > 0) {
				$price_data['feeamount_note'] = ' <span class="iosrt_sfeenote" id="iosrt_sfeenote">('.$feepc.'%)</span>';
			} else {
				$price_data['feeamount_note'] = ' <span class="iosrt_invisible" id="iosrt_sfeenote">('.$feepc.'%)</span>';
			}
		}

		$price_data['feeamount'] = number_format($price_data['feeamount'], 2, '.', '');
		if ($price_data['feeamount'] > $price_data['totalprice']) { $price_data['feeamount'] = $price_data['totalprice']; }

		$html .= '<form name="fmbook" id="iosrt_fmbook" method="post" action="'.$this->options['clienturl'].'/ajax.php" class="iosrt_form" onsubmit="return false;">'."\n";

		$html .= '<div class="iosrt_book_flexwrap">'."\n";
		$html .= '<div class="iosrt_book_mainbox">'."\n";

		if ($response->extras) {
			$html .= $this->makeExtrasHTML($response->extras, $response->extrasgroups, $response->pickup, $nextblock, $eLang, $currency, $company, $response->vehicle['modelparams'], $response->charge_days);
			$nextblock++;

			$html .= "</div>\n";//iosrt_book_mainbox

			$html .= '<div class="iosrt_book_sidebox">'."\n";
			$html .= '<div class="iosrt_pad">'."\n";
			$html .= $this->priceAnalysis($response, $price_data, $company, $currency, $eLang);
			$html .= "</div>\n";//iosrt_pad
			$html .= "</div>\n";//iosrt_book_sidebox

			$html .= "</div>\n";//iosrt_book_flexwrap

			$html .= $this->cashPaymentSelection($response, $price_data, $eLang);
			$html .= $this->personalDetails($response, $nextblock, $eLang, false);
			$nextblock++;
		} else {
			$html .= $this->personalDetails($response, $nextblock, $eLang, true);
			$nextblock++;
			$html .= "</div>\n";//iosrt_book_mainbox

			$html .= '<div class="iosrt_book_sidebox">'."\n";
			$html .= '<div class="iosrt_pad">'."\n";
			$html .= $this->priceAnalysis($response, $price_data, $company, $currency, $eLang);
			$html .= "</div>\n";//iosrt_pad
			$html .= "</div>\n";//iosrt_book_sidebox

			$html .= "</div>\n";//iosrt_book_flexwrap

			$html .= $this->cashPaymentSelection($response, $price_data, $eLang);
		}

		if ($response->paymods) {
			$html .= $this->paymentOptions($response, $nextblock, $eLang, $price_data);
			$nextblock++;
		}

		if (trim($company->terms) != '') {
			$html .= '<label class="iosrt_checkwrap">'.$eLang->get('AGREE_TERMS_CONDS');
			$html .= '<input type="checkbox" name="agreeterms" id="iosrt_agreeterms"><span class="iosrt_ccheckbox"></span></label>'."\n";
			$html .= '<div class="iosrt_terms">'.stripslashes($company->terms)."</div>\n";
			$html .= '<div id="iosrt_warnterms" class="iosrt_invisible">'.$eLang->get('CONMUSTAGREE_TERMS_CONDS')."</div>\n";
		}

		$html .= '<div class="iosrt_bookbtn_wrap"><button type="button" name="bookbtn" class="iosrt_bookbtn" onclick="iosRTBook();">'.$eLang->get('BOOK_BUTTON').'</button></div>'."\n";

		$html .= '<input type="hidden" name="rentalfee" id="iosrt_rentalfee" dir="ltr" value="'.number_format($response->rentalfee, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="extrahoursfee" id="iosrt_extrahoursfee" dir="ltr" value="'.number_format($response->extrahoursfee, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="pickupfee" id="iosrt_pickupfee" dir="ltr" value="'.number_format($response->pickupfee, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="dropofffee" id="iosrt_dropofffee" dir="ltr" value="'.number_format($response->dropofffee, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="afterhourspick" id="iosrt_afterhourspick" dir="ltr" value="'.number_format($response->afterhourspick, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="afterhoursdrop" id="iosrt_afterhoursdrop" dir="ltr" value="'.number_format($response->afterhoursdrop, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="couponcode" id="iosrt_couponcode" dir="ltr" value="" />'."\n";
		$html .= '<input type="hidden" name="couponamount" id="iosrt_couponamount" dir="ltr" value="0.00" data-amount="0.00" data-ispc="1" />'."\n";
		$html .= '<input type="hidden" name="extrasprice" id="iosrt_extrasprice" dir="ltr" value="'.number_format($response->preselextras, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="earlybook" id="iosrt_earlybook" dir="ltr" value="'.number_format($earlybook, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="fullamount" id="iosrt_fullamount" dir="ltr" value="'.number_format($fullamount, 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="subtotal" id="iosrt_subtotal" dir="ltr" value="'.number_format($price_data['subtotal'], 2, '.', '').'" />'."\n";
		$html .= '<input type="hidden" name="tax" id="iosrt_tax" dir="ltr" value="'.number_format($price_data['tax'], 2, '.', '').'" data-cfgtax="'.$company->tax.'" data-cfgtaxispc="'.$company->taxispc.'" />'."\n";
		$html .= '<input type="hidden" name="totalprice" id="iosrt_totalprice" dir="ltr" value="'.number_format($price_data['totalprice'], 2, '.', '').'" />'."\n";
		$surfeepc = ($price_data['use_fullamount']) ? '100pc' : $company->surfee;
		$html .= '<input type="hidden" name="feeamount" id="iosrt_feeamount" dir="ltr" value="'.number_format($price_data['feeamount'], 2, '.', '').'" data-surfee="'.$surfeepc.'" data-paymodsurfee="'.$cur_paymod_surfee.'" />'."\n";
		$html .= '<input type="hidden" name="pickup" id="iosrt_pickup" dir="ltr" value="'.$response->pickup.'" />'."\n";
		$html .= '<input type="hidden" name="dropoff" id="iosrt_dropoff" dir="ltr" value="'.$response->dropoff.'" />'."\n";
		$html .= '<input type="hidden" name="picklid" id="iosrt_picklid" dir="ltr" value="'.$response->picklid.'" />'."\n";
		$html .= '<input type="hidden" name="droplid" id="iosrt_droplid" dir="ltr" value="'.$response->droplid.'" />'."\n";
		//$html .= '<input type="hidden" name="days" id="iosrt_days" dir="ltr" value="'.$response->days.'" />'."\n";
		//$html .= '<input type="hidden" name="hours" id="iosrt_hours" dir="ltr" value="'.$response->hours.'" />'."\n";
		//$html .= '<input type="hidden" name="extrahours" id="erextrahours" dir="ltr" value="'.$response->extrahours.'" />'."\n";
		$html .= '<input type="hidden" name="charge_days" id="iosrt_charge_days" dir="ltr" value="'.$response->charge_days.'" />'."\n";
		$html .= '<input type="hidden" name="charge_hours" id="iosrt_charge_hours" dir="ltr" value="'.$response->charge_hours.'" />'."\n";

		$html .= '<input type="hidden" name="vtype" id="iosrt_vtype" dir="ltr" value="'.$response->vehicle['vtype'].'" />'."\n";
		$html .= '<input type="hidden" name="gid" id="iosrt_gid" dir="ltr" value="'.$response->vehicle['gid'].'" />'."\n";
		$html .= '<input type="hidden" name="mid" id="iosrt_mid" dir="ltr" value="'.$response->vehicle['mid'].'" />'."\n";
		$html .= '<input type="hidden" name="pid" id="iosrt_pid" dir="ltr" value="'.$cur_pid.'" />'."\n";
		$html .= '<input type="hidden" name="paymod" id="iosrt_paymod" dir="ltr" value="'.$cur_paymod.'" />'."\n";
		$html .= '<input type="hidden" name="ubase" id="iosrt_ubase" dir="ltr" value="'.base64_encode($this->makeFormURL()).'" />'."\n";

		if (isset($_SERVER['HTTP_X_FORWARDED_FOR']) && ($_SERVER['HTTP_X_FORWARTDED_FOR'] != '')) {
			$ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
		} else if (isset($_SERVER['REMOTE_ADDR'])) {
			$ipaddress = $_SERVER['REMOTE_ADDR'];
		} else {
			$ipaddress = '';
		}
		$html .= '<input type="hidden" name="clip" id="iosrt_clip" dir="ltr" value="'.$ipaddress.'" />'."\n";

		$fuelprices = array('gasoline' => 0.00, 'diesel' => 0.00, 'gas' => 0.00, 'hydrogen' => 0.00, 'electric' => 0.00);
		if (trim($company->fuelprices) != '') { $fuelprices = json_decode(stripslashes($company->fuelprices), true); }
		$model_fueldata = array('tank' => 0, 'fuel' => 0);
		if (trim($response->vehicle['modelparams']) != '') {
			$pairs = explode(';', $response->vehicle['modelparams']);
			foreach ($pairs as $pair) {
				$parts = explode(':', $pair);
				if (count($parts) == 2) {
					$name = $parts[0];
					if ($name == 'tank') { $model_fueldata['tank'] = (int)$parts[1]; } else if ($name == 'fuel') { $model_fueldata['fuel'] = (int)$parts[1]; }
				}
			}
		}
		$html .= '<div class="iosrt_invisible" id="iosrt_algofuel" data-refuelcost="'.$company->refuelcost.'" data-gasoline="'.number_format($fuelprices['gasoline'], 2, '.', '').'" data-diesel="'.number_format($fuelprices['diesel'], 2, '.', '').'" data-gas="'.number_format($fuelprices['gas'], 2, '.', '').'" data-hydrogen="'.number_format($fuelprices['hydrogen'], 2, '.', '').'" data-electric="'.number_format($fuelprices['electric'], 2, '.', '').'"></div>'."\n";
		$html .= '<div class="iosrt_invisible" id="iosrt_algomodel" data-tank="'.$model_fueldata['tank'].'" data-fuel="'.$model_fueldata['fuel'].'"></div>'."\n";
		unset($fuelprices, $model_fueldata);

		$html .= "</form>\n";

		$html .= '<form name="fmppapi" id="iosrt_fmppapi" action="'.$this->options['clienturl'].'/ajax.php" method="post" class="iosrt_invisible">'."\n";
		$html .= '<button type="submit" name="ppsub">Pay</button>'."\n";
		$html .= "</form>\n";

		$html .= $this->makeFooter($eLang, $company);
		return $html;
	}


	/**************************************************/
	/* STEP 3/4/5 :: THANKS/PAY SUCCESS/PAY FAIL PAGE */
	/**************************************************/
	private function finalPage($page) {
		$eLang = $this->lang;

		$company = $this->helper->basicRequest('company');
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}
		$categories = $this->helper->basicRequest('categories');

		$this->helper->clearOldBookings();

		$rescode = $this->getFromRequest('rescode', '');
		$rtdata = array();
		if ($rescode != '') {
			$pat = "#([\']|[\.]|[\/]|[\!]|[\(]|[\)]|[\;]|[\"]|[\$]|[\#]|[\<]|[\>]|[\*]|[\%]|[\~]|[\`]|[\^]|[\|]|[\{]|[\}]|[\\\])#u";
			$rescode = urldecode($rescode);
			$rescode2 = trim(preg_replace($pat, '', $rescode));
			if ($rescode2 != $rescode) {//invalid
				$rescode = '';
			} else {
				$fname = strtolower($rescode);
				$cache_file = $this->helper->getCacheFile($fname, false, true);
				if ($cache_file == '') {
					$rescode = '';
				} else {
					include($cache_file);//$rtdata
				}
			}
		}

		$this->setHead();

		$html = '<div id="iosrt_pgwrap">'."\n";//global wrapper, used for printing this div
		if ($page == 4) {//payment success
			$html .= '<h1 class="iosrt_h1 iosrt_success">'.$eLang->get('YRESERV_COMPLETE')."</h1>\n";
			$html .= '<p class="iosrt_p">'.$eLang->get('YRESERV_PAIDCONF_DESC');
			if ($rescode != '') { $html .= '<br />'.$eLang->get('RESERVATION_CODE').': <strong>'.$rescode.'</strong>'; }
			$html .= "</p>\n";
		} else if ($page == 5) {//payment failure
			$html .= '<h1 class="iosrt_h1 iosrt_fail">'.$eLang->get('PAYMENT_FAILED')."</h1>\n";
			$html .= '<p class="iosrt_p">'.$eLang->get('YRESERV_FAIL_DESC');
			if ($rescode != '') { $html .= '<br />'.$eLang->get('RESERVATION_CODE').': <strong>'.$rescode.'</strong>'; }
			$html .= "</p>\n";
		} else {//3, success
			$html .= '<h1 class="iosrt_h1 iosrt_success">'.$eLang->get('YRESERV_COMPLETE')."</h1>\n";
			$html .= '<p class="iosrt_p">'.$eLang->get('YRESERV_COMPLETE_DESC');
			if ($rescode != '') { $html .= '<br />'.$eLang->get('RESERVATION_CODE').': <strong>'.$rescode.'</strong>'; }
			$html .= "</p>\n";
		}

		if ($rtdata) {
			$html .= '<h3 class="iosrt_side_h3">'.$eLang->get('BOOKING_DETAILS')."</h3>\n";
			$html .= '<div class="iosrt_box iosrt_pad">'."\n";
			$html .= '<table class="iosrt_thankstbl">'."\n";
			$txt = $this->formatDate($rtdata['pickdate'], $eLang->get('DATETIME_FORMAT_1')).'<br />'.$eLang->get('FROM').' '.$rtdata['pickarea'].', '.$rtdata['picklocation'];
			$html .= '<tr><th>'.$eLang->get('PICKUP').'</th><td>'.$txt.'</td></tr>'."\n";
			$txt = $this->formatDate($rtdata['dropdate'], $eLang->get('DATETIME_FORMAT_1')).'<br />'.$eLang->get('AT').' '.$rtdata['droparea'].', '.$rtdata['droplocation'];
			$html .= '<tr><th>'.$eLang->get('DROPOFF').'</th><td>'.$txt.'</td></tr>'."\n";
			$txt = ($rtdata['vtype'] == 'boat') ? $rtdata['vehiclemodel'] : $rtdata['vehiclemodel'].' '.$eLang->get('OR_SIMILAR');
			$txt .= '<br />'.$this->makeModelSubtitle($rtdata['vehiclegroup'], $rtdata['vtype'], $rtdata['vehiclecategories'], $categories);
			$title = ($rtdata['vtype'] == 'boat') ? $eLang->get('BOAT') : $eLang->get('VEHICLE');
			$html .= '<tr><th>'.$title.'</th><td>'.$txt.'</td></tr>'."\n";
			$html .= '<tr><th>'.$eLang->get('RESERVATION_CODE').'</th><td><strong>'.$rtdata['rescode'].'</strong></td></tr>'."\n";
			$amount_txt = '<strong>'.number_format($rtdata['totalprice'], 2, $eLang->get('DECIMALS_SEP'), $eLang->get('THOUSANDS_SEP')).' '.$rtdata['currency'].'</strong>';
			if ($rtdata['usercurrency'] != '') {
				$v = $rtdata['totalprice'] * $rtdata['usercurrencyrate'];
				$amount_txt .= ' <span dir="ltr">('.number_format($v, 2, $eLang->get('DECIMALS_SEP'), $eLang->get('THOUSANDS_SEP')).' '.$rtdata['usercurrency'].')</span>';
			}
			$html .= '<tr><th>'.$eLang->get('TOTAL').'</th><td>'.$amount_txt.'</td></tr>'."\n";
			if ($rtdata['feeamount'] > 0) {
				$amount_txt = number_format($rtdata['feeamount'], 2, $eLang->get('DECIMALS_SEP'), $eLang->get('THOUSANDS_SEP')).' '.$rtdata['currency'];
				if ($rtdata['usercurrency'] != '') {
					$v = $rtdata['feeamount'] * $rtdata['usercurrencyrate'];
					$amount_txt .= ' <span dir="ltr">('.number_format($v, 2, $eLang->get('DECIMALS_SEP'), $eLang->get('THOUSANDS_SEP')).' '.$rtdata['usercurrency'].')</span>';
				}
				$html .= '<tr><th>'.$eLang->get('SURETY_FEE').'</th><td>'.$amount_txt.'</td></tr>'."\n";
			}
			if (isset($rtdata['paymoduledata']['title'])) {//display it even with 0 feeamount (eg for cash on delivery)
				if ($rtdata['paymoduledata']['title'] != '') {
					$html .= '<tr><th>'.$eLang->get('PAYMENT_METHOD').'</th><td>'.$rtdata['paymoduledata']['title'].'</td></tr>'."\n";
				}
			}
			$html .= "</table>\n";
			$html .= "</div>\n";

			if ($page == 3) {
				if ($rtdata['paymoduledata']) {
					if ($rtdata['paymoduledata']['page'] == 0) {//offline payment
						if ($rtdata['paymoduledata']['listitems']) {
							$html .= '<h3 class="iosrt_side_h3">'.$eLang->get('PAYMENT_INFO')."</h3>\n";
							if ($rtdata['paymoduledata']['listtop'] != '') {
								$html .= '<p class="iosrt_p">'.$rtdata['paymoduledata']['listtop']."</p>\n";
							}
							$html .= '<div class="iosrt_box iosrt_pad">'."\n";
							$html .= '<table class="iosrt_thankstbl">'."\n";
							foreach ($rtdata['paymoduledata']['listitems'] as $k => $v) {
								$html .= '<tr><th>'.$v['ititle'].'</th><td>'.$v['itext'].'</td></tr>'."\n";
							}
							$html .= "</table>\n";
							$html .= "</div>\n";
							if ($rtdata['paymoduledata']['listbottom'] != '') {
								$html .= '<p class="iosrt_p">'.$rtdata['paymoduledata']['listbottom']."</p>\n";
							}
						}
					}
				}
			}

			if ($page == 5) {//payment failure
				if ($rtdata['paymoduledata']) {
					if ($rtdata['paymoduledata']['page'] == 1) {//online payment
						if (($rtdata['paymoduledata']['formaction'] != '') && ($rtdata['paymoduledata']['formitems'])) {
							if ($rtdata['paymod'] != 'alphacom') {//Alpha e-Commerce will generate an error due to dublicated orderid
								$html .= '<form name="fmppapi" id="iosrt_fmppapi2" action="'.$rtdata['paymoduledata']['formaction'].'" method="post">'."\n";
								foreach ($rtdata['paymoduledata']['formitems'] as $k => $v) {
									$html .= '<input type="hidden" name="'.$v['iname'].'" value="'.$v['ivalue'].'" />'."\n";
								}
								$html .= '<div class="iosrt_vpad iosrt_center"><button type="submit" name="ppsub" class="iosrt_bookbtn">'.$eLang->get('RETRY_PAYMENT')."</button></div>\n";
								$html .= "</form>\n";
							}
						}
					}
				}
			}
		}

		if (($page == 3) || ($page == 4)) {
			$html .= '<br />';
			$html .= '<p class="iosrt_thanks">'.$eLang->get('THANKS_CHOOSE')."</p>\n";
			$html .= '<p class="iosrt_print"><a href="javascript:window.print();">'.$eLang->get('PRINT_PAGE')."</a></p>\n";
		}

		$html .= '<div class="iosrt_combox">'."\n";
		$html .= '<h4 class="iosrt_combox_h4">'.$company->title."</h4>\n";
		if ($company->address != '') {
			$address = $company->address;
			if ($company->postalcode != '') { $address .= ', '.$company->postalcode; }
			if ($company->city != '') { $address .= ', '.$company->city; }
			if ($company->country != '') { $address .= ', '.$company->country; }
			$html .= '<div class="iosrt_combox_line">'.$address."</div>\n";
		}

		$parts = array();
		if ($company->phone != '') { $parts[] = $company->phone; }
		if ($company->mobile != '') { $parts[] = $company->mobile; }
		if ($company->email != '') {
			$arr = str_split ($company->email);
			$enc_mailto = '&#109;&#97;&#105;&#108;&#116;&#111;&#58;';
			$enc_mail = '';
			foreach ($arr as $a) { $enc_mail .= '&#'.ord($a).';'; }
			$parts[] = '<a href="'.$enc_mailto.$enc_mail.'">'.$enc_mail.'</a>';
		}
		if ($parts) { $html .= '<div class="iosrt_combox_line">'.implode(', ', $parts).'</div>'; }
		$html .= "</div>\n";//iosrt_combox
		$html .= "</div>\n";//iosrt_pgwrap

		return $html;
	}


	/*********************************************/
	/* STEP 6 :: PAY SUCCESS FOR ALPHA BANK PAGE */
	/*********************************************/
	private function finalPageAlpha() {
		$rescode = $this->getFromRequest('rescode', '');

		$alf_data = array();
		$alf_data['mid'] = isset($_POST['mid']) ? trim($_POST['mid']) : '';
		$alf_data['orderid'] = isset($_POST['orderid']) ? trim($_POST['orderid']) : '';
		$alf_data['status'] = isset($_POST['status']) ? $_POST['status'] : '';
		$alf_data['orderAmount'] = isset($_POST['orderAmount']) ? $_POST['orderAmount'] : '';
		$alf_data['currency'] = isset($_POST['currency']) ? $_POST['currency'] : '';
		$alf_data['paymentTotal'] = isset($_POST['paymentTotal']) ? $_POST['paymentTotal'] : '';
		$alf_data['message'] = isset($_POST['message']) ? $_POST['message'] : '';
		$alf_data['riskScore'] = isset($_POST['riskScore']) ? $_POST['riskScore'] : '';
		$alf_data['payMethod'] = isset($_POST['payMethod']) ? $_POST['payMethod'] : '';
		$alf_data['txId'] = isset($_POST['txId']) ? $_POST['txId'] : '';
		$alf_data['paymentRef'] = isset($_POST['paymentRef']) ? $_POST['paymentRef'] : '';
		$alf_data['digest'] = isset($_POST['digest']) ? $_POST['digest'] : '';

		if (($rescode != '') && ($alf_data['orderid'] != '') || ($alf_data['mid'] != '') || ($alf_data['digest'] != '')) {
			$result = $this->helper->payNotifyRequest($alf_data, 'alphacom');//-1: connection problems, 0: notification failed/rejected by IOSRT, 1: notification success on IOSRT
		}

		$html = $this->finalPage(4);
		return $html;
	}


	/****************************************************/
	/* STEP 7 :: PAY SUCCESS FOR PIRAEUS BANK EPOS PAGE */
	/****************************************************/
	private function finalPageePOS() {
		$html = $this->finalPageGenericSuccess('epos');
		return $html;
	}



	/*************************************/
	/* STEP 8 :: PAY SUCCESS FOR PAYUBIZ */
	/*************************************/
	private function finalPagePayUBiz() {
		$html = $this->finalPageGenericSuccess('payubiz');
		return $html;
	}


	/**********************************************/
	/* STEP 9 :: PAY SUCCESS PAGE FOR VIVA WALLET */
	/**********************************************/
	private function finalPageVivaWallet() {
		$rescode = $this->getFromRequest('rescode', '');

		$viva_data = array();
		$viva_data['s'] = isset($_GET['s']) ? trim($_GET['s']) : '';
		$viva_data['t'] = isset($_GET['t']) ? trim($_GET['t']) : '';
		if ($viva_data['s'] == '') {
			$viva_data['s'] = isset($_POST['s']) ? trim($_POST['s']) : '';
		}
		if ($viva_data['t'] == '') {
			$viva_data['t'] = isset($_POST['t']) ? trim($_POST['t']) : '';
		}

		if (($viva_data['s'] != '') || ($viva_data['t'] != '')) {
			$result = $this->helper->payNotifyRequest($viva_data, 'vivawallet');//-1: connection problems, 0: notification failed/rejected by IOSRT, 1: notification success on IOSRT
		}

		$html = $this->finalPage(4);
		return $html;
	}


	/******************************************************/
	/* GENERIC PAY SUCCESS PAGE WITH PAYMENT NOTIFICATION */
	/******************************************************/
	private function finalPageGenericSuccess($iosrpaymethod) {
		$rescode = $this->getFromRequest('rescode', '');

		$post_data = array();
		if (isset($_POST)) {
			if (count($_POST) > 0) {
				foreach ($_POST as $k => $v) {
					$post_data[$k] = htmlentities($v, ENT_QUOTES);
				}
			}
		}
		if (($rescode != '') && !isset($post_data['rescode'])) { $post_data['rescode'] = $rescode; }

		if (count($post_data) > 0) {
		//if (($rescode != '') && (count($post_data) > 1)) {
			$result = $this->helper->payNotifyRequest($post_data, $iosrpaymethod);//-1: connection problems, 0: notification failed/rejected by IOSRT, 1: notification success on IOSRT
			if ($result == -1) { //-1: connection problems => thanks page, success
				$html = $this->finalPage(3);
			} else if ($result == 0) { //0: notification failed/rejected by IOSRT => thanks page, payment failure
				$html = $this->finalPage(5);
			} else if ($result == 1) { //1: notification success on IOSRT => //thanks page, payment success
				$html = $this->finalPage(4);
			} else {//just in case => thanks page, success
				$html = $this->finalPage(3);
			}
		} else {//thanks page, success
			$html = $this->finalPage(3);
		}
		return $html;
	}


	/************************************************/
	/* BOOK FORM: SELECT TO PAY REST AMOUNT IN CASH */
	/************************************************/
	private function cashPaymentSelection($response, $price_data, $eLang) {
		if ($response->cashpc < 1) { return ''; }
		if ($price_data['use_fullamount'] === true) { return ''; }
		if ($price_data['feeamount'] >= $price_data['totalprice']) { return ''; }

		$txt = sprintf($eLang->get('PAYCASH_GETDISCOUNT'), '<strong>'.$response->cashpc.'%</strong>');
		$html = '<div class="iosrt_vpad" id="iosrt_cashpay_wrap">'."\n";
		$html .= '<label class="iosrt_checkwrap">'.$txt;
		$html .= '<input type="checkbox" name="cash" id="iosrt_cash" value="'.$response->cashpc.'" onchange="iosRTToggleCash();"><span class="iosrt_ccheckbox"></span></label>'."\n";
		$html .= "</div>\n";
		return $html;
	}


	/***********************************/
	/* PRICE ANALYSIS FOR BOOKING FORM */
	/***********************************/
	private function priceAnalysis($response, $price_data, $company, $currency, $eLang) {
		$cursymbol = $currency->currencySymbol();

		$html = '<h3 class="iosrt_side_h3">'.$eLang->get('PRICE_ANALYSIS')."</h3>\n";
		$html .= '<table class="iosrt_sumtable">';
		$html .= '<tr><th>'.$eLang->get('RENTAL_FEE').'</th><td><span id="iosrt_analysis_rentalfee">'.number_format($response->rentalfee, 2, '.', '').'</span> '.$cursymbol.'</td></tr>';

		if ($response->extrahoursfee > 0) {
			$tooltip_str = '';
			if ($response->charge_hours > 0) {
				$free_hours = $response->hours - $response->charge_hours;
				$title = sprintf($eLang->get('EXTRA_HOURS_DESC'), $response->charge_hours);
				if ($free_hours == 1) {
					$title .= ' '.$eLang->get('FIRST_HOUR_FREE');
				} else if ($free_hours > 1) {
					$title .= ' '.sprintf($eLang->get('FIRST_HOURS_FREE'), $free_hours);
				}
				$tooltip_str = ' title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo"';
			}
			$html .= '<tr><th'.$tooltip_str.'>'.$eLang->get('EXTRA_HOURS').'</th><td><span id="iosrt_analysis_extrahoursfee">'.number_format($response->extrahoursfee, 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
		}
		if ($response->pickupfee > 0) {
			$txt = addslashes($response->pickarea.', '.$response->picklocation);
			$title = sprintf($eLang->get('VEH_PICKUP_FROM'), $txt);
			$html .= '<tr><th title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('PICK_UP').'</th><td><span id="iosrt_analysis_pickupfee">'.number_format($response->pickupfee, 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
		}
		if ($response->dropofffee > 0) {
			$txt = addslashes($response->droparea.', '.$response->droplocation);
			$title = sprintf($eLang->get('VEH_DROPOFF_TO'), $txt);
			$html .= '<tr><th title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('DROP_OFF').'</th><td><span id="iosrt_analysis_dropofffee">'.number_format($response->dropofffee, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if ($response->afterhourspick > 0) {
			$html .= '<tr><th title="'.$eLang->get('AFTER_WORKHOURS_PICKUP').'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('PICKUP_AFTERH').'</th><td><span id="iosrt_analysis_afterhourspick">'.number_format($response->afterhourspick, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if ($response->afterhoursdrop > 0) {
			$html .= '<tr><th title="'.$eLang->get('AFTER_WORKHOURS_DROPOFF').'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('DROPOFF_AFTERH').'</th><td><span id="iosrt_analysis_afterhoursdrop">'.number_format($response->afterhoursdrop, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}

		if ($response->extras) {
			$html .= '<tr><th>'.$eLang->get('EXTRAS').'</th><td><span id="iosrt_analysis_extrasprice">'.number_format($response->preselextras, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if ($this->options['showcoupon'] == 1) {
			$html .= '<tr><th>'.$eLang->get('COUPON').'</th><td><span id="iosrt_analysis_couponamount">0.00</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if ($response->earlybookpc > 0) {
			$earlybook = ($response->earlybookpc > 0) ? ($response->rentalfee * $response->earlybookpc) / 100 : 0.00;
			$title = sprintf($eLang->get('EARLYDISC_PC_DAYS_PICKUP'), $response->earlybookpc.'%', $response->earlybookdays);
			$html .= '<tr><th title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('EARLY_BOOKING').'</th><td><span id="iosrt_analysis_earlybook">-'.number_format($earlybook, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if ($price_data['use_fullamount']) {
			$fullamount = ($response->rentalfee * $response->fullamountpc) / 100;
			$title = sprintf($eLang->get('FULLAMOUNT_DISC_PAYNOW'), $response->fullamountpc.'%');
			$html .= '<tr><th title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('100_PREPAYMENT').'</th><td><span id="iosrt_analysis_fullamount">-'.number_format($fullamount, 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
		}
		if (($response->cashpc > 0) && ($price_data['use_fullamount'] == false)) {
			if ($price_data['feeamount'] < $price_data['totalprice']) {
				$title = sprintf($eLang->get('PAYCASH_GETDISCOUNT'), $response->cashpc.'%');
				$html .= '<tr><th title="'.$title.'" class="iosrt_wtooltip iosrt_analysisinfo">'.$eLang->get('CASH_DISCOUNT').'</th><td><span id="iosrt_analysis_cash">0.00</span> '.$cursymbol.'</td></tr>'."\n";
			}
		}

		if ($company->tax > 0) {
			$html .= '<tr><th class="iosrt_important">'.$eLang->get('SUBTOTAL').'</th><td class="iosrt_important"><span id="iosrt_analysis_subtotal">'.number_format($price_data['subtotal'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
			$txt = ($company->taxispc == 1) ? ' <span class="iosrt_fontnormal">('.$company->tax.'%)</span>' : ' <span class="iosrt_fontnormal">('.$company->tax.' '.$cursymbol.')</span>';
			$html .= '<tr><th>'.$eLang->get('TAX').$txt.'</th><td><span id="iosrt_analysis_tax">'.number_format($price_data['tax'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
			$html .= '<tr><th class="iosrt_vimportant">'.$eLang->get('TOTAL').'</th><td class="iosrt_vimportant"><span id="iosrt_analysis_totalprice">'.number_format($price_data['totalprice'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
			$html .= '<tr><th class="iosrt_cellsfee">'.$eLang->get('SURETY_FEE').$price_data['feeamount_note'].'</th><td class="iosrt_cellsfee"><span id="iosrt_analysis_feeamount">'.number_format($price_data['feeamount'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
			$html .= "</table>\n";
		} else {
			$html .= '<tr><th class="iosrt_vimportant">'.$eLang->get('TOTAL').'</th><td class="iosrt_vimportant"><span id="iosrt_analysis_totalprice">'.number_format($price_data['totalprice'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>';
			$html .= '<tr><th class="iosrt_cellsfee">'.$eLang->get('SURETY_FEE').$price_data['feeamount_note'].'</th><td class="iosrt_cellsfee"><span id="iosrt_analysis_feeamount">'.number_format($price_data['feeamount'], 2, '.', '').'</span> '.$cursymbol.'</td></tr>'."\n";
			$html .= "</table>\n";
			$html .= '<div class="iosrt_princtax">'.$eLang->get('PRICES_INC_TAXES')."</div>\n";
		}

		return $html;
	}


	/*************************************/
	/* PERSONAL DETAILS FOR BOOKING FORM */
	/*************************************/
	private function personalDetails($response, $nextblock, $eLang, $has_sidebox) {
		$bookelements = $response->bookelements;
		$vtype = $response->vehicle['vtype'];

		$buffer = array();
		foreach ($bookelements as $idx => $element) {
			switch ($idx) {
				case 'ocomments': break;//display later
				case 'ofirstname':
					$buffer[] = $this->helper->addText($idx, '', $eLang->get('FIRSTNAME').'*', array('required' => 'required', 'placeholder' => $eLang->get('FIRSTNAME'), 'labelclass' => 'iosrt_labelblock'), false);
				break;
				case 'olastname':
					$buffer[] = $this->helper->addText($idx, '', $eLang->get('LASTNAME').'*', array('required' => 'required', 'placeholder' => $eLang->get('LASTNAME'), 'labelclass' => 'iosrt_labelblock'), false);
				break;
				case 'oemail':
					$buffer[] = $this->helper->addText($idx, '', $eLang->get('EMAIL').'*', array('type' => 'email', 'required' => 'required', 'placeholder' => $eLang->get('EMAIL'), 'labelclass' => 'iosrt_labelblock'), false);
					$buffer[] = $this->helper->addText('oemailconf', '', $eLang->get('EMAIL_CONFIRMATION').'*', array('type' => 'email', 'required' => 'required', 'placeholder' => $eLang->get('EMAIL'), 'labelclass' => 'iosrt_labelblock'), false);
				break;
				case 'licnumber': case 'ocity': case 'oaddress': case 'opostalcode': case 'ophone': case 'omobile':
				case 'flight': case 'ship': case 'hotel': case 'driverage':
					$data = $this->makeTextElementData($idx, $element, $eLang, $vtype);
					$buffer[] = $this->helper->addText($idx, '', $data['title'], $data['attrs'], false);
				break;
				case 'ocountry':
					if (file_exists($this->path.'/language/countries/countries.'.$this->lang->guilang.'.php')) {
						include($this->path.'/language/countries/countries.'.$this->lang->guilang.'.php');
					} else {
						include($this->path.'/language/countries/countries.en.php');
					}
					switch($this->lang->guilang) {
						case 'el': $curcountry = 'GR'; break;
						case 'it': $curcountry = 'IT'; break;
						case 'de': $curcountry = 'DE'; break;
						case 'fr': $curcountry = 'FR'; break;
						case 'es': $curcountry = 'ES'; break;
						case 'ru': $curcountry = 'RU'; break;
						case 'zt': case 'zh': $curcountry = 'CN'; break;
						case 'pt': $curcountry = 'PT'; break;
						case 'en': default: $curcountry = 'GB'; break;
					}
					$curcountry = $this->detectVisitorCountry($countries, $curcountry);
					$options = array();
					foreach ($countries as $k => $v) { $options[] = array($k, $v); }
					$buffer[] = $this->helper->addSelect('ocountry', $eLang->get('COUNTRY'), $curcountry, $options,  array('labelclass' => 'iosrt_labelblock'), false);
					unset($countries, $curcountry, $options);
				break;
				case 'custom1': case 'custom2': case 'custom3': case 'custom4': case 'custom5': case 'custom6': case 'custom7': case 'custom8': 
					$elementname = 'c'.$idx;
					$attrs = array();
					if ($bookelements[$idx]['type'] == 'text') {
						$attrs['placeholder'] = $bookelements[$idx]['title'];
						$attrs['labelclass'] = 'iosrt_labelblock';
						if ($bookelements[$idx]['show'] == 2) {
							$label = $bookelements[$idx]['title'].'*';
							$attrs['required'] = 'required';
						} else {
							$label = $bookelements[$idx]['title'];
						}
						if ($bookelements[$idx]['tip'] != '') { $attrs['tip'] = $bookelements[$idx]['tip']; }
						$buffer[] = $this->helper->addText($elementname, '', $label, $attrs, false);
					} else if ($bookelements[$idx]['type'] == 'select') {
						if ($bookelements[$idx]['options'] != '') {
							$parts = explode('|', $bookelements[$idx]['options']);
							$v = $parts[0];
							$options = array();
							if ($bookelements[$idx]['show'] == 1) { $v = ''; $options[] = array('', '- '.$eLang->get('SELECT').' -'); }
							foreach ($parts as $part) { $options[] = array($part, $part); }
							if ($bookelements[$idx]['show'] == 2) {
								$label = $bookelements[$idx]['title'].'*';
							} else {
								$label = $bookelements[$idx]['title'];
							}
							if ($bookelements[$idx]['tip'] != '') { $attrs['tip'] = $bookelements[$idx]['tip']; }
							$attrs['labelclass'] = 'iosrt_labelblock';
							$buffer[] = $this->helper->addSelect($elementname, $label, $v, $options, $attrs, false);
						}
					} else if ($bookelements[$idx]['type'] == 'yesno') {
						$tip = '';
						if ($bookelements[$idx]['tip'] != '') { $tip = $bookelements[$idx]['tip']; }
						$yesnotxt = '<label class="iosrt_labelblock" for="iosrt_'.$elementname.'">'.$bookelements[$idx]['title']."</label>\n";
						$yesnotxt .= '<div class="iosrt_zero"><label class="iosrt_switch">';
						$yesnotxt .= '<input type="checkbox" name="'.$elementname.'" id="iosrt_'.$elementname.'" class="iosrt_switchinput" value="1" />';
						$yesnotxt .= '<span class="iosrt_switchlabel" data-on="'.$eLang->get('YES').'" data-off="'.$eLang->get('NO').'"></span>';
						$yesnotxt .= '<span class="iosrt_switchhandle"></span></label>'."\n";
						if ($tip != '') { $yesnotxt .= '<div class="iosrt_tip">'.$tip."</div>\n"; }
						$yesnotxt .= "</div>\n";
						$buffer[] = $yesnotxt;
						unset($yesnotxt);
					}
				break;
				default: break;
			}
		}

		$html = '<h2 class="iosrt_block_h2"><span class="circle_block">'.$nextblock.'</span> '.$eLang->get('PERSONAL_DETAILS')."</h2>\n";
		if ($has_sidebox) {
			$html .= '<div class="iosrt_box iosrt_boxbookfix iosrt_pad">'."\n";
		} else {
			$html .= '<div class="iosrt_box iosrt_pad">'."\n";
		}

		$nextfirst = 0;
		foreach ($buffer as $k => $v) {
			if ($k < $nextfirst) { continue; }
			$second = $k + 1;
			if (isset($buffer[$second])) {
				$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
				$html .= $v;
				$html .= '</div><div class="iosrt_persright">'."\n";
				$html .= $buffer[$second];
				$html .= "</div>\n</div>\n";
			} else {
				$html .= '<div class="iosrt_perswrap">'."\n";
				$html .= $v;
				$html .= "</div>\n";
			}
			$nextfirst = $k + 2;
		}
		unset($buffer);

		if (isset($bookelements['ocomments']) && ($bookelements['ocomments']['show'] <> 0) && ($bookelements['ocomments']['show'] <> 3)) {
			$html .= '<div class="iosrt_perswrap">'."\n";
			$html .= '<label class="iosrt_labelblock" for="iosrt_comments">'.$eLang->get('COMMENTS')."</label>\n";
			$html .= '<textarea name="ocomments" id="iosrt_comments" class="iosrt_textarea" placeholder="'.$eLang->get('COMMENTS').'"></textarea>'."\n";
			$html .= '<div class="iosrt_tip">'.$eLang->get('COMMENTS_TIP')."</div>\n";
			$html .= "</div>\n";
		}

		if ($this->options['showinvoice'] == 1) {
			$html .= '<div class="iosrt_perswrap">'."\n";
			$html .= '<label class="iosrt_checkwrap">'.$eLang->get('ISSUE_INVOICE_TIP');
			$html .= '<input type="checkbox" name="invoice" id="iosrt_invoice" value="1" onchange="iosRTToggleInvoice();"><span class="iosrt_ccheckbox"></span></label>'."\n";
			$html .= "</div>\n";
		}
		$html .= '<div class="iosrt_invisible" id="iosrt_invoice_wrap">';
		$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
		$html .= $this->helper->addText('invtitle', '', $eLang->get('COMPANY_TITLE'), array('placeholder' => $eLang->get('COMPANY_TITLE'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= '</div><div class="iosrt_persright">'."\n";
		$html .= $this->helper->addText('invact', '', $eLang->get('COMPANY_ACTIV'), array('placeholder' => $eLang->get('COMPANY_ACTIV'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= "</div>\n</div>\n";
		$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
		$html .= $this->helper->addText('invaddress', '', $eLang->get('ADDRESS'), array('placeholder' => $eLang->get('ADDRESS'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= '</div><div class="iosrt_persright">'."\n";
		$html .= $this->helper->addText('invcity', '', $eLang->get('CITY'), array('placeholder' => $eLang->get('CITY'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= "</div>\n</div>\n";
		$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
		$html .= $this->helper->addText('invirs', '', $eLang->get('IRS'), array('placeholder' => $eLang->get('IRS'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= '</div><div class="iosrt_persright">'."\n";
		$html .= $this->helper->addText('invafm', '', $eLang->get('AFM'), array('placeholder' => $eLang->get('AFM'), 'labelclass' => 'iosrt_labelblock'), false);
		$html .= "</div>\n</div>\n";
		$html .= '<div class="iosrt_tip">'.$eLang->get('INVOICE_FIELDS_NOTE')."</div>\n";
		$html .= "</div>\n";

		$html .= "</div>\n";//iosrt_box

		return $html;
	}


	/*********************************************/
	/* MAKE HTML FOR PERSONAL DATA FORM ELEMENTS */
	/*********************************************/
	private function makeTextElementData($idx, $element, $eLang, $vtype='') {
		$data = array(
			'title' => $element['title'],
			'attrs' => array(
				'type' => 'text',
				'placeholder' => $element['title'],
				'labelclass' => 'iosrt_labelblock'
			)
		);

		switch ($idx) {
			case 'licnumber': $data['title'] = $eLang->get('LICENSE_NUMBER'); break;
			case 'ocity': $data['title'] = $eLang->get('CITY'); break;
			case 'oaddress': $data['title'] = $eLang->get('ADDRESS'); break;
			case 'opostalcode': $data['title'] = $eLang->get('POSTALCODE'); break;
			case 'ophone': $data['title'] = $eLang->get('TELEPHONE'); $data['attrs']['dir'] = 'ltr'; break;
			case 'omobile': $data['title'] = $eLang->get('MOBILE'); $data['attrs']['dir'] = 'ltr'; break;
			case 'flight': $data['title'] = $eLang->get('FLIGHT_NUMBER'); $data['attrs']['dir'] = 'ltr'; break;
			case 'ship': $data['title'] = $eLang->get('SHIP_NAME'); $data['attrs']['dir'] = 'ltr'; break;
			case 'hotel': $data['title'] = $eLang->get('HOTEL'); $data['attrs']['dir'] = 'ltr'; break;
			case 'driverage':
				$agetxt = ($vtype == 'boat') ? $eLang->get('AGE') : $eLang->get('DRIVER_AGE');
				$data['title'] = $agetxt; $data['attrs']['type'] = 'number'; $data['attrs']['dir'] = 'ltr'; $data['attrs']['min'] = '0'; $data['attrs']['max'] = '100'; $data['attrs']['step'] = '1';
			break;
			default: break;
		}
		$data['placeholder'] = $data['title'];

		if ($element['show'] == 2) {
			$data['title'] .= '*';
			$data['attrs']['required'] = 'required';
		}

		return $data;
	}


	/************************************/
	/* PAYMENT OPTIONS FOR BOOKING FORM */
	/************************************/
	private function paymentOptions($response, $nextblock, $eLang, $price_data) {
		if (!$response->paymods) { return ''; }//just in case

		$all_pids = array();
		$html = '<h2 class="iosrt_block_h2"><span class="circle_block">'.$nextblock.'</span> '.$eLang->get('PAYMENT_OPTIONS')."</h2>\n";
		$html .= '<div class="iosrt_box iosrt_pad">'."\n";
		$html .= '<table class="iosrt_paymodstbl">'."\n";
		$i = 0;
		foreach ($response->paymods as $pid => $paymod) {
			if (($paymod['name'] == 'cashondelivery') && ($price_data['use_fullamount'] == true)) { continue; }//do not use "cashondelivery" with fullamount discount
			$has_cc = false;
			if ($paymod['name'] == 'creditcardoff') {
				if (!isset($paymod['creditcards'])) { continue; }
				if (trim($paymod['creditcards']) == '') { continue; } else { $has_cc = true; }
			} else if ($paymod['name'] == 'cashondelivery') {
				if (isset($paymod['creditcards'])) {
					if (trim($paymod['creditcards']) != '') { $has_cc = true; }
				}
			}

			$all_pids[] = $pid;
			if ($i == 0) {
				$checked = ' checked="checked"';
				$row_class = ' class="iosrt_paymodsel"';
			} else {
				$checked = '';
				$row_class = '';
			}

			$html .= '<tr id="iosrt_paymodrow'.$pid.'"'.$row_class.'><td class="iosrt_paymods_pid"><label class="iosrt_radiowrap"><span class="iosrt_labeltext">'.$paymod['title'].'</span>';
			$html .= '<input type="radio" name="pid" id="iosrt_pid'.$pid.'" value="'.$pid.'"'.$checked.' onclick="iosRTCheckPaymods();" data-module="'.$paymod['name'].'" /><span class="iosrt_cradio"></span></label></td>'."\n";
			$html .= '<td><a href="javascript:void(null);" onclick="iosRTPickPaymod('.$pid.');" class="iosrt_paymodlink"><strong>'.$paymod['title'].'</strong>';
			if ($paymod['description'] != '') { $html .= '<div class="iosrt_paymods_desc">'.$paymod['description'].'</div>'; }
			$html .= '</a>';

			if ($has_cc) {
				$show = ($i == 0) ? true : false;
				$html .= $this->makeCCOffineHTML($paymod['creditcards'], $eLang, $pid, $show);
			}
			$html .= "</td>\n</tr>\n";
			$i++;
		}

		if (!$all_pids) { return ''; }//maybe was only "cashondelivery" and disabled due to fullamount discount

		$html .= "</table>\n";
		$html .= "</div>\n";//iosrt_box
		$txt = ($all_pids) ? implode(',', $all_pids) : '';
		$html .= '<input type="hidden" name="pids" id="iosrt_allpids" dir="ltr" value="'.$txt.'" />'."\n";

		return $html;
	}


	/************************************************/
	/* CREDIT CARD OFFLINE DETAILS FOR BOOKING FORM */
	/************************************************/
	private function makeCCOffineHTML($creditcards, $eLang, $pid, $show) {
		$class = ($show == true) ? 'iosrt_vpad' : 'iosrt_invisible';

		$html = '<div id="iosrt_paymoddetails_box'.$pid.'" class="'.$class.'">'."\n";
		
		$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
		$html .= '<label class="iosrt_labelblock" for="iosrt_ccowner'.$pid.'">'.$eLang->get('CC_OWNER')."</label>\n";
		$html .= '<input type="text" name="ccowner'.$pid.'" id="iosrt_ccowner'.$pid.'" class="iosrt_text" maxlength="80" value="" placeholder="'.$eLang->get('CC_OWNER').'" />'."\n";
		$html .= '</div><div class="iosrt_persright">'."\n";
		$html .= '<label class="iosrt_labelblock" for="iosrt_cctype'.$pid.'">'.$eLang->get('CC_TYPE')."</label>\n";
		$html .= '<select name="cctype'.$pid.'" id="iosrt_cctype'.$pid.'" dir="ltr" class="iosrt_select">'."\n";

		$ccards = explode(',', $creditcards);
		$i = 0;
		foreach ($ccards as $ccard) {
			switch ($ccard) {
				case 'visa': $name = 'Visa'; break;
				case 'master': $name = 'MasterCard'; break;
				case 'amex': $name = 'American Express'; break;
				case 'discover': $name = 'Discover'; break;
				case 'diners': $name = 'Diners Club'; break;
				case 'jcb': $name = 'JCB'; break;
				case 'austr': $name = 'Australian BankCard'; break;
				case 'maestro': $name = 'Maestro'; break;
				case 'laser': $name = 'Laser'; break;
				case 'enroute': $name = 'EnRoute'; break;
				case 'switch': $name = 'Switch'; break;
				default: $name = ucfirst($ccard); break;
			}
			$sel = ($i == 0) ? ' selected="selected"' : '';
			$html .= '<option value="'.$ccard.'"'.$sel.'>'.$name."</option>\n";
			$i++;
		}
		$html .= '</select>'."\n";
		$html .= "</div>\n</div>\n";//iosrt_perswrap

		$html .= '<div class="iosrt_perswrap"><div class="iosrt_persleft">'."\n";
		$html .= '<label class="iosrt_labelblock" for="iosrt_ccnumber'.$pid.'">'.$eLang->get('CC_NUMBER')."</label>\n";
		$html .= '<input type="text" name="ccnumber'.$pid.'" id="iosrt_ccnumber'.$pid.'" maxlength="50" class="iosrt_text" value="" placeholder="'.$eLang->get('CC_NUMBER').'" />'."\n";
		$html .= '</div><div class="iosrt_persright">'."\n";
		$html .= '<label class="iosrt_labelblock" for="iosrt_ccseccode'.$pid.'">'.$eLang->get('CC_SECURITY_CODE')."</label>\n";
		$html .= '<input type="text" name="ccseccode'.$pid.'" id="iosrt_ccseccode'.$pid.'" maxlength="6" dir="ltr" class="iosrt_text" value="" placeholder="'.$eLang->get('CC_SECURITY_CODE').'" />'."\n";
		$html .= "</div>\n</div>\n";//iosrt_perswrap

		$html .= '<div class="iosrt_perswrap">'."\n";
		$html .= '<label class="iosrt_labelblock" for="iosrt_ccexpire'.$pid.'">'.$eLang->get('CC_EXPDATE')."</label>\n";
		$html .= '<select name="ccexpire'.$pid.'" id="iosrt_ccexpire'.$pid.'" dir="ltr" class="iosrt_select">'."\n";
		$y = gmdate('Y');
		$m = gmdate('m');
		for ($i = 0; $i < 16; $i++) {
			$cy = $y + $i;
			if ($i == 0) { $sm = $m; } else { $sm = 1; }
			for ($k = $sm; $k < 13; $k++) {
				$mo = sprintf("%02d", $k);
				$sel = '';
				if ($i == 0) {
					if ($k == $sm) { $sel = ' selected="selected"'; }
				}
				$html .= '<option value="'.$mo.'-'.$cy.'"'.$sel.'>'.$mo.' / '.$cy."</option>\n";
			}
		}
		$html .= '</select>'."\n";
		$html .= "</div>\n";//iosrt_perswrap
		$html .= "</div>\n";//iosrt_paymoddetails_box

		return $html;
	}


	/***********************************/
	/* MAKE EXTRAS HTML (BOOKING FORM) */
	/***********************************/
	private function makeExtrasHTML($extras, $extrasgroups, $pickup, $nextblock, $eLang, $currency, $company, $modelparams, $charge_days) {
		if (!$extras) { return ''; }//just in case

		$html = '<h2 class="iosrt_block_h2"><span class="circle_block">'.$nextblock.'</span> '.$eLang->get('EXTRAS')."</h2>\n";
		$html .= '<div class="iosrt_box iosrt_boxbookfix iosrt_pad">'."\n";
		$html .= '<table class="iosrt_extrastbl">'."\n";

		$pickupts = strtotime($pickup);
		$pickupyear = intval(substr($pickup, 0, 4));

		$all_xids = array();
		foreach ($extras as $xid => $extra) {
			$seasonaldata = array(
				'price' => $extra['price'],
				'priceperday' => $extra['priceperday'],
				'priceformula' => $extra['priceformula'],
				'freeitems' => $extra['freeitems'],
				'priceduration' => $extra['priceduration']
			);
			if ($extra['seasons'] != '') {
				$seasons = json_decode($extra['seasons'], true);
				if ($seasons) {
					foreach ($seasons as $season) {
						$p = explode('-', $season['from']);
						$start = mktime(0, 0, 0, intval($p[1]), intval($p[0]), $pickupyear);
						$p = explode('-', $season['to']);
						$end = mktime(23, 59, 59, intval($p[1]), intval($p[0]), $pickupyear);
						if (($start <= $pickupts) &&  ($end >= $pickupts)) {
							$priceduration = isset($season['priceduration']) ? $season['priceduration'] : '';//IOS Rentals 1.8
							$seasonaldata = array(
								'price' => $season['price'],
								'priceperday' => $season['priceperday'],
								'priceformula' => $season['priceformula'],
								'freeitems' => $season['freeitems'],
								'priceduration' => $priceduration
							);
							break;
						}
					}
				}
			}

			$all_xids[] = $xid;
			$price_note = '';
			$x_init_price = 0.00;

			if ($seasonaldata['priceformula'] == '') {
				if (trim($seasonaldata['priceduration']) != '') {//price based on rental duration, IOS Rentals 1.8
					$last = 0.00;
					$found = false;
					$parts = explode(',', $seasonaldata['priceduration']);
					foreach ($parts as $part) {
						$daysprice = explode(':', $part);
						$d = (int)$daysprice[0];
						$last = $daysprice[1];
						if ($d >= $charge_days) {
							$seasonaldata['price'] = $daysprice[1];
							$found = true;
							break;
						}
					}
					if (!$found) { $seasonaldata['price'] = $last; }
				}

				$parts = array();
				$fp = $currency->shortprice($seasonaldata['price']);
				if ($seasonaldata['priceperday'] == 1) {
					$parts[] = sprintf($eLang->get('PRICE_PER_DAY'), $fp);
				} else {
					$parts[] = sprintf($eLang->get('PRICE_PER_RESERVATION'), $fp);
				}
				if ($seasonaldata['freeitems'] == 1) {
					$parts[] = $eLang->get('FIRST_FREE');
				} else if ($seasonaldata['freeitems'] > 1) {
					$parts[] = sprintf($eLang->get('FIRST_NUM_FREE'), $seasonaldata['freeitems']);
				}
				$price_note = implode(', ', $parts);
				if ($extra['preselitems'] > 0) {
					if ($seasonaldata['priceperday'] == 1) {
						$x_init_price = $seasonaldata['price'] * $charge_days * ($extra['preselitems'] - $seasonaldata['freeitems']);
					} else {
						$x_init_price = $seasonaldata['price'] * ($extra['preselitems'] - $seasonaldata['freeitems']);
					}
					if ($x_init_price < 0) { $x_init_price = 0; }
					if (($extra['maxprice'] > 0) && ($x_init_price > $extra['maxprice'])) { $x_init_price = $extra['maxprice']; }
				}
			} else if ($seasonaldata['priceformula'] == 'FULLEMPTY') {
				if ((trim($company->fuelprices) != '') && (trim($modelparams) != '')) {
					$fuelprices = json_decode(stripslashes($company->fuelprices), true);
					$model_fueldata = array('tank' => 0, 'fuel' => 0);
					$pairs = explode(';', $modelparams);
					foreach ($pairs as $pair) {
						$parts = explode(':', $pair);
						if (count($parts) == 2) {
							$name = $parts[0];
							if ($name == 'tank') { $model_fueldata['tank'] = (int)$parts[1]; } else if ($name == 'fuel') { $model_fueldata['fuel'] = (int)$parts[1]; }
						}
					}
					if (($model_fueldata['tank'] > 0) && ($model_fueldata['fuel'] > 0)) {
						switch ($model_fueldata['fuel']) {
							case 1: case 3: $fprice = isset($fuelprices['gasoline']) ? $fuelprices['gasoline'] : 0; break;
							case 2: $fprice = isset($fuelprices['diesel']) ? $fuelprices['diesel'] : 0; break;
							case 4: $fprice = isset($fuelprices['gas']) ? $fuelprices['gas'] : 0; break;
							case 5: $fprice = isset($fuelprices['hydrogen']) ? $fuelprices['hydrogen'] : 0; break;
							case 6: $fprice = isset($fuelprices['electric']) ? $fuelprices['electric'] : 0; break;
							default: $fprice = 0; break;
						}
						$price = ($fprice * $model_fueldata['tank']) + $company->refuelcost;
						if (($extra['maxprice'] > 0) && ($price > $extra['maxprice'])) { $price = $extra['maxprice']; }
						if ($price > 0) {
							$fp = $currency->shortprice($price);
							$price_note = sprintf($eLang->get('PRICE_PER_RESERVATION'), $fp);
						}

						if ($extra['preselitems'] > 0) {
							$x_init_price = $price * ($extra['preselitems'] - $seasonaldata['freeitems']);
							if ($x_init_price < 0) { $x_init_price = 0; }
							if (($extra['maxprice'] > 0) && ($x_init_price > $extra['maxprice'])) { $x_init_price = $extra['maxprice']; }
						}
					}
				}
			}

			$jseasons = ($extra['seasons'] != '') ? htmlentities($extra['seasons'], ENT_QUOTES, 'UTF-8') : '';
			$dclass = ($extra['identifier'] != '') ? 'iosrt_extra iosrt_extra_i'.$extra['identifier'] : 'iosrt_extra';
			$xgid = (int)$extra['xgid'];
			$singleselect = 0;
			if ($xgid > 0) {
				if ($extrasgroups) {
					if (isset($extrasgroups[$xgid])) { $singleselect = (int)$extrasgroups[$xgid]['singleselect']; }
				}
			}

			if ($extra['required'] == 1) {
				$html .= '<tr><td><a href="javascript:void(null);" class="'.$dclass.'"><strong>'.$extra['title'].'</strong>';
			} else {
				$html .= '<tr><td><a href="javascript:void(null);" onclick="iosRTPickExtra('.$xid.', 1, '.$extra['xgid'].');" class="'.$dclass.'"><strong>'.$extra['title'].'</strong>';
			}
			if ($extra['description'] != '') { $html .= '<div class="iosrt_extra_desc">'.$extra['description'].'</div>'; }
			if ($price_note != '') { $html .= '<div class="iosrt_extra_pricetip">'.$price_note.'</div>'; }
			$html .= "</a></td>\n";
			$html .= '<td class="iosrt_extraquatd"><select name="quantity_x'.$xid.'" id="iosrt_quantity_x'.$xid.'" data-xgid="'.$extra['xgid'].'" data-singleselect="'.$singleselect.'" class="iosrt_select" onchange="iosRTPickExtra('.$xid.', 0, '.$extra['xgid'].');" title="'.$eLang->get('QUANTITY').'">'."\n";
			if ($extra['required'] == 1) {
				$html .= '<option value="'.$extra['preselitems'].'" selected="selected">'.$extra['preselitems']."</option>\n";
			} else {
				for ($q=0; $q < ($extra['maxitems'] + 1); $q++) {
					$sel = ($q == $extra['preselitems']) ? ' selected="selected"' : '';
					$html .= '<option value="'.$q.'"'.$sel.'>'.$q."</option>\n";
				}
			}
			$html .= "</select></td>\n";

			$xamount_class = 'iosrt_extra_amount';
			if ($extra['preselitems'] > 0) { $xamount_class = 'iosrt_extra_amount iosrt_extra_amount_selected'; }

			$html .= '<td class="iosrt_extraamatd"><div id="iosrt_whamount_x'.$xid.'" class="'.$xamount_class.'"><span id="iosrt_hamount_x'.$xid.'">'.number_format($x_init_price, 2, '.', '').'</span> '.$currency->currencySymbol()."</div>\n";
			$html .= '<input type="hidden" name="price_x'.$xid.'" id="iosrt_price_x'.$xid.'" dir="ltr" value="0.00" data-price="'.$seasonaldata['price'].'" data-maxprice="'.$extra['maxprice'].'" data-priceperday="'.$seasonaldata['priceperday'].'" data-priceformula="'.$seasonaldata['priceformula'].'" data-freeitems="'.$seasonaldata['freeitems'].'" data-priceduration="'.$seasonaldata['priceduration'].'" />'."\n";
			$html .= '<input type="hidden" name="xid_x'.$xid.'" id="iosrt_xid_x'.$xid.'" dir="ltr" value="'.$xid.'" data-name="'.addslashes($extra['title']).'" data-seasons="'.$jseasons.'" />'."\n";
			$html .= "</td>\n</tr>\n";
		}

		$html .= "</table>\n";
		$html .= "</div>\n";//iosrt_box

		$txt = ($all_xids) ? implode(',', $all_xids) : '';
		$html .= '<input type="hidden" name="xids" id="iosrt_allxids" dir="ltr" value="'.$txt.'" />'."\n";

		return $html;
	}


	/************************************/
	/* MAKE MODEL HTML (SEARCH RESULTS) */
	/************************************/
	private function makeModelHTML($gid, $mid, $group, $modelsfeatures, $priceincludes, $categories, $durtxt, $company, $eLang, $currency, $bdata, $places) {
		$model = $group['models'][$mid];

		$images = array();
		if ($model['image'] != '') { $images[] = $company->imagesurl.'large/'.$model['image']; }
		if (count($model['images']) > 0) {
			foreach ($model['images'] as $img) { $images[] = $company->imagesurl.'large/'.$img; }
		}
		if (!$images) { $images[] = $this->options['clienturl'].'/images/noimage.png'; }

		$fullamountpc = (isset($group['price']['fullamountpc'])) ? $group['price']['fullamountpc'] : 0;
		if ($company->surfee == '100pc') { $fullamountpc = 0; }//no fullamount discount if surety fee is 100%

		$modelprice = $this->calcModelPrice($group, $company, $currency, $eLang, $fullamountpc);

		if ($group['categories'] != '') {
			$modelfilters = 'categories:'.$group['categories'];
			$ctgs = explode(',', $group['categories']);
			$parts = array();
			foreach ($ctgs as $ctg) { $parts[] = 'categories:'.$ctg; }
			$modelfilters = implode(';', $parts);
			$modelfilters .= ';vtype:'.$group['vtype'];
			if ($model['modelparams'] != '') { $modelfilters .= ';'.$model['modelparams']; }
		} else {
			$modelfilters = 'vtype:'.$group['vtype'];
			if ($model['modelparams'] != '') { $modelfilters .= ';'.$model['modelparams']; }
		}

		$html = '<div class="iosrt_model" id="iosrt_model_'.$mid.'" data-filters="'.$modelfilters.'">'."\n";
		$html .= '<div class="iosrt_model_colswrap">'."\n";

		$html .= '<div class="iosrt_model_main">'."\n";
		//$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span><span class="iosrt_model_badge">Badge here</span></h2>'."\n";

		if ($this->options['titlepos'] == 0) {
			if ($group['vtype'] == 'boat') {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].'</span></h2>'."\n";
			} else {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span></h2>'."\n";
			}
			$html .= '<div class="iosrt_model_subtitle">'.$this->makeModelSubtitle($group['title'], $group['vtype'], $group['categories'], $categories)."</div>\n";
		}

		switch ($this->options['imagewidth']) {
			case 480: $imgaddon = ' iosrt_model_img480'; $feataddon = ' iosrt_model_feat480'; break;
			case 400: $imgaddon = ' iosrt_model_img400'; $feataddon = ' iosrt_model_feat400'; break;
			case 320: $imgaddon = ' iosrt_model_img320'; $feataddon = ' iosrt_model_feat320'; break;
			case 240: default: $imgaddon = ''; $feataddon = ''; break;
		}

		$html .= '<div class="iosrt_model_imgfeat">'."\n";
		if (count($images) > 1) {
			$this->hasgallery = true;
			$html .= '<div class="iosrt_model_img'.$imgaddon.'">'."\n";
			$html .= '<div class="iosrt_swiper">'."\n";
			$html .= '<div class="swiper-wrapper">'."\n";
			foreach ($images as $img) {
				$html .= '<figure class="swiper-slide"><img src="'.$img.'" alt="vehicle image" /></figure>'."\n";
			}
			$html .= "</div>\n";
			$html .= '<div class="swiper-pagination"></div>'."\n";
			$html .= '<div class="swiper-button-prev"></div>'."\n";
			$html .= '<div class="swiper-button-next"></div>'."\n";
			$html .= "</div>\n";
			$html .= "</div>\n";//iosrt_model_img
		} else {
			$html .= '<figure class="iosrt_model_img'.$imgaddon.'"><img src="'.$images[0].'" alt="'.$model['title'].'" /></figure>'."\n";
		}

		$txt = '';
		if ($this->options['titlepos'] == 1) {
			if ($group['vtype'] == 'boat') {
				$txt .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].'</span></h2>'."\n";
			} else {
				$txt .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span></h2>'."\n";
			}
			$txt .= '<div class="iosrt_model_subtitle">'.$this->makeModelSubtitle($group['title'], $group['vtype'], $group['categories'], $categories)."</div>\n";
		}
		$txt .= $this->makeModelParamsHTML($model['modelparams'], $group['vtype'], $modelsfeatures, $eLang, $this->options['modelfeaticons']);
		if ($txt != '') { $html .= '<div class="iosrt_model_feat'.$feataddon.'">'.$txt.'</div>'."\n"; }
		$html .= "</div>\n";//iosrt_model_imgfeat

		if ($this->options['titlepos'] == 2) {
			if ($group['vtype'] == 'boat') {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].'</span></h2>'."\n";
			} else {
				$html .= '<h2 class="iosrt_model_h2"><span class="iosrt_model_title">'.$model['title'].' <span>'.$eLang->get('OR_SIMILAR').'</span></span></h2>'."\n";
			}
			$html .= '<div class="iosrt_model_subtitle">'.$this->makeModelSubtitle($group['title'], $group['vtype'], $group['categories'], $categories)."</div>\n";
		}

		$txt = $this->availableInLocs($model['lids'], $places, $group['vtype'], $eLang);
		if ($txt != '') { $html .= $txt; }

		if ($model['description'] != '') { $html .= '<div class="iosrt_model_description">'.$model['description']."</div>\n"; }
		$txt = $this->makePriceIncludesHTML($mid, $group['priceincludes'], $priceincludes, $eLang);
		if ($txt != '') { $html .= '<div class="iosrt_model_priceinc">'.$txt.'</div>'."\n"; }
		$html .= "</div>\n";//iosrt_model_main

		$fullamountpc = 0;
		if ($company->surfee != '100pc') {
			if (isset($group['price']['fullamountpc'])) {
				if ($group['price']['fullamountpc'] > 0) { $fullamountpc = $group['price']['fullamountpc']; }
			}
		}

		$class = ($fullamountpc > 0) ? 'iosrt_model_side_data2box' : 'iosrt_model_side_databox';
		$html .= '<div class="iosrt_model_side iosrt_model_bgside">'."\n";
		$html .= '<div class="'.$class.'">'."\n";
		if ($model['ribbon'] != '') {
			$css_addon = ($model['ribboncolor'] != '') ? ' iosrt_ribbon_'.$model['ribboncolor'] : '';
			$html .= '<h3 class="iosrt_model_ribbon'.$css_addon.'"><div>'.$model['ribbon'].'</div></h3>';
		}

		if ($modelprice['normalamount'] > ($modelprice['amount'] + 1)) {//plus 1 to avoid rounding errors
			if ($modelprice['discounts']) {
				$tip = implode(', ', $modelprice['discounts']);
				$html .= '<div class="iosrt_model_normalprice iosrt_wtooltip" title="'.$tip.'">'.$currency->shortprice($modelprice['normalamount'], $this->options['roundprices']).'</div>';	
			} else {
				$html .= '<div class="iosrt_model_normalprice">'.$currency->shortprice($modelprice['normalamount'], $this->options['roundprices']).'</div>';
			}
		}
		$html .= '<div class="iosrt_model_price">'.$currency->formatprice($modelprice['amount'], 1, $this->options['roundprices']).'<span>'.$currency->currencyISO().'</span></div>';
		$html .= '<div class="iosrt_model_pricedesc">'.$durtxt.'</div>'."\n";
		if ($modelprice['rest_amount'] > 0) {
			$html .= '<div class="iosrt_model_priceother" id="iosrt_rest_trig'.$mid.'">'.$eLang->get('OTHER_CHARGES').' <span>'.$currency->shortprice($modelprice['rest_amount'], $this->options['roundprices']).'</span></div>';
			if ($modelprice['rest_parts']) {
				$html .= '<div id="iosrt_rest_tip'.$mid.'" class="iosrt_invisible">'.implode('<br />', $modelprice['rest_parts']).'</div>';
				$html .= '<script>tippy(\'#iosrt_rest_trig'.$mid.'\', { arrow: true, html: \'#iosrt_rest_tip'.$mid.'\' })</script>'."\n";
			}
		}
		if ($modelprice['vat_note'] != '') { $html .= '<div class="iosrt_model_vatnote">'.$modelprice['vat_note']."</div>\n"; }
		$html .= "</div>\n";//iosrt_model_side_databox / iosrt_model_side_data2box

		$link = $bdata['urlbase'].'?pickup='.urlencode($bdata['pickup']).'&amp;dropoff='.urlencode($bdata['dropoff']).'&amp;picklid='.$bdata['picklid'].'&amp;droplid='.$bdata['droplid'].'&amp;mid='.$mid.'&amp;step=2';
		if ($fullamountpc > 0) {
			$linkfull = $link.'&amp;fullamount=1';
			$tip = '';
			if (strpos($company->surfee, 'pc') === false) {
				$sf = (int)$company->surfee;
				if ($sf > 0) {
					$v = $currency->longprice($sf);
					$tip = sprintf($eLang->get('BOOK_PAYNOW_AMOUNT'), $v);
				} else {
					$tip = $eLang->get('BOOK_PAYNOW_NOTHING');
				}
			} else {
				$sf = intval(str_replace('pc', '', $company->surfee));
				if ($sf > 0) {
					$tip = sprintf($eLang->get('BOOK_PAYNOW_PC'), $sf.'%');
				} else {
					$tip = $eLang->get('BOOK_PAYNOW_NOTHING');
				}
			}

			$html .= '<div class="iosrt_model_side_book2box">'."\n";
			$html .= '<a href="'.$link.'" class="iosrt_abtn1book iosrt_wtooltip" title="'.$tip.'">'.$eLang->get('BOOK_BUTTON').'<span class="iosrt_btnnote">'.$currency->longprice($modelprice['amount']).'</span></a>'."\n";
			$html .= '<div class="iosrt_btnbook_or">'.$eLang->get('OR_BOOK').'</div>';
			$tip = sprintf($eLang->get('FULLAMOUNT_DISC_PAYNOW'), $fullamountpc.'%').' '.$eLang->get('NONREFUND_DISCOUNT');
			$html .= '<a href="'.$linkfull.'" class="iosrt_abtn2book iosrt_wtooltip" title="'.$tip.'">'.$currency->longprice($modelprice['fullamount']);
			$html .= '<span class="iosrt_btnnote">'.$eLang->get('100_PREPAYMENT').'</span>';
			$html .= '</a>'."\n";
			$html .= '<div class="iosrt_btnbook_explain">'.$eLang->get('NONREFUND_DISCOUNT').'</div>';
		} else {
			$html .= '<div class="iosrt_model_side_bookbox">'."\n";
			$html .= '<a href="'.$link.'" class="iosrt_abtn">'.$eLang->get('BOOK_BUTTON').'</a>'."\n";
		}

		//$html .= '<a href="javascript:void(null);" onclick="alert(\'TODO\');" class="iosrt_model_more">'.$eLang->get('MORE_INFO')."</a>\n";
		//available vehicles: '.$group['availvehicles']

		if ($model['similar']) {
			if ($group['vtype'] == 'car') {
				$title = $eLang->get('SIMILAR_CARS');
			} else if ($group['vtype'] == 'moto') {
				$title = $eLang->get('SIMILAR_MOTOS');
			} else if ($group['vtype'] == 'boat') {
				$title = $eLang->get('SIMILAR_BOATS');
			} else {
				$title = $eLang->get('SIMILAR_VEHICLES');
			}
			$sims = array();
			foreach ($model['similar'] as $smid => $sim) { $sims[] = $sim['title']; }
			$html .= '<a href="javascript:void(null);" class="iosrt_model_more" id="iosrt_similar_trig'.$mid.'">'.$title."</a>\n";
			$html .= '<div id="iosrt_similar_tip'.$mid.'" class="iosrt_invisible">'.implode('<br />', $sims).'</div>';
			$html .= '<script>tippy(\'#iosrt_similar_trig'.$mid.'\', { arrow: true, html: \'#iosrt_similar_tip'.$mid.'\' })</script>'."\n";
		}

		$html .= "</div>\n";//iosrt_side_bookbox

		$html .= "</div>\n";//iosrt_model_side
		$html .= "</div>\n";//iosrt_model_colswrap

		$html .= "</div>\n";//iosrt_model

		return $html;
	}


	/**********************************************/
	/* GENERATE MODEL AVAILABLE IN LOCATIONS TEXT */
	/**********************************************/
	private function availableInLocs($lids, $places, $vtype, $eLang, $justlocs=false) {
		if ($this->options['modellocs'] == 0) { return ''; }

		$n = count($lids);
		$tip = '';
		if ($n == 0) {//available in all locs
			if ($this->options['modellocs'] == 1) { return ''; } //dont display
			$txt = $eLang->get('AVAILABLE_IN_ALL_LOCATIONS');
		} else {
			$locations = array();
			if ($places) {
				foreach ($places as $country => $data) {
					if (!$data['areas']) { continue; }
					foreach ($data['areas'] as $aid => $area) {
						if (!$area['locations']) { continue; }
						foreach ($area['locations'] as $lid => $loc) {
							$locations[$lid] = array('area' => $area['title'], 'location' => $loc['title']);
						}
					}
				}
			}

			$parts = array();
			foreach ($lids as $lid) {
				if (!isset($locations[$lid])) { continue; }
				$parts[] = $locations[$lid]['location'].' '.$locations[$lid]['area'];
			}

			if ($parts) {
				$tip = addslashes(implode(', ', $parts));
			} else {
				$tip = '';
			}

			$display_tip = $parts ? true : false;
			if ($justlocs) {
				if ($n > 2) {
					$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATIONS'), $n);
				} else if ($n == 1) {
					$tip = '';
					$lid = $lids[0];
					if (isset($locations[$lid])) {
						$txt = $locations[$lid]['location'].', '.$locations[$lid]['area'];
					} else {
						$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATIONS'), 1);
					}
				} else {
					$tip = '';
					$lid1 = $lids[0];
					$lid2 = $lids[1];
					if (isset($locations[$lid1]) && isset($locations[$lid2])) {
						$txt = $locations[$lid1]['location'].', '.$locations[$lid1]['area'].' &amp; '.$locations[$lid2]['location'].', '.$locations[$lid2]['area'];
					} else {
						$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATIONS'), 2);
					}
				}
			} else {
				if ($n > 1) {
					$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATIONS'), $n);
				} else {
					$tip = '';
					$lid = $lids[0];
					if (isset($locations[$lid])) {
						$name = $locations[$lid]['location'].', '.$locations[$lid]['area'];
						$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATION'), $name);
					} else {
						$txt = sprintf($eLang->get('AVAILABLE_IN_LOCATIONS'), 1);
					}
				}
			}
		}

		if ($justlocs) {
			if ($tip != '') {
				$html = '<div class="iosrt_availinjlocs iosrt_wtooltip" title="'.$tip.'"><span class="iosrt_availinjlocs_'.$vtype.'">'.$txt.'</span></div>';
			} else {
				$html = '<div class="iosrt_availinjlocs"><span class="iosrt_availinjlocs_'.$vtype.'">'.$txt.'</span></div>';
			}
		} else {
			if ($tip != '') {
				$html = '<div class="iosrt_availinlocs iosrt_availinlocs_'.$vtype.' iosrt_wtooltip" title="'.$tip.'">'.$txt.'</div>';
			} else {
				$html = '<div class="iosrt_availinlocs iosrt_availinlocs_'.$vtype.'">'.$txt.'</div>';
			}			
		}

		return $html;
	}


	/************************************************/
	/* MAKE SIDE COLUMN (SEARCH/CATEGORIES/FILTERS) */
	/************************************************/
	private function sideColumn($groups, $categories, $modelsfeatures, $fdata, $eLang) {
		if ($this->options['sidecolumn'] == '') { return ''; }//just in case

		$blocks = explode(',', $this->options['sidecolumn']);
		$searchidx = array_search('search', $blocks);
		$vtypeidx = array_search('vtype', $blocks);
		$categoriesidx = array_search('categories', $blocks);
		if (($searchidx !== false) || ($vtypeidx !== false) || ($categoriesidx !== false)) {//search must be first, vtype second and categories third
			$tmpblocks = $blocks;
			$blocks = array();
			if ($searchidx !== false) { $blocks[] = 'search'; }
			if ($vtypeidx !== false) { $blocks[] = 'vtype'; }
			if ($categoriesidx !== false) { $blocks[] = 'categories'; }

			foreach ($tmpblocks as $block) {
				if (($block == 'search') || ($block == 'categories') || ($block == 'vtype')) { continue; }
				$blocks[] = $block;
			}
			unset($tmpblocks);
		}
		unset($searchidx, $categoriesidx);
		if (!$blocks) { return ''; }

		$filter_categories = array();
		$features = array();//all features and their values
		$vtypes = array();

		foreach ($groups as $gid => $group) {
			$gvtype = $group['vtype'];
			if (!in_array($gvtype, $vtypes)) { $vtypes[] = $gvtype; }
			if ($group['categories'] != '') {
				$parts = explode(',', $group['categories']);
				foreach ($parts as $part) {
					if (isset($categories[$gvtype][$part])) { $filter_categories[$part] = $categories[$gvtype][$part]; }
				}
			}

			if (!$group['models']) { continue; }
			foreach ($group['models'] as $mid => $model) {
				if (trim($model['modelparams']) == '') { continue; }
				$parts = explode(';', $model['modelparams']);
				foreach ($parts as $part) {
					$feat = explode(':', $part);
					if (count($feat) != 2) { continue; }
					$name = $feat[0];
					if (!in_array('all', $blocks)) {//"all" keyword is used for all model features
						if (!in_array($name, $blocks)) { continue; }
					} else {
						if (!in_array($name, $blocks)) { $blocks[] = $name; }
					}
					if (!isset($features[$name])) {
						$features[$name] = array();
						$features[$name][] = $feat[1];
					} else {
						if (!in_array($feat[1], $features[$name])) { $features[$name][] = $feat[1]; }
					}
				}
			}
		}

		$vtype = (count($vtypes) !== 1) ? '' : $vtypes[0];
		if (count($vtypes) < 2) {//dont display vtypes filters if only 1 vehicle type
			$k = array_search('vtype', $blocks);
			if ($k !== false) { unset($blocks[$k]); }
		}

		$filters = array();
		if ($vtype == 'car') {
			$filters['general'] = array('title' => $eLang->get('CAR_SPECIFICATIONS'), 'options' => array());
		} else if ($vtype == 'boat') {
			$filters['general'] = array('title' => $eLang->get('SPECIFICATION'), 'options' => array());
		} else {
			$filters['general'] = array('title' => $eLang->get('VEHICLE_SPECIFICATIONS'), 'options' => array());
		}
		if ($features) {
			foreach ($features as $name => $values) {
				$item = false;
				if ($vtype != '') {
					if (isset($modelsfeatures[$vtype])) {
						if (isset($modelsfeatures[$vtype][$name])) { $item = $modelsfeatures[$vtype][$name]; }
					}
				} else {
					foreach ($modelsfeatures as $vtype => $mfeatures) {
						if (isset($mfeatures[$name])) { $item = $mfeatures[$name]; break; }
					}
				}
				if (!$item) { continue; }

				$type = $item['type'];
				if ($type == 'number') {
					if (count($values) > 1) {
						$filters[$name] = array('title' => $item['title'], 'options' => array());
						foreach ($values as $value) {
							$filters[$name]['options'][] = array('title' => $value, 'value' => $value, 'feature' => $name);
						}
					} else {
						$filters['general']['options'][] = array('title' => $item['title'].': '.$values[0], 'value' => $values[0], 'feature' => $name);
					}
				} else if ($type == 'select') {
					if (!$item['options']) { continue; }
					$n = count($values);
					if ($n > 1) {
						$filters[$name] = array('title' => $item['title'], 'options' => array());
					}
					foreach($item['options'] as $opt) {
						foreach ($values as $value) {
							if ($value == $opt['value']) {
								if ($n > 1) {
									$filters[$name]['options'][] = array('title' => $opt['title'], 'value' => $opt['value'], 'feature' => $name);
								} else {
									$filters['general']['options'][] = array('title' => $item['title'].': '.$opt['title'], 'value' => $opt['value'], 'feature' => $name);
								}
							}
						}
					}
				} else if ($type == 'yesno') {
					if ((count($values) == 1) && ($values[0] == 1)) {
						$filters['general']['options'][] = array('title' => $item['title'], 'value' => 1, 'feature' => $name);
					}
				} else {
					continue;//not supported type!
				}
			}
		}

		$html = '';
		$filters_box_opened = false;
		$all_filters = array();
		foreach ($blocks as $block) {
			if ($block == 'all') { continue; }
			if ($block == 'search') {
				//convert from system to user format (YYYY-MM-DD => DD-MM-YYYY)
				$pd = explode('-', $fdata['pickdate']);
				$fdata['pickdate'] = $pd[2].'-'.$pd[1].'-'.$pd[0];
				$dd = explode('-', $fdata['dropdate']);
				$fdata['dropdate'] = $dd[2].'-'.$dd[1].'-'.$dd[0];

				$html .= '<div class="iosrt_filters_searchbox">'."\n";
				$html .= '<h3 class="iosrt_filter_stitle"><a href="javascript:void(null);" class="iosrt_filter_lnkup" onclick="iosRTToggleSearch(this);">'.$eLang->get('NEW_SEARCH')."</a></h3>\n";
				$html .= '<h3 class="iosrt_filter_stitlemob"><a href="javascript:void(null);" class="iosrt_filter_lnkdown" onclick="iosRTToggleSearch(this);">'.$eLang->get('NEW_SEARCH')."</a></h3>\n";
				$html .= $this->availabilityForm('', true, $fdata);
				$html .= "</div>\n";
				continue;
			}
			if (!$filters_box_opened) {
				$html .= '<div class="iosrt_filters_box">'."\n";
				$filters_box_opened = true;
			}

			if ($block == 'vtype') {
				$html .= '<div class="iosrt_filters_wrapbox">'."\n";
				$html .= '<h3 class="iosrt_filter_title"><a href="javascript:void(null);" class="iosrt_filter_lnkup" onclick="iosRTToggleFilterGroup(\'vtype\', this);">'.$eLang->get('VEHICLE_TYPE')."</a></h3>\n";
				$html .= '<h3 class="iosrt_filter_titlemob"><a href="javascript:void(null);" class="iosrt_filter_lnkdown" onclick="iosRTToggleFilterGroup(\'vtype\', this);">'.$eLang->get('VEHICLE_TYPE')."</a></h3>\n";
				$html .= '<ul class="iosrt_filters" id="iosrt_filter_group_vtype">'."\n";
				foreach ($vtypes as $vtype) {
					switch ($vtype) {
						case 'car': $title = $eLang->get('CARS'); break;
						case 'moto': $title = $eLang->get('MOTORCYCLES'); break;
						case 'scooter': $title = $eLang->get('SCOOTERS'); break;
						case 'boat': $title = $eLang->get('BOATS'); break;
						case 'atv': $title = $eLang->get('ATVS'); break;
						case 'buggy': $title = $eLang->get('BUGGIES'); break;
						case 'bicycle': $title = $eLang->get('BICYCLES'); break;
						case 'other': $title = $eLang->get('OTHER_VEHICLES'); break;
						default: $title = ucfirst($vtype); break;
					}
					$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_vtype:'.$vtype.'">'.$title.'<input type="checkbox" value="'.$vtype.'" id="iosrt_filter_vtype:'.$vtype.'" onchange="iosRTFilter(false);" data-orgroup="vtype" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
					$all_filters[] = 'vtype:'.$vtype;
				}
				$html .= "</ul>\n";
				$html .= "</div>\n";
				continue;
			}

			if ($block == 'categories') {
				if (!$filter_categories) { continue; }
				$html .= '<div class="iosrt_filters_wrapboxctg">'."\n";
				$html .= '<h3 class="iosrt_filter_title"><a href="javascript:void(null);" class="iosrt_filter_lnkup" onclick="iosRTToggleFilterGroup(\'categories\', this);">'.$eLang->get('CATEGORY')."</a></h3>\n";
				$html .= '<h3 class="iosrt_filter_titlemob"><a href="javascript:void(null);" class="iosrt_filter_lnkdown" onclick="iosRTToggleFilterGroup(\'categories\', this);">'.$eLang->get('CATEGORY')."</a></h3>\n";
				$html .= '<ul class="iosrt_filters" id="iosrt_filter_group_categories">'."\n";
				foreach ($filter_categories as $name => $title) {
					$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_categories:'.$name.'">'.$title.'<input type="checkbox" value="'.$name.'" id="iosrt_filter_categories:'.$name.'" onchange="iosRTFilter(false);" data-orgroup="categories" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
					$all_filters[] = 'categories:'.$name;
				}
				$html .= "</ul>\n";
				$html .= "</div>\n";
				continue;
			}

			if (!isset($filters[$block])) { continue; }//filter put in "general" block

			$html .= '<div class="iosrt_filters_wrapbox">'."\n";
			$html .= '<h3 class="iosrt_filter_title"><a href="javascript:void(null);" class="iosrt_filter_lnkup" onclick="iosRTToggleFilterGroup(\''.$block.'\', this);">'.$filters[$block]['title']."</a></h3>\n";
			$html .= '<h3 class="iosrt_filter_titlemob"><a href="javascript:void(null);" class="iosrt_filter_lnkdown" onclick="iosRTToggleFilterGroup(\''.$block.'\', this);">'.$filters[$block]['title']."</a></h3>\n";
			$html .= '<ul class="iosrt_filters" id="iosrt_filter_group_'.$block.'">'."\n";
			foreach ($filters[$block]['options'] as $opt) {
				$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'">'.$opt['title'].'<input type="checkbox" value="'.$opt['value'].'" id="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'" onchange="iosRTSingleFilter(\''.$opt['feature'].'\', this, false);" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
				$all_filters[] = $opt['feature'].':'.$opt['value'];
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";
		}

		if ($filters['general']['options']) {
			if (!$filters_box_opened) {
				$html .= '<div class="iosrt_filters_box">'."\n";
				$filters_box_opened = true;
			}
			$html .= '<div class="iosrt_filters_wrapboxgen">'."\n";
			$html .= '<h3 class="iosrt_filter_title"><a href="javascript:void(null);" class="iosrt_filter_lnkup" onclick="iosRTToggleFilterGroup(\'general\', this);">'.$filters['general']['title']."</a></h3>\n";
			$html .= '<h3 class="iosrt_filter_titlemob"><a href="javascript:void(null);" class="iosrt_filter_lnkdown" onclick="iosRTToggleFilterGroup(\'general\', this);">'.$filters['general']['title']."</a></h3>\n";
			$html .= '<ul class="iosrt_filters" id="iosrt_filter_group_general">'."\n";
			foreach ($filters['general']['options'] as $opt) {
				$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'">'.$opt['title'].'<input type="checkbox" value="'.$opt['value'].'" id="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'" onchange="iosRTFilter(false);" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
				$all_filters[] = $opt['feature'].':'.$opt['value'];
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";
		}

		if ($filters_box_opened) {
			$html .= "</div>\n";//iosrt_filters_box
		}

		$html .= '<div class="iosrt_invisible" id="iosrt_all_filters" data-filters="'.implode(';', $all_filters).'"></div>'."\n";

		return $html;
	}


	/****************************************************/
	/* CALCULATE PRICE DISPLAYED IN SEARCH RESULTS PAGE */
	/****************************************************/
	private function calcModelPrice($group, $company, $currency, $eLang, $fullamountpc=0) {
		$modelprice = array(
			'normalamount' => 0.00,//amount without discounts
			'amount' => 0.00,
			'fullamount' => 0.00,//amount without fullamount discount applied
			'rest_amount' => 0.00,
			'rest_title' => '',
			'rest_parts' => array(),
			'vat_note' => '',
			'discounts' => array()
		);

		if ($group['price']['fakepc'] > 0) {
			$modelprice['discounts'][] = sprintf($eLang->get('SPECIAL_OFFER_DISCOUNT'), $group['price']['fakepc'].'%');
		}

		if ($group['price']['earlybookpc'] > 0) {
			$modelprice['discounts'][] = sprintf($eLang->get('EARLYDISC_PC_DAYS_PICKUP'), $group['price']['earlybookpc'].'%', $group['price']['earlybookdays']);
		}

		//Note: $group['price']['total'] is total price without VAT
		switch ($this->options['modelprice']) {
			case 0: //rental fee
				$modelprice['amount'] = $group['price']['rentalfee'] * ((100 - $group['price']['earlybookpc']) / 100);
				$pcd = $group['price']['earlybookpc'] + $group['price']['fakepc'];
				$modelprice['normalamount'] = (100 / (100 - $pcd)) * $modelprice['amount'];
				$modelprice['fullamount'] = ($modelprice['amount'] * (100 - $fullamountpc)) / 100;
			break;
			case 2://rental fee and rest charges
				$modelprice['amount'] = $group['price']['rentalfee'] * ((100 - $group['price']['earlybookpc']) / 100);
				$pcd = $group['price']['earlybookpc'] + $group['price']['fakepc'];
				$modelprice['normalamount'] = (100 / (100 - $pcd)) * $modelprice['amount'];
				$modelprice['fullamount'] = ($modelprice['amount'] * (100 - $fullamountpc)) / 100;
				$modelprice['rest_amount'] = $group['price']['total'] - $group['price']['rentalfee'];
				if ($modelprice['rest_amount'] > 0) {
					$parts = array();
					if ($group['price']['extrahoursfee'] > 0) {
						if (($company->taxispc == 1) && ($this->options['applyvat'] == 1)) {
							$p = $group['price']['extrahoursfee'] + (($group['price']['extrahoursfee'] * $company->tax) / 100);
						} else {
							$p = $group['price']['extrahoursfee'];
						}
						$parts[] = $eLang->get('EXTRA_HOURS_FEE').': '.$currency->longPrice($p);
					}
					if ($group['price']['afterhourspick'] > 0) {
						if (($company->taxispc == 1) && ($this->options['applyvat'] == 1)) {
							$p = $group['price']['afterhourspick'] + (($group['price']['afterhourspick'] * $company->tax) / 100);
						} else {
							$p = $group['price']['afterhourspick'];
						}
						$parts[] = $eLang->get('AFTER_WORKHOURS_PICKUP').': '.$currency->longPrice($p);
					}
					if (($group['price']['afterhoursdrop'] > 0) && ($this->options['applyvat'] == 1)) {
						if ($company->taxispc == 1) {
							$p = $group['price']['afterhoursdrop'] + (($group['price']['afterhoursdrop'] * $company->tax) / 100);
						} else {
							$p = $group['price']['afterhoursdrop'];
						}
						$parts[] = $eLang->get('AFTER_WORKHOURS_DROPOFF').': '.$currency->longPrice($p);
					}
					if ($group['price']['pickupfee'] > 0) {
						if (($company->taxispc == 1) && ($this->options['applyvat'] == 1)) {
							$p = $group['price']['pickupfee'] + (($group['price']['pickupfee'] * $company->tax) / 100);
						} else {
							$p = $group['price']['pickupfee'];
						}
						$parts[] = $eLang->get('PICKUP_FEE').': '.$currency->longPrice($p);
					}
					if ($group['price']['dropofffee'] > 0) {
						if (($company->taxispc == 1) && ($this->options['applyvat'] == 1)) {
							$p = $group['price']['dropofffee'] + (($group['price']['dropofffee'] * $company->tax) / 100);
						} else {
							$p = $group['price']['dropofffee'];
						}
						$parts[] = $eLang->get('DROPOFF_FEE').': '.$currency->longPrice($p);
					}
					if (($company->tax > 0) && ($company->taxispc == 0) && ($this->options['applyvat'] == 1)) {
						$parts[] = $eLang->get('VAT').': '.$currency->longPrice($company->tax);
					}
					$modelprice['rest_title'] = implode(', ', $parts);
					$modelprice['rest_parts'] = $parts;
					unset($parts);
				}
			break;
			case 1: default:
				$rest = $group['price']['total'] - $group['price']['rentalfee'];
				$disceb = ($group['price']['rentalfee'] * $group['price']['earlybookpc']) / 100;
				$discfa = ($group['price']['rentalfee'] * $fullamountpc) / 100;
				$modelprice['amount'] = $group['price']['rentalfee'] - $disceb + $rest;
				$pcd = $group['price']['earlybookpc'] + $group['price']['fakepc'];
				$modelprice['normalamount'] = (100 / (100 - $pcd)) * $modelprice['amount'];
				$modelprice['fullamount'] = $group['price']['rentalfee'] - $disceb - $discfa + $rest;
			break; //total price
		}

		if ($company->tax > 0) {
			if ($this->options['applyvat'] == 1) {
				if ($company->taxispc == 1) {
					$modelprice['amount'] = $modelprice['amount'] + (($modelprice['amount'] * $company->tax) / 100);
					$modelprice['normalamount'] = $modelprice['normalamount'] + (($modelprice['normalamount'] * $company->tax) / 100);
					$modelprice['rest_amount'] = $modelprice['rest_amount'] + (($modelprice['rest_amount'] * $company->tax) / 100);
					$modelprice['fullamount'] = $modelprice['fullamount'] + (($modelprice['fullamount'] * $company->tax) / 100);
				} else {
					if ($modelprice['rest_amount'] > 0) {
						$modelprice['rest_amount'] += $company->tax;
					} else {
						$modelprice['amount'] += $company->tax;
						$modelprice['normalamount'] += $company->tax;
						$modelprice['fullamount'] += $company->tax;
					}
				}
			}
		}

		if ($this->options['vatnote'] == 1) {
			if ($company->tax > 0) {
				if ($this->options['applyvat'] == 1) {
					$modelprice['vat_note'] = sprintf($eLang->get('VAT_X_INCLUDED'), $company->tax.'%');
				} else {
					$modelprice['vat_note'] = sprintf($eLang->get('VAT_X_NOT_INCLUDED'), $company->tax.'%');
				}
			} else {
				$modelprice['vat_note'] = $eLang->get('PRICES_INC_TAXES');
			}
		}

		return $modelprice;
	}


	/***********************/
	/* MAKE MODEL SUBTITLE */
	/***********************/
	private function makeModelSubtitle($title, $vtype, $categories, $allcategories) {
		$ctgs_final = array();
		if ($categories != '') {
			$parts = explode(',', $categories);
			foreach ($parts as $part) {
				if (isset($allcategories[$vtype][$part])) { $ctgs_final[] = $allcategories[$vtype][$part]; }
			}
		}
		if ($ctgs_final) {
			if ($vtype == 'boat') {
				return implode(', ', $ctgs_final);
			} else {
				return $title.' / '.implode(', ', $ctgs_final);
			}
		} else {
			return $title;
		}
	}


	/***************************************/
	/* MAKE MODEL PARAMETERS/FEATURES HTML */
	/***************************************/
	private function makeModelParamsHTML($modelparams, $vtype, $modelsfeatures, $eLang, $modelfeaticons=6, $onlydata=false) {
		if (trim($modelparams) == '') {
			if ($onlydata) {
				$data = array('iconitems' => array(), 'textitems' => array());
				return $data;
			}
			return '';
		}
		if (!isset($modelsfeatures[$vtype])) {
			if ($onlydata) {
				$data = array('iconitems' => array(), 'textitems' => array());
				return $data;
			}
			return '';
		}
		if (!$modelsfeatures[$vtype]) {
			if ($onlydata) {
				$data = array('iconitems' => array(), 'textitems' => array());
				return $data;
			}
			return '';
		}
		$modelfeaticons = (int)$modelfeaticons;

		$html = '';
		$parts = explode(';', $modelparams);
		$textitems = array();
		$iconitems = array();

		foreach ($parts as $part) {
			$arr = explode(':', $part);
			if (count($arr) != 2) { continue; }
			$name = trim($arr[0]);
			$original_v = trim($arr[1]);
			if (($name == '') || !isset($modelsfeatures[$vtype][$name])) { continue; }

			if ($name == 'tank') { continue; }//dont display tank
			if ($name == 'airbag') {
				$show = true;
				foreach ($parts as $part) {
					if (strpos($part, 'airnumbag:') === 0) { $show = false; break; }//number of airbags exist, dont display airbag yes/no
				}
				if (!$show) { continue; }
			}

			$type = $modelsfeatures[$vtype][$name]['type'];
			$title = $modelsfeatures[$vtype][$name]['title'];
			if ($type == 'number') {
				$v = (int)$arr[1];
				if (isset($modelsfeatures[$vtype][$name]['attributes']['notuse'])) {
					if ($modelsfeatures[$vtype][$name]['attributes']['notuse'] == $v) { continue; }
				}
				$value = $v;
				if (isset($modelsfeatures[$vtype][$name]['attributes']['unit'])) {
					$value .= $modelsfeatures[$vtype][$name]['attributes']['unit'];
				}
			} else if ($type == 'yesno') {
				$v = (int)$arr[1];
				if ($v == 0) { continue; }//do not display not supported features
				$value = '';
			} else if ($type == 'select') {
				if (!$modelsfeatures[$vtype][$name]['options']) { continue; }
				$v = $arr[1];
				$value = '';
				foreach($modelsfeatures[$vtype][$name]['options'] as $opt) {
					if ($opt['value'] == $v) { $value = $opt['title']; break; }
				}
				if ($value == '') { continue; }
				if ($name == 'cooling') { $title = $value; $value = ''; }
			} else {//text/other...
				$value = $arr[1];
			}

			if (count($iconitems) < $modelfeaticons) {
				switch ($name) {
					case 'baggages': case 'baggageslarge': case 'baggagessmall': case 'doors': case 'passengers':
						if (($name == 'passengers') && ($vtype == 'atv')) {
							$icon = $this->options['clienturl'].'/images/features/riders-atv.png';
						} else {
							$icon = $this->options['clienturl'].'/images/features/'.$name.'.png';
						}
						$itempop = '';
						if ($name == 'doors') {
							$itempop = 'numberOfDoors';
						} else if ($name == 'passengers') {
							$itempop = 'seatingCapacity';
						}
						$iconitems[] = array(
							'icon' => $icon,
							'text' => ($value != '') ? $value : $title,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'gearbox':
						if ($original_v == 2) {//Automatic
							$iconitems[] = array(
								'icon' => $this->options['clienturl'].'/images/features/transmition_auto.png',
								'text' => 'A',
								'title' => ($value != '') ? $title.': '.$value : $title
							);
						} else {//manual
							$iconitems[] = array(
								'icon' => $this->options['clienturl'].'/images/features/transmition_manual.png',
								'text' => 'M',
								'title' => ($value != '') ? $title.': '.$value : $title
							);
						}
					break;
					case 'aircondition':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/aircondition.png',
							'text' => 'AC',
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'wifi':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
							'text' => $title,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'airbag': case 'luggage': case 'helmetcase': case '4x4':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
							'text' => $eLang->get('YES'),
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'airnumbag':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/airbag.png',
							'text' => $original_v,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'abs': case 'gps': 
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
							'text' => strtoupper($name),
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'horsepower':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/horsepower.png',
							'text' => $original_v.'hp',
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'enginecc':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/engine.png',
							'text' => $original_v.'cc',
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'enginestrokes':
						$strokes = ($original_v == 2) ? '4T' : '2T';
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/strokes.png',
							'text' => $strokes,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'gearnumber':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
							'text' => $original_v,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'discbrakes':
						$txt = ($original_v == 0) ? $eLang->get('YES') : $original_v;
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
							'text' => $txt,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'riders':
						if ($vtype == 'scooter') {
							$icon = $this->options['clienturl'].'/images/features/riders-scooter.png';
						} else if ($vtype == 'atv') {
							$icon = $this->options['clienturl'].'/images/features/riders-atv.png';
						} else if ($vtype == 'buggy') {
							$icon = $this->options['clienturl'].'/images/features/riders-buggy.png';
						} else if ($original_v == 1) {
							$icon = $this->options['clienturl'].'/images/features/rider.png';
						} else {
							$icon = $this->options['clienturl'].'/images/features/riders.png';
						}
						$iconitems[] = array(
							'icon' => $icon,
							'text' => $original_v,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'wheelsize':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/wheel.png',
							'text' => $original_v,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					case 'tank':
						$iconitems[] = array(
							'icon' => $this->options['clienturl'].'/images/features/fuel.png',
							'text' => $original_v.'lt',
							'title' => ($value != '') ? $title.': '.$value.'lt' : $title
						);
					break;
					case 'skipper':
						if ($original_v == 0) {
							$icon = $this->options['clienturl'].'/images/features/skipper0.png';
							$txt = $eLang->get('NO');
						} else if ($original_v == 1) {
							$icon = $this->options['clienturl'].'/images/features/skipper1.png';
							$txt = $eLang->get('YES');
						} else if ($original_v == 2) {
							$icon = $this->options['clienturl'].'/images/features/skipper2.png';
							$txt = $eLang->get('YES').'/'.$eLang->get('NO');
						} else {
							$icon = $this->options['clienturl'].'/images/features/skipper.png';
							$txt = 'Skipper';
						}
						$iconitems[] = array(
							'icon' => $icon,
							'text' => $txt,
							'title' => ($value != '') ? $title.': '.$value : $title
						);
					break;
					default:
						if (file_exists($this->path.'/images/features/'.$name.$original_v.'.png')) {//auto icon (v1.4+)
							$icon = $this->options['clienturl'].'/images/features/'.$name.$original_v.'.png';
							if ($type == 'yesno') {
								$txt = ($original_v == 1) ? $eLang->get('YES') : $eLang->get('NO');
							} else {
								$txt = $original_v;
							}
							$iconitems[] = array(
								'icon' => $this->options['clienturl'].'/images/features/'.$name.$original_v.'.png',
								'text' => $txt,
								'title' => ($value != '') ? $title.': '.$value : $title
							);
						} else if (file_exists($this->path.'/images/features/'.$name.'.png')) {//auto icon (v1.4+)
							$icon = $this->options['clienturl'].'/images/features/'.$name.'.png';
							if ($type == 'yesno') {
								$txt = ($original_v == 1) ? $eLang->get('YES') : $eLang->get('NO');
							} else {
								$txt = $original_v;
							}
							$iconitems[] = array(
								'icon' => $this->options['clienturl'].'/images/features/'.$name.'.png',
								'text' => $txt,
								'title' => ($value != '') ? $title.': '.$value : $title
							);
						} else {
							$textitems[] = ($value != '')? $title.': '.$value : $title;
						}
					break;
				}
			} else {
				$textitems[] = ($value != '') ? $title.': '.$value : $title;
			}
		}

		if ($onlydata) {
			$data = array('iconitems' => $iconitems, 'textitems' => $textitems);
			return $data;
		}

		$html = '';
		if ($iconitems) {
			$html .= '<div class="iosrt_feat_icons">'."\n";
			foreach ($iconitems as $item) {

				$html .= '<div class="iosrt_feat_icon"><div><img src="'.$item['icon'].'" alt="'.$item['title'].'" title="'.$item['title'].'" /></div><span>'.$item['text']."</span></div>\n";
			}
			$html .= "</div>\n";
		}
		if ($textitems) {
			$html .= '<ul class="iosrt_feat_list">'."\n";
			foreach ($textitems as $item) {
				$html .= '<li>'.$item."</li>\n";
			}
			$html .= "</ul>\n";
		}

		return $html;
	}


	/**********************************/
	/* MAKE MODEL PRICE/INCLUDES HTML */
	/**********************************/
	private function makePriceIncludesHTML($mid, $priceincludes, $rtdata, $eLang, $twocolumns=true) {
		if (trim($priceincludes) == '') { return ''; }
		if (!$rtdata) { return ''; }

		$incs = explode(',', $priceincludes);

		$items = array();
		foreach ($incs as $inc) {
			if (!isset($rtdata[$inc])) { continue; }
			$items[] = array('title' => $rtdata[$inc]['title'], 'description' => $rtdata[$inc]['description']);
		}
		if (!$items) { return ''; }

		$total = count($items);
		$half = ceil($total / 2);
		if ($twocolumns) {
			if ($total < 2) { $twocolumns = false; }
		}

		$html = '<h3 class="iosrt_model_h3"><a href="javascript:void(null);" onclick="iosRTTogglePI('.$mid.');" title="'.$eLang->get('SHOW_HIDE_DETAILS').'">'.$eLang->get('OUR_PRICES_INCLUDE')."</a></h3>\n";

		if ($twocolumns) {
			$html .= '<div class="iosrt_model_pi2cols" id="iosrt_piwrap'.$mid.'">'."\n";
			$html .= '<div class="iosrt_model_pileft">'."\n";
			$html .= '<ul class="iosrt_princ_list">'."\n";
			for ($i=0; $i < $half; $i++) {
				if ($items[$i]['description'] != '') {
					$html .= '<li class="iosrt_wtooltip" title="'.$items[$i]['description'].'">'.$items[$i]['title']."</li>\n";
				} else {
					$html .= '<li>'.$items[$i]['title']."</li>\n";
				}
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";//iosrt_model_pileft
			$html .= '<div class="iosrt_model_piright">'."\n";
			$html .= '<ul class="iosrt_princ_list">'."\n";
			for ($i=0; $i < $total; $i++) {
				if ($i < $half) { continue; }
				if ($items[$i]['description'] != '') {
					$html .= '<li class="iosrt_wtooltip" title="'.$items[$i]['description'].'">'.$items[$i]['title']."</li>\n";
				} else {
					$html .= '<li>'.$items[$i]['title']."</li>\n";
				}
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";//iosrt_model_piright
			$html .= "</div>\n";//iosrt_model_pi2cols
			return $html;
		}

		$html .= '<div class="iosrt_model_pi1cols" id="iosrt_piwrap'.$mid.'">'."\n";
		$html .= '<ul class="iosrt_princ_list">'."\n";
		for ($i=0; $i < $total; $i++) {
			if ($items[$i]['description'] != '') {
				$html .= '<li class="iosrt_wtooltip" title="'.$items[$i]['description'].'">'.$items[$i]['title']."</li>\n";
			} else {
				$html .= '<li>'.$items[$i]['title']."</li>\n";
			}
		}
		$html .= "</ul>\n";
		$html .= "</div>\n";//iosrt_model_pi1cols

		return $html;
	}


	/***************************/
	/* CHECKCOUPON (AJAX TASK) */
	/***************************/
	private function checkCoupon() {
		$eLang = $this->lang;

		$pat = "#([\']|[\!]|[\(]|[\)]|[\;]|[\"]|[\$]|[\#]|[\<]|[\>]|[\*]|[\%]|[\~]|[\`]|[\^]|[\|]|[\{]|[\}]|[\\\])#u";
		$couponcode = filter_input(INPUT_POST, 'couponcode', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		if ($couponcode != '') { $couponcode = trim(preg_replace($pat, '', $couponcode)); }
		if (strlen($couponcode) < 3) {
			$response = array('success' => 0, 'message' => $eLang->get('INV_COUPON_CODE'));
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$pickup = trim(filter_input(INPUT_POST, 'pickup', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH));
		$dropoff = trim(filter_input(INPUT_POST, 'dropoff', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH));
		$charge_days = isset($_POST['charge_days']) ? (int)$_POST['charge_days'] : 0;

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'couponcode' => $couponcode,
			'pickup' => $pickup,
			'dropoff' => $dropoff,
			'charge_days' => $charge_days,
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/checkcoupon';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/checkcoupon';
		}

		$xml = $this->helper->rcpost($taskurl, $params);
		if ($this->helper->getError() != '') {//RC connection failed
			$response = array('success' => 0, 'message' => $this->helper->getError());
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$results = $this->helper->parseCheckCouponResults($xml);
		if (!$results || ($this->helper->getError() != '')) {
			$response = array('success' => 0, 'message' => $this->helper->getError());
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$response = array('success' => 1, 'message' => '', 'code' => $results->code, 'ispc' => $results->ispc, 'amount' => $results->amount, 'currency' => $results->currency);
		$encoded = json_encode($response);
		$this->pageHeaders('application/json');
		echo $encoded;
		exit;
	}


	/************************************/
	/* MAKE CALENDAR CONFIGURATION DATA */
	/************************************/
	public function calendarData() {
		$eLang = $this->lang;

		$company = $this->helper->basicRequest('company');
		$pickplushours = isset($company->pickplushours) ? (int)$company->pickplushours : 0;//v1.8

		$calendar = array(
			'disranges' => array(),
			'nwranges' => array(),
			'rentdranges' => array(),
			'minrentdays' => 0,
			'maxrentdays' => 100,
			'pickupdayofweek' => $this->options['pickupday'],
			'dropoffdayofweek' => $this->options['dropoffday'],
			'hidedrop' => $this->options['hidedrop'],
			'pickplushours' => $pickplushours
		);

		$places = $this->helper->basicRequest('places', true);
		if (!$places) {
			$this->errormsg = $this->helper->getError();
			return $calendar;
		}

		$lids = array();
		foreach ($places as $countrydata) {
			if (!$countrydata['areas']) { continue; }
			foreach ($countrydata['areas'] as $area) {
				if (!$area['locations']) { continue; }
				foreach ($area['locations'] as $lid => $ldata) { $lids[] = $lid; }
			}
		}

		if (file_exists($this->path.'/cache/'.$this->options['apiid'].'/noworkhours.'.$this->lang->guilang.'.php')) {
			$path = $this->path.'/cache/'.$this->options['apiid'].'/noworkhours.'.$this->lang->guilang.'.php';
		} else if (file_exists($this->path.'/cache/'.$this->options['apiid'].'/noworkhours.php')) {
			$path = $this->path.'/cache/'.$this->options['apiid'].'/noworkhours.php';
		} else {
			$path = '';
		}

		if ($path == '') {
			$noworkhours = $this->helper->basicRequest('noworkhours');
			unset($noworkhours);
			if (file_exists($this->path.'/cache/'.$this->options['apiid'].'/noworkhours.'.$this->lang->guilang.'.php')) {
				$path = $this->path.'/cache/'.$this->options['apiid'].'/noworkhours.'.$this->lang->guilang.'.php';
			} else if (file_exists($this->path.'/cache/'.$this->options['apiid'].'/noworkhours.php')) {
				$path = $this->path.'/cache/'.$this->options['apiid'].'/noworkhours.php';
			} else {
				$path = '';
			}
		}

		if ($path == '') { return $calendar; }

		include($path);

		if (isset($rtdata)) {
			$noworkdates = $this->getNoworkDates($rtdata, $lids);
		} else {
			$noworkdates = array();
		}

		$disabled_dates = array();
		$nowork_hours = array();
		if ($noworkdates) {
			foreach ($noworkdates as $rlid => $data) {
				if (!isset($disabled_dates[$rlid])) { $disabled_dates[$rlid] = array(); }
				if (!isset($nowork_hours[$rlid])) { $nowork_hours[$rlid] = array(); }
				foreach ($data as $yz => $rows) {
					if (($rows[0]['start'] == '00:00') && ($rows[0]['end'] == '23:59')) {//disabled date
						$disabled_dates[$rlid][$yz] = gmdate('d-m-Y', $rows[0]['ts']);
						continue;
					}
					foreach ($rows as $row) {
						if (!isset($nowork_hours[$rlid][$yz])) {
							$nowork_hours[$rlid][$yz] = $row['start'].'-'.$row['end'];
						} else {
							$nowork_hours[$rlid][$yz] .= ','.$row['start'].'-'.$row['end'];
						}
					}
				}
			}
		}
		unset($noworkdates);

		$disabled_ranges = array();
		if ($disabled_dates) {//make ranges
			foreach ($disabled_dates as $rlid => $data) {
				if (!isset($disabled_ranges[$rlid])) { $disabled_ranges[$rlid] = array(); }
				$start = -1;
				foreach ($data as $yz => $d) {
					if ($start == -1) { $start = $yz; }
					$n = $yz + 1;
					if (isset($disabled_dates[$rlid][$n])) { continue; }
					$disabled_ranges[$rlid][] = array('from' => $disabled_dates[$rlid][$start], 'to' => $disabled_dates[$rlid][$yz]);
					$start = -1;
				}
			}
		}
		unset($disabled_dates);

		$nowork_ranges = array();
		if ($nowork_hours) {//make ranges
			foreach ($nowork_hours as $rlid => $data) {
				if (!isset($nowork_ranges[$rlid])) { $nowork_ranges[$rlid] = array(); }
				$start = -1;
				foreach ($data as $yz => $hours) {
					if ($start == -1) { $start = $yz; }
					$n = $yz + 1;
					if (isset($nowork_hours[$rlid][$n])) {
						if ($nowork_hours[$rlid][$start] == $nowork_hours[$rlid][$n]) { continue; }//next day has same settings
					}
					$nowork_ranges[$rlid][] = array('from' => $start, 'to' => $yz, 'hours' => $nowork_hours[$rlid][$start]);
					$start = -1;
				}
			}
		}
		unset($nowork_hours, $rtdata);

		$calendar['disranges'] = $disabled_ranges;
		$calendar['nwranges'] = $nowork_ranges;

		if (!file_exists($this->path.'/cache/'.$this->options['apiid'].'/rentdurations.php')) { return $calendar; }
		include($this->path.'/cache/'.$this->options['apiid'].'/rentdurations.php');

		if (!isset($rtdata)) { return $calendar; }
	
		$rentdurations = $this->getRentalDurations($rtdata, $lids);

		$rentdur_ranges = array();
		if ($rentdurations) {//make ranges
			foreach ($rentdurations as $rlid => $data) {
				if (!isset($rentdur_ranges[$rlid])) { $rentdur_ranges[$rlid] = array(); }
				$start = -1;
				foreach ($data as $yz => $hours) {
					if ($start == -1) { $start = $yz; }
					$n = $yz + 1;
					if (isset($rentdurations[$rlid][$n])) {
						if ($rentdurations[$rlid][$start] == $rentdurations[$rlid][$n]) { continue; }//next day has same settings
					}
					$rentdur_ranges[$rlid][] = array('from' => $start, 'to' => $yz, 'min' => $rentdurations[$rlid][$start]['minrentdays'], 'max' => $rentdurations[$rlid][$start]['maxrentdays']);
					$start = -1;
				}
			}
		}
		$calendar['rentdranges'] = $rentdur_ranges;
		unset($rentdurations, $rtdata, $rentdur_ranges);

		$this->errormsg = $this->helper->getError();
		$calendar['minrentdays'] = (int)$company->minrentdays;
		$calendar['maxrentdays'] = (int)$company->maxrentdays;
		unset($company);

		return $calendar;
	}


	/***************************************/
	/* GET NO WORK DATES FOR ALL LOCATIONS */
	/***************************************/
	private function getNoworkDates($rtdata, $lids) {
		$noworkdates = array();
		if (!$lids) { return $noworkdates; }
		if (!is_array($rtdata)) { return $noworkdates; }
		if (count($rtdata) == 0) { return $noworkdates; }

		$now = time();
		$year = gmdate('Y');
		$yearnext = $year + 1;
		
		if (isset($rtdata[0])) {//first pass, apply general settings to all locations
			foreach ($lids as $lid) {
				if (!isset($noworkdates[$lid])) { $noworkdates[$lid] = array(); }
				foreach ($rtdata[0] as $row) {
					if (strpos($row['sdatetime'], '2000') === 0) {
						$edt = str_replace('2000', $year, $row['edatetime']);
						$test_tse = strtotime($edt);
						if ($test_tse < $now) {//past period in current year => make entry for next year
							$sdt = str_replace('2000', $yearnext, $row['sdatetime']);
							$edt = str_replace('2000', $yearnext, $row['edatetime']);
						} else {
							$sdt = str_replace('2000', $year, $row['sdatetime']);
						}
						$sdate = substr($sdt, 0, 10);
						$edate = substr($edt, 0, 10);
					} else {
						$sdate = substr($row['sdatetime'], 0, 10);
						$edate = substr($row['edatetime'], 0, 10);
					}

					$stime = substr($row['sdatetime'], 11, 5);
					$etime = substr($row['edatetime'], 11, 5);

					switch($row['weekdays']) {
						case 'mo-fr': $weekdays = array(1, 2, 3, 4, 5); break;
						case 'sa': $weekdays = array(6); break;
						case 'su': $weekdays = array(0); break;
						case '': default: $weekdays = array(); break;
					}

					$ts_start = strtotime($sdate.' 12:00:00');
					$continue = true;
					$i = 0;

					while ($continue) {
						$ts = $ts_start + ($i * 86400);
						$ymd = gmdate('Y-m-d', $ts);
						if ($weekdays) {
							$weekday = gmdate('w', $ts);
							if (!in_array($weekday, $weekdays)) {
								if ($ymd == $edate) { $continue = false; break; }
								$i++; continue;
							}
						}

						$yz = gmdate('Y', $ts).sprintf("%03d", gmdate('z', $ts));
						if (!isset($noworkdates[$lid][$yz])) {
							$noworkdates[$lid][$yz] = array();
							$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
						} else if (($noworkdates[$lid][$yz][0]['lid'] == 0) && ($lid > 0)) {
							$noworkdates[$lid][$yz] = array();
							$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
						} else if (($stime == '00:00') && ($etime == '23:59')) {
							$noworkdates[$lid][$yz] = array();
							$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
						} else {
							if (($noworkdates[$lid][$yz][0]['start'] == '00:00') && ($noworkdates[$lid][$yz][0]['end'] == '23:59')) {
								//do not add
							} else {
								$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
							}
						}
						if ($ymd == $edate) { $continue = false; break; }
						$i++;
					}
				}
			}
		}

		//second pass, location specific settings
		foreach ($rtdata as $lid => $rows) {
			if ($lid == 0) { continue; }//general settings already passed
			if (!$rows) { continue; }
			if (!isset($noworkdates[$lid])) { continue; }//not existing location!

			foreach ($rows as $row) {
				if (strpos($row['sdatetime'], '2000') === 0) {
					$edt = str_replace('2000', $year, $row['edatetime']);
					$test_tse = strtotime($edt);
					if ($test_tse < $now) {//past period in current year => make entry for next year
						$sdt = str_replace('2000', $yearnext, $row['sdatetime']);
						$edt = str_replace('2000', $yearnext, $row['edatetime']);
					} else {
						$sdt = str_replace('2000', $year, $row['sdatetime']);
					}
					$sdate = substr($sdt, 0, 10);
					$edate = substr($edt, 0, 10);
				} else {
					$sdate = substr($row['sdatetime'], 0, 10);
					$edate = substr($row['edatetime'], 0, 10);
				}

				$stime = substr($row['sdatetime'], 11, 5);
				$etime = substr($row['edatetime'], 11, 5);

				switch($row['weekdays']) {
					case 'mo-fr': $weekdays = array(1, 2, 3, 4, 5); break;
					case 'sa': $weekdays = array(6); break;
					case 'su': $weekdays = array(0); break;
					case '': default: $weekdays = array(); break;
				}

				$ts_start = strtotime($sdate.' 12:00:00');
				$continue = true;
				$i = 0;

				while ($continue) {
					$ts = $ts_start + ($i * 86400);
					$ymd = gmdate('Y-m-d', $ts);
					if ($weekdays) {
						$weekday = gmdate('w', $ts);
						if (!in_array($weekday, $weekdays)) {
							if ($ymd == $edate) { $continue = false; break; }
							$i++; continue;
						}
					}

					$yz = gmdate('Y', $ts).sprintf("%03d", gmdate('z', $ts));
					if (!isset($noworkdates[$lid][$yz])) {
						$noworkdates[$lid][$yz] = array();
						$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
					} else if (($noworkdates[$lid][$yz][0]['lid'] == 0) && ($lid > 0)) {
						$noworkdates[$lid][$yz] = array();
						$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
					} else if (($stime == '00:00') && ($etime == '23:59')) {
						$noworkdates[$lid][$yz] = array();
						$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
					} else {
						if (($noworkdates[$lid][$yz][0]['start'] == '00:00') && ($noworkdates[$lid][$yz][0]['end'] == '23:59')) {
							//do not add
						} else {
							$noworkdates[$lid][$yz][] = array('ts' => $ts, 'start' => $stime, 'end' => $etime, 'lid' => $lid);
						}
					}
					if ($ymd == $edate) { $continue = false; break; }
					$i++;
				}
			}
		}

		return $noworkdates;
	}


	/******************************************/
	/* GET RENTAL DURATIONS FOR ALL LOCATIONS */
	/******************************************/
	private function getRentalDurations($rtdata, $lids) {
		$rentdurations = array();
		if (!$lids) { return $rentdurations; }
		if (!is_array($rtdata)) { return $rentdurations; }
		if (count($rtdata) == 0) { return $rentdurations; }

		$now = time();
		$year = gmdate('Y');
		$yearnext = $year + 1;
		if (isset($rtdata[0])) {//first pass, settings for all locations
			foreach ($lids as $lid) {
				if (!isset($rentdurations[$lid])) { $rentdurations[$lid] = array(); }
				foreach ($rtdata[0] as $row) {
					if (strpos($row['sdate'], '2000') === 0) {
						$edate = str_replace('2000', $year, $row['edate']);
						$test_tse = strtotime($edate.' 00:00:00');
						if ($test_tse < $now) {//past period in current year => make entry for next year
							$sdate = str_replace('2000', $yearnext, $row['sdate']);
							$edate = str_replace('2000', $yearnext, $row['edate']);
						} else {
							$sdate = str_replace('2000', $year, $row['sdate']);
						}
					} else {
						$sdate = substr($row['sdate'], 0, 10);
						$edate = substr($row['edate'], 0, 10);
					}
					$ts_start = strtotime($sdate.' 12:00:00');
					$continue = true;
					$i = 0;
					while ($continue) {
						$ts = $ts_start + ($i * 86400);
						$ymd = gmdate('Y-m-d', $ts);
						$yz = gmdate('Y', $ts).sprintf("%03d", gmdate('z', $ts));
						$rentdurations[$lid][$yz] = array('minrentdays' => $row['minrentdays'], 'maxrentdays' => $row['maxrentdays'], 'lid' => 0);
						if ($ymd == $edate) { $continue = false; break; }
						$i++;
					}
				}
			}
		}

		//second pass, location specific settings
		foreach ($rtdata as $lid => $rows) {
			if ($lid == 0) { continue; }//general settings already passed
			if (!$rows) { continue; }
			if (!isset($rentdurations[$lid])) {
				if (isset($rtdata[0])) {
					continue;//not existing location!
				} else {
					$rentdurations[$lid] = array();
				}
			}

			foreach ($rows as $row) {
				if (strpos($row['sdate'], '2000') === 0) {
					$edate = str_replace('2000', $year, $row['edate']);
					$test_tse = strtotime($edate.' 00:00:00');
					if ($test_tse < $now) {//past period in current year => make entry for next year
						$sdate = str_replace('2000', $yearnext, $row['sdate']);
						$edate = str_replace('2000', $yearnext, $row['edate']);
					} else {
						$sdate = str_replace('2000', $year, $row['sdate']);
					}
				} else {
					$sdate = substr($row['sdate'], 0, 10);
					$edate = substr($row['edate'], 0, 10);
				}
				$ts_start = strtotime($sdate.' 12:00:00');
				$continue = true;
				$i = 0;
				while ($continue) {
					$ts = $ts_start + ($i * 86400);
					$ymd = gmdate('Y-m-d', $ts);
					$yz = gmdate('Y', $ts).sprintf("%03d", gmdate('z', $ts));
					$rentdurations[$lid][$yz] = array('minrentdays' => $row['minrentdays'], 'maxrentdays' => $row['maxrentdays'], 'lid' => $lid);
					if ($ymd == $edate) { $continue = false; break; }
					$i++;
				}
			}
		}

		return $rentdurations;
	}


	/********************/
	/* BOOK (AJAX TASK) */
	/********************/
	private function book() {
		$eLang = $this->lang;

		$pat = "#([\']|[\!]|[\(]|[\)]|[\;]|[\"]|[\$]|[\#]|[\<]|[\>]|[\*]|[\%]|[\~]|[\`]|[\^]|[\|]|[\{]|[\}]|[\\\])#u";
		$pat2 = "#([\']|[\"]|[\$]|[\#]|[\<]|[\>]|[\*]|[\%]|[\~]|[\`]|[\^]|[\|]|[\{]|[\}]|[\\\])#u";

		$params = array();
		$params['ofirstname'] = filter_input(INPUT_POST, 'ofirstname', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		$params['ofirstname'] = trim(preg_replace($pat2, '', $params['ofirstname']));
		$params['olastname'] = filter_input(INPUT_POST, 'olastname', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		$params['olastname'] = trim(preg_replace($pat2, '', $params['olastname']));
		$params['oemail'] = filter_input(INPUT_POST, 'oemail', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		if (isset($_POST['driverage'])) { $params['driverage'] = (int)$_POST['driverage']; }
		if (isset($_POST['licnumber'])) {
			$params['licnumber'] = filter_input(INPUT_POST, 'licnumber', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['licnumber'] = trim(preg_replace($pat, '', $params['licnumber']));
		}
		if (isset($_POST['ocountry'])) {
			$params['ocountry'] = filter_input(INPUT_POST, 'ocountry', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['ocountry'] = trim(preg_replace($pat, '', $params['ocountry']));
		}
		if (isset($_POST['ocity'])) {
			$params['ocity'] = filter_input(INPUT_POST, 'ocity', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['ocity'] = trim(preg_replace($pat2, '', $params['ocity']));
		}
		if (isset($_POST['oaddress'])) {
			$params['oaddress'] = filter_input(INPUT_POST, 'oaddress', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['oaddress'] = trim(preg_replace($pat2, '', $params['oaddress']));
		}
		if (isset($_POST['opostalcode'])) {
			$params['opostalcode'] = filter_input(INPUT_POST, 'opostalcode', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['opostalcode'] = trim(preg_replace($pat2, '', $params['opostalcode']));
		}
		if (isset($_POST['ophone'])) {
			$params['ophone'] = filter_input(INPUT_POST, 'ophone', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['ophone'] = trim(preg_replace($pat2, '', $params['ophone']));
		}
		if (isset($_POST['omobile'])) {
			$params['omobile'] = filter_input(INPUT_POST, 'omobile', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['omobile'] = trim(preg_replace($pat2, '', $params['omobile']));
		}
		if (isset($_POST['flight'])) {
			$params['flight'] = filter_input(INPUT_POST, 'flight', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['flight'] = trim(preg_replace($pat2, '', $params['flight']));
		}
		if (isset($_POST['ship'])) {
			$params['ship'] = filter_input(INPUT_POST, 'ship', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['ship'] = trim(preg_replace($pat2, '', $params['ship']));
		}
		if (isset($_POST['hotel'])) {
			$params['hotel'] = filter_input(INPUT_POST, 'hotel', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['hotel'] = trim(preg_replace($pat2, '', $params['hotel']));
		}
		if (isset($_POST['extras'])) {
			$params['extras'] = filter_input(INPUT_POST, 'extras', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
			$params['extras'] = trim(preg_replace($pat, '', $params['extras']));
		}
		if (isset($_POST['ocomments'])) {
			$params['ocomments'] = strip_tags(filter_input(INPUT_POST, 'ocomments', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW));
			if ($params['ocomments'] != '') { $params['ocomments'] = trim(preg_replace($pat2, '', $params['ocomments'])); }
		}

		for ($i=1; $i < 9; $i++) {
			$idx = 'ccustom'.$i;
			if (isset($_POST[$idx])) {
				$params[$idx] = filter_input(INPUT_POST, $idx, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
				$params[$idx] = trim(preg_replace($pat2, '', $params[$idx]));
			}
		}

		$params['invoice'] = (isset($_POST['invoice'])) ? (int)$_POST['invoice'] : 0;
		if ($params['invoice'] == 1) {
			$params['invtitle'] = filter_input(INPUT_POST, 'invtitle', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invtitle'] = trim(preg_replace($pat2, '', $params['invtitle']));
			$params['invact'] = filter_input(INPUT_POST, 'invact', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invact'] = trim(preg_replace($pat2, '', $params['invact']));
			$params['invaddress'] = filter_input(INPUT_POST, 'invaddress', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invaddress'] = trim(preg_replace($pat2, '', $params['invaddress']));
			$params['invcity'] = filter_input(INPUT_POST, 'invcity', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invcity'] = trim(preg_replace($pat2, '', $params['invcity']));
			$params['invirs'] = filter_input(INPUT_POST, 'invirs', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invirs'] = trim(preg_replace($pat2, '', $params['invirs']));
			$params['invafm'] = filter_input(INPUT_POST, 'invafm', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
			$params['invafm'] = trim(preg_replace($pat2, '', $params['invafm']));
		}

		if (isset($_POST['pid'])) { $params['pid'] = (int)$_POST['pid']; }
		if (isset($_POST['paymod'])) {
			$params['paymod'] = filter_input(INPUT_POST, 'paymod', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
			$params['paymod'] = trim(preg_replace($pat, '', $params['paymod']));
			if (isset($_POST['ccowner']) && isset($_POST['cctype']) && isset($_POST['ccnumber']) && isset($_POST['ccseccode']) && isset($_POST['ccexpire'])) {//creditcardoff / cashondelivery / other
				$params['ccowner'] = filter_input(INPUT_POST, 'ccowner', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
				$params['ccowner'] = trim(preg_replace($pat2, '', $params['ccowner']));
				$params['cctype'] = filter_input(INPUT_POST, 'cctype', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
				$params['cctype'] = trim(preg_replace($pat, '', $params['cctype']));
				$params['ccnumber'] = filter_input(INPUT_POST, 'ccnumber', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
				$params['ccnumber'] = trim(preg_replace($pat, '', $params['ccnumber']));
				$params['ccseccode'] = filter_input(INPUT_POST, 'ccseccode', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
				$params['ccseccode'] = trim(preg_replace($pat, '', $params['ccseccode']));
				$params['ccexpire'] = filter_input(INPUT_POST, 'ccexpire', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
				$params['ccexpire'] = trim(preg_replace($pat2, '', $params['ccexpire']));
			}
		}

		$params['couponcode'] = filter_input(INPUT_POST, 'couponcode', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		$params['couponcode'] = trim(preg_replace($pat, '', $params['couponcode']));
		$params['pickup'] = filter_input(INPUT_POST, 'pickup', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		$params['pickup'] = trim(preg_replace($pat2, '', $params['pickup']));
		$params['dropoff'] = filter_input(INPUT_POST, 'dropoff', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		$params['dropoff'] = trim(preg_replace($pat2, '', $params['dropoff']));
		$params['picklid'] = (isset($_POST['picklid'])) ? (int)$_POST['picklid'] : 0;
		$params['droplid'] = (isset($_POST['droplid'])) ? (int)$_POST['droplid'] : 0;
		$params['vtype'] = filter_input(INPUT_POST, 'vtype', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		$params['vtype'] = trim(preg_replace($pat2, '', $params['vtype']));
		$params['gid'] = (isset($_POST['gid'])) ? (int)$_POST['gid'] : 0;
		$params['mid'] = (isset($_POST['mid'])) ? (int)$_POST['mid'] : 0;
		$params['usecash'] = (isset($_POST['usecash'])) ? (int)$_POST['usecash'] : 0;
		$params['usefullamount'] = (isset($_POST['usefullamount'])) ? (int)$_POST['usefullamount'] : 0;

		$ipaddress = '';
		if (isset($_POST['clip'])) { $ipaddress = trim($_POST['clip']); }
		if (($ipaddress != '') && filter_var($ipaddress, FILTER_VALIDATE_IP)) {
			$params['ipaddress'] = $ipaddress;
		} else if (isset($_SERVER['HTTP_X_FORWARDED_FOR']) && ($_SERVER['HTTP_X_FORWARTDED_FOR'] != '')) {
			$params['ipaddress'] = $_SERVER['HTTP_X_FORWARDED_FOR'];
		} else if (isset($_SERVER['REMOTE_ADDR'])) {
			$params['ipaddress'] = $_SERVER['REMOTE_ADDR'];
		} else {
			$params['ipaddress'] = '';
		}

		//quick validation (full validation is performed on API server)
		$errormsg = '';
		if ($params['picklid'] < 1) {
			$errormsg = $eLang->get('PLEASE_SEL_PICKLOC');
		} else if ($params['droplid'] < 1) {
			$errormsg = $eLang->get('PLEASE_SEL_DROPLOC');
		} else if (($params['pickup'] == '') || (strlen($params['pickup']) != 19)) {
			$errormsg = $eLang->get('INVALID_PICKDATE');
		} else if (($params['dropoff'] == '') || (strlen($params['dropoff']) != 19)) {
			$errormsg = $eLang->get('INVALID_DROPDATE');
		} else if (($params['gid'] < 1) || ($params['mid'] < 1) || ($params['vtype'] == '')) {
			$errormsg = 'No vehicle selected!';
		} else if ($params['ofirstname'] == '') {
			$errormsg = sprintf($eLang->get('FIELD_NOEMPTY'), $eLang->get('FIRSTNAME'));
		} else if ($params['olastname'] == '') {
			$errormsg = sprintf($eLang->get('FIELD_NOEMPTY'), $eLang->get('LASTNAME'));
		} else if (($params['oemail'] == '') || !filter_var($params['oemail'], FILTER_VALIDATE_EMAIL)) {
			$errormsg = $eLang->get('INV_EMAIL');
		}

		$response = array('success' => 0, 'message' => '', 'processor' => 0, 'sucurl' => '');
		if ($errormsg != '') {
			$response['message'] = $errormsg;
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params['apiid'] = $this->options['apiid'];
		$params['currency'] = $this->options['currency'];
		$params['type'] = 'XML';
		$params['ts'] = $ts;
		$params['enc'] = $enc;
		$params['ubase'] = isset($_POST['ubase']) ? trim($_POST['ubase']) : '';//this is base64 encoded booking form URL. Used for Alpha Bank e-Commerce or other pay method

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/book';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/book';
		}

		$xml = $this->helper->rcpost($taskurl, $params);
		if ($this->helper->getError() != '') {//RC connection failed
			$response['message'] = $this->helper->getError();
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$results = $this->helper->parseBookResults($xml);
		if (!$results || ($this->helper->getError() != '')) {
			$response['message'] = $this->helper->getError();
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$fname = strtolower($results['rescode']);
		$this->helper->saveToCache($fname, $results, false, false, true);

		$response['success'] = 1;
		if (($results['feeamount'] > 0) && ($results['paymoduledata']['page'] == 1) && ($results['paymoduledata']['formaction'] != '') && ($results['paymoduledata']['formitems'])) {
			$response['processor'] = 1;
			$response['formaction'] = $results['paymoduledata']['formaction'];
			$response['formitems'] = $results['paymoduledata']['formitems'];
			$encoded = json_encode($response);
			$this->pageHeaders('application/json');
			echo $encoded;
			exit;
		}

		$response['sucurl'] = $results['formurl'].'?step=3&rescode='.urlencode($results['rescode']);

		$encoded = json_encode($response);
		$this->pageHeaders('application/json');
		echo $encoded;
		exit;
	}


	/***********************/
	/* GET CURRENT URL URI */
	/***********************/
	private function getURI() {
		if (isset($_SERVER['REQUEST_URI'])) {
			$uri = ltrim(filter_input(INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_URL), '/');
			if ($uri == '') { //php 5.2 bug or frontpage, retry to make sure value is ok
				$uri = ltrim($_SERVER['REQUEST_URI'], '/');
			}
			return $uri;
		}
		if (isset($_SERVER['QUERY_STRING'])) {
			$query_str = filter_input(INPUT_SERVER, 'QUERY_STRING', FILTER_SANITIZE_URL);
		} else if (@getenv('QUERY_STRING')) {
			$query_str = filter_input(INPUT_ENV, 'QUERY_STRING', FILTER_SANITIZE_URL);
		} else {
			$query_str = '';
		}
		if ($query_str != '') { $query_str = '?'.$query_str; }
		if (isset($_SERVER['PATH_INFO'])) {
			$uri = ltrim(filter_input(INPUT_SERVER, 'PATH_INFO', FILTER_SANITIZE_URL), '/');
			return $uri.$query_str;
		} elseif (@getenv('PATH_INFO')) {
			$uri = ltrim(filter_input(INPUT_ENV, 'PATH_INFO', FILTER_SANITIZE_URL), '/');
			return $uri.$query_str;
		}
		if (isset($_SERVER['PHP_SELF'])) {
			$uri = ltrim(filter_input(INPUT_SERVER, 'PHP_SELF', FILTER_SANITIZE_URL), '/');
			return $uri.$query_str;
		} elseif (@getenv('PHP_SELF')) {
			$uri = ltrim(filter_input(INPUT_ENV, 'PHP_SELF', FILTER_SANITIZE_URL), '/');
			return $uri.$query_str;
		} else {
			return $query_str;
		}
	}


	/*************************/
	/* DETECT URL SSL ON/OFF */
	/*************************/
	private function detectSSL() {
		if (isset($_SERVER['HTTPS'])) {
			if (($_SERVER['HTTPS'] == 'on') || ($_SERVER['HTTPS'] == 1)) { return true; }
		}
		if (isset($_SERVER['SERVER_PORT']) && ($_SERVER['SERVER_PORT'] == 443)) { return true; }
		if (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && (strtolower($_SERVER['HTTP_X_FORWARDED_PROTO']) == 'https')) { return true; }
		return false;
	}


	/****************/
	/* GET URL HOST */
	/****************/
	private function getHost() {
		if ($this->detectSSL() === true) {
			return 'https://'.$_SERVER['HTTP_HOST'];
		} else {
			return 'http://'.$_SERVER['HTTP_HOST'];
		}
	}


	private function makeFormURL() {
		$url = $this->host.'/'.$this->uri;
		$n = strpos($url, '?');
		if ($n !== false) { $url = substr($url, 0, $n); } //no query string => SEO URLs only!
		return $url;
	}


	private function getFromRequest($var, $defvalue) {
		if (isset($_POST[$var])) { return $_POST[$var]; }
		if (isset($_GET[$var])) { return $_GET[$var]; }
		return $defvalue;
	}

 
	/*************************************************/
	/* GET/VALIDATE AVAILABILITY FORM SUBMITTED DATA */
	/*************************************************/
	private function getValAVFdata($eLang, $date_userformat=true) {
		$data = array('ok' => false, 'errormsg' => '');

		$data['picklid'] = (int)$this->getFromRequest('picklid', 0);
		$data['pickhour'] = (int)$this->getFromRequest('pickhour', 0);
		$data['pickmin'] = (int)$this->getFromRequest('pickmin', 0);
		$data['pickdate'] = trim($this->getFromRequest('pickdate', ''));

		$data['droplid'] = (int)$this->getFromRequest('droplid', 0);
		$data['drophour'] = (int)$this->getFromRequest('drophour', 0);
		$data['dropmin'] = (int)$this->getFromRequest('dropmin', 0);
		$data['dropdate'] = trim($this->getFromRequest('dropdate', ''));

		$data['vtype'] = trim($this->getFromRequest('vtype', ''));
		$data['mid'] = (int)$this->getFromRequest('mid', 0);//from fleet page
		$data['gid'] = (int)$this->getFromRequest('gid', 0);//from fleet page

		//if standard full dates have been provided use these ones
		$pickup = trim($this->getFromRequest('pickup', ''));
		$dropoff = trim($this->getFromRequest('dropoff', ''));
		if (($pickup != '') || ($dropoff != '') && (strlen($pickup) == 19) && (strlen($dropoff) == 19)) {
			$date_userformat = false;
			$data['pickdate'] = substr($pickup, 0, 10);
			$data['pickhour'] = intval(substr($pickup, 11, 2));
			$data['pickmin'] = intval(substr($pickup, 14, 2));
			$data['dropdate'] = substr($dropoff, 0, 10);
			$data['drophour'] = intval(substr($dropoff, 11, 2));
			$data['dropmin'] = intval(substr($dropoff, 14, 2));
		}

		if ($data['picklid'] < 1) { $data['errormsg'] = $eLang->get('PLEASE_SEL_PICKLOC'); return $data; }
		if ($data['droplid'] < 1) { $data['errormsg'] = $eLang->get('PLEASE_SEL_DROPLOC'); return $data; }
		if (($data['pickdate'] == '') || (strlen($data['pickdate']) != 10)) { $data['errormsg'] = $eLang->get('PLEASE_SEL_PICKDATE'); return $data; }
		if (($data['dropdate'] == '') || (strlen($data['dropdate']) != 10)) { $data['errormsg'] = $eLang->get('PLEASE_SEL_DROPDATE'); return $data; }

		if (($data['pickhour'] < 0) || ($data['pickhour'] > 23) || ($data['pickmin'] < 0) || ($data['pickmin'] > 59)) {
			$data['errormsg'] = $eLang->get('PLEASE_SEL_PICKDATE'); return $data;
		}
		if (($data['drophour'] < 0) || ($data['drophour'] > 23) || ($data['dropmin'] < 0) || ($data['dropmin'] > 59)) {
			$data['errormsg'] = $eLang->get('PLEASE_SEL_DROPDATE'); return $data;
		}

		if ($date_userformat) {//DD-MM-YYYY
			$pday = intval(substr($data['pickdate'], 0, 2));
			$pmonth = intval(substr($data['pickdate'], 3, 2));
			$pyear = intval(substr($data['pickdate'], 6, 4));
			$dday = intval(substr($data['dropdate'], 0, 2));
			$dmonth = intval(substr($data['dropdate'], 3, 2));
			$dyear = intval(substr($data['dropdate'], 6, 4));
		} else {//YYYY-MM-DD
			$pyear = intval(substr($data['pickdate'], 0, 4));
			$pmonth = intval(substr($data['pickdate'], 5, 2));
			$pday = intval(substr($data['pickdate'], 8, 2));
			$dyear = intval(substr($data['dropdate'], 0, 4));
			$dmonth = intval(substr($data['dropdate'], 5, 2));
			$dday = intval(substr($data['dropdate'], 8, 2));
		}

		$pok = false;
		if (($pyear > 2017) && ($pmonth > 0) && ($pmonth < 13) && ($pday > 0) && ($pday < 32)) {
			if (checkdate($pmonth, $pday, $pyear)) { $pok = true; }
		}
		if (!$pok) { $data['errormsg'] = $eLang->get('INVALID_PICKDATE'); return $data; }
		$dok = false;
		if (($dyear > 2017) && ($dmonth > 0) && ($dmonth < 13) && ($dday > 0) && ($dday < 32)) {
			if (checkdate($dmonth, $dday, $dyear)) { $dok = true; }
		}
		if (!$dok) { $data['errormsg'] = $eLang->get('INVALID_DROPDATE'); return $data; }

		$data['pickdate'] = $pyear.'-'.sprintf("%02d", $pmonth).'-'.sprintf("%02d", $pday);//always to YYYY-MM-DD format
		$data['dropdate'] = $dyear.'-'.sprintf("%02d", $dmonth).'-'.sprintf("%02d", $dday);//always to YYYY-MM-DD format
		$data['pickhour'] = sprintf("%02d", $data['pickhour']);
		$data['pickmin'] = sprintf("%02d", $data['pickmin']);
		$data['drophour'] = sprintf("%02d", $data['drophour']);
		$data['dropmin'] = sprintf("%02d", $data['dropmin']);

		$pickdt = new DateTime($data['pickdate'].' '.$data['pickhour'].':'.$data['pickmin'].':00');
		$dropdt = new DateTime($data['dropdate'].' '.$data['drophour'].':'.$data['dropmin'].':00');
		if ($pickdt >= $dropdt) { $data['errormsg'] = $eLang->get('INVALID_DROPDATE'); return $data; }

		if ($data['vtype'] != '') {
			if (strpos($data['vtype'], ',') !== false) {//multiple vtypes, comma separated, example: moto,scooter,atv
				$reqvtypes = explode(',', $data['vtype']);
			} else {
				$reqvtypes = array($data['vtype']);
			}
			foreach ($reqvtypes as $vt) {
				if (!in_array($vt, array('car', 'moto', 'scooter', 'boat', 'atv', 'buggy', 'bicycle', 'other'))) {
					$data['errormsg'] = 'Invalid vehicle type!';
					break;
				}
			}
			if ($data['errormsg'] != '') { return $data; }
		} else if (($data['vtype'] == '') && ($this->options['vtype'] != '')) {
			$data['vtype'] = $this->options['vtype'];
		}

		$data['ok'] = true;
		return $data;
	}


	/*******************/
	/* FORMAT DATETIME */
	/*******************/
	private function formatDate($datetime, $format) {
		$ts = strtotime($datetime);
		switch (strtoupper(substr(PHP_OS, 0, 3))) {
			case 'WIN':
				$format = str_replace('%e', '%d', $format);
			break;
			case 'MAC':
				$format = str_replace('%P', '%p', $format);
			break;
			default: break;
		}

		return ($this->lang->guilang == 'en') ? strftime($format, $ts) : $this->lang->i18n_strftime($format, $ts);
	}


	/*************************************/
	/* GET/VALIDATE MODEL SELECTION LINK */
	/*************************************/
	private function getValModelseldata($eLang) {
		$data = array('ok' => false, 'errormsg' => '');

		$data['picklid'] = isset($_GET['picklid']) ? (int)$_GET['picklid'] : 0;
		$data['droplid'] = isset($_GET['droplid']) ? (int)$_GET['droplid'] : 0;
		$data['mid'] = isset($_GET['mid']) ? (int)$_GET['mid'] : 0;
		$data['pickup'] = isset($_GET['pickup']) ? trim($_GET['pickup']) : '';
		if ($data['pickup'] != '') { $data['pickup'] = urldecode($data['pickup']); }
		$data['dropoff'] = isset($_GET['dropoff']) ? trim($_GET['dropoff']) : '';
		if ($data['dropoff'] != '') { $data['dropoff'] = urldecode($data['dropoff']); }

		if ($data['picklid'] < 1) { $data['errormsg'] = $eLang->get('PLEASE_SEL_PICKLOC'); return $data; }
		if ($data['droplid'] < 1) { $data['errormsg'] = $eLang->get('PLEASE_SEL_DROPLOC'); return $data; }
		if ($data['mid'] < 1) { $data['errormsg'] = $eLang->get('PLEASE_SEL_VEHICLE'); return $data; }
		if (($data['pickup'] == '') || (strlen($data['pickup']) != 19)) { $data['errormsg'] = $eLang->get('PLEASE_SEL_PICKDATE'); return $data; }
		if (($data['dropoff'] == '') || (strlen($data['dropoff']) != 19)) { $data['errormsg'] = $eLang->get('PLEASE_SEL_DROPDATE'); return $data; }

		$data['ok'] = true;
		return $data;
	}


	/************************/
	/* GENERATE FOOTER HTML */
	/************************/
	private function makeFooter($eLang, $company) {
		$footer = "\n".'<script>tippy(\'.iosrt_wtooltip\', { arrow: true })</script>'."\n";

		if ($company->rcancel == 1) {//cancel/change reservation modal box
			$footer .= '<div class="iosrt_modal" id="iosrt_modal77"><div class="iosrt_modalcon" id="iosrt_modalcon77">';
			$footer .= '<div class="iosrt_modalhead"><a href="javascript:void(null);" onclick="iosRTModalClose(77);" title="'.$eLang->get('CLOSE').'">x</a><h4>'.$eLang->get('CANCEL_CHANGE_RESERVATION').'</h4></div>'."\n";
			$footer .= '<div class="iosrt_modalbody" id="iosrt_modalbody77">'."\n";
			$footer .= '<form name="fmrtreschange" id="fmrtreschange" method="post" action="'.$this->options['clienturl'].'/ajax.php">'."\n";
			$msg = $eLang->get('CHANGE_RESERVATION_TIP');
			if ($company->rcanceldays > 0) { $msg .= ' '.sprintf($eLang->get('NOCANCEL_RESERVATION_DAYS'), $company->rcanceldays); }
			$footer .= '<div class="iosrt_formrow"><div class="iosrt_tip">'.$msg."</div></div>\n";
			$footer .= '<div class="iosrt_invisible" id="iosrt_modalmessage77"></div>'."\n";
			$footer .= '<div class="iosrt_formrow">';
			$footer .= '<label class="iosrt_label" for="iosrt_crsubject">'.$eLang->get('SUBJECT')."</label>\n";
			$footer .= '<div class="iosrt_labelside"><select name="subject" id="iosrt_crsubject" class="iosrt_select">'."\n";
			$footer .= '<option value="0" selected="selected">'.$eLang->get('INFORMATION')."</option>\n";
			$footer .= '<option value="1">'.$eLang->get('CHANGE_RESERVATION')."</option>\n";
			$footer .= '<option value="2">'.$eLang->get('CANCEL_RESERVATION')."</option>\n";
			$footer .= "</select></div>\n";
			$footer .= "</div>\n";
			$footer .= '<div class="iosrt_formrow">';
			$footer .= '<label class="iosrt_label" for="iosrt_crrescode">'.$eLang->get('RESERVATION_CODE')."</label>\n";
			$footer .= '<div class="iosrt_labelside"><input type="text" name="rescode" id="iosrt_crrescode" class="iosrt_text" value="" placeholder="'.$eLang->get('RESERVATION_CODE').'" required="required" dir="ltr" /></div>'."\n";
			$footer .= "</div>\n";
			$footer .= '<div class="iosrt_formrow">';
			$footer .= '<label class="iosrt_label" for="iosrt_cremail">'.$eLang->get('EMAIL')."</label>\n";
			$footer .= '<div class="iosrt_labelside"><input type="email" name="email" id="iosrt_cremail" class="iosrt_text iosrt_iemail" value="" placeholder="'.$eLang->get('EMAIL').'" required="required" dir="ltr" /></div>'."\n";
			$footer .= "</div>\n";
			$footer .= '<div class="iosrt_formrow">';
			$footer .= '<label class="iosrt_label" for="iosrt_crcomments">'.$eLang->get('COMMENTS')."</label>\n";
			$footer .= '<div class="iosrt_labelside"><textarea name="comments" id="iosrt_crcomments" class="iosrt_textarea" placeholder="'.$eLang->get('COMMENTS').'"></textarea></div>'."\n";
			$footer .= "</div>\n";
			//$footer .= '<input type="hidden" name="apiid" id="iosrt_crapiid" value="'.$this->options['apiid'].'" />'."\n";
			$footer .= '<div class="iosrt_formrow">'."\n";
			$footer .= '<button type="button" class="iosrt_btn" name="subbtn" id="iosrt_crsubbtn" onclick="iosRTChangeReservation();">'.$eLang->get('SUBMIT_REQUEST')."</button>\n";
			$footer .= "</div>\n";
			$footer .= "</form>\n";
			$footer .= "</div>\n";//iosrt_modalbody
			$footer .= '</div></div>'."\n";
		}

		$footer .= '<div id="iosrt_condata" class="iosrt_invisible" data-guilang="'.$this->options['guilang'].'" data-lang="'.$this->options['lang'].'" data-currency="'.$this->options['currency'].'" data-apiid="'.$this->options['apiid'].'"></div>'."\n";
		$footer .= '<div class="iosrt_pgloading" id="iosrt_pgloading"><div class="iosrt_pgloadingcon"><div class="iosrt_pgloadingicon">&#160;</div><div class="iosrt_pgloadingtext">'.$eLang->get('PLEASE_WAIT').'</div></div></div>'."\n";
		$footer .= '<!-- End of IOSR Rentals API client integration. Copyright Ioannis Sannos https://www.isopensource.com -->'."\n";
		return $footer;
	}


	/**************************/
	/* DETECT VISITOR COUNTRY */
	/**************************/
	private function detectVisitorCountry($countries, $default='GB') {
		if ($default == '') { $default = 'GB'; }
		if (!isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) { return $default; }
		if (trim($_SERVER['HTTP_ACCEPT_LANGUAGE']) == '') { return $default; }

		$res = array();
		$aceptlangs = explode(',', str_replace(array(';','0','1','2','3','4','5','6','7','8','9','.',"q="), array(',','','','','','','','','','','','',""), $_SERVER['HTTP_ACCEPT_LANGUAGE']));
		foreach($aceptlangs as $i => $v) {
			if (trim($v) != '') { $res[] = trim($v); }
		}
		if (!$res) { return $default; }

		$country = '';
		foreach ($res as $lc) {
			$n = strpos($lc, '-');
			if ($n !== false) {
				$c = strtoupper(substr($lc, $n + 1));
				if (isset($countries[$c])) { $country = $c; break; }
				$l = strtolower(substr($lc, 0, $n));
			} else {
				$l = strtolower($lc);
			}
			switch ($l) {
				case 'af': $country = 'ZA'; break;
				case 'ar': $country = 'AE'; break;
				case 'az': $country = 'AZ'; break;
				case 'bg': $country = 'BG'; break;
				case 'bn': $country = 'BD'; break;
				case 'bs': $country = 'BA'; break;
				case 'cs': $country = 'CZ'; break;
				case 'da': $country = 'DK'; break;
				case 'de': $country = 'DE'; break;
				case 'el': $country = 'GR'; break;
				case 'en': $country = 'GB'; break;
				case 'es': $country = 'ES'; break;
				case 'fa': $country = 'IR'; break;
				case 'fi': $country = 'FI'; break;
				case 'he': $country = 'IL'; break;
				case 'hr': $country = 'HR'; break;
				case 'hu': $country = 'HU'; break;
				case 'hy': $country = 'AM'; break;
				case 'id': $country = 'ID'; break;
				case 'it': $country = 'IT'; break;
				case 'ja': $country = 'JP'; break;
				case 'ka': $country = 'GE'; break;
				case 'ki': case 'pu': case 'sa': case 'sd': case 'ta': case 'te': $country = 'IN'; break;
				case 'kk': $country = 'KZ'; break;
				case 'ko': $country = 'KR'; break;
				case 'lv': $country = 'LV'; break;
				case 'lt': $country = 'LT'; break;
				case 'nl': $country = 'NL'; break;
				case 'nn': case 'no': $country = 'NO'; break;
				case 'ph': $country = 'AF'; break;
				case 'pl': $country = 'PL'; break;
				case 'pt': $country = 'PT'; break;
				case 'ro': $country = 'RO'; break;
				case 'ru': $country = 'RU'; break;
				case 'rs': $country = 'RS'; break;
				case 'sk': $country = 'SK'; break;
				case 'sl': $country = 'SI'; break;
				case 'sq': $country = 'AL'; break;
				case 'sv': $country = 'SE'; break;
				case 'tr': $country = 'TR'; break;
				case 'uk': $country = 'UA'; break;
				case 'ug': case 'zh': case 'zt': $country = 'CN'; break;
				case 'ur': $country = 'PK'; break;
				case 'vi': $country = 'VN'; break;
				default: break;
			}
			if ($country != '') { break; }
		}

	    return ($country == '') ? $default : $country;
	}


	/*********************/
	/* ECHO PAGE HEADERS */
	/*********************/
	private function pageHeaders($type='text/plain') {
		if (ob_get_length() > 0) { ob_end_clean(); }
		header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
		header('Last-Modified: '.gmdate('D, d M Y H:i:s').'GMT');
		header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
		header('Cache-Control: post-check=0, pre-check=0', false);
		header('Pragma: no-cache');
		header('Content-type: '.$type.'; charset=utf-8');
	}


	/**************************/
	/* GET LAST ERROR MESSAGE */
	/**************************/
	public function getError() {
		return $this->errormsg;
	}


	/********************************/
	/* GET API CONFIGURATION OPTION */
	/********************************/
	public function getOption($key, $defval) {
		if (isset($this->options[$key])) { return $this->options[$key]; }
		return $defval;
	}


	/***************************************/
	/* GET LANGUAGE STRING FROM LANG CLASS */
	/***************************************/
	public function getLang($key) {
		if (!$this->lang) { return $key; }
		return $this->lang->get($key);
	}


	/*****************************************/
	/* GET GUI LANGUAGE LOADED BY LANG CLASS */
	/*****************************************/
	public function getGUILang() {
		if (!$this->lang) { return ''; }
		return $this->lang->guilang;
	}


	/**************/
	/* GET HELPER */
	/**************/
	public function getHelper() {
		return $this->helper;
	}


	//------------------------ FLEET --------------------------//


	/***********************/
	/* LIST FLEET VEHICLES */
	/***********************/
	private function listFleet() {
		$eLang = $this->lang;

		$this->setHead();

		$company = $this->helper->basicRequest('company', true);
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}

		$noavail_lng = ($this->options['vtype'] == 'boat') ? $eLang->get('NO_AVAILABLE_BOATS') : $eLang->get('NO_AVAILABLE_VEHICLES');

		$results = $this->helper->fleetRequest($this->options['vtype']);
		if ($results === false) {
			$this->errormsg = $this->helper->getError();
			$errormsg = ($this->errormsg == '') ? $noavail_lng : $this->errormsg;
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$errormsg."</p></div>\n";
		}

		if (!$results->models) {
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$noavail_lng."</p></div>\n";
		}

		$places = $this->helper->basicRequest('places', true);

		require_once($this->path.'/currency.class.php');
		$currency = new iosrtApiCurrency($results->currency, $eLang->get('THOUSANDS_SEP'), $eLang->get('DECIMALS_SEP'), $eLang->get('CURRENCYFRONT'));

		if ($this->options['vtype'] == 'car') {
			$title = $eLang->get('CARS_FLEET');
		} else if ($this->options['vtype'] == 'moto') {
			$title = $eLang->get('MOTORCYCLES');
		} else {
			$title = $eLang->get('FLEET');
		}

		$modelsfeatures = $this->helper->basicRequest('modelsfeatures');
		$priceincludes = $this->helper->basicRequest('priceincludes');
		$categories = $this->helper->basicRequest('categories');

		$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";
		if ($this->options['platform'] == 'standalone') { $html .= '<h1 class="iosrt_h1">'.$title.'</h1>'; }

		$html .= $this->fleetFilters($results->models, $categories, $modelsfeatures, $eLang);

		$all_mids = array();
		$html .= '<div class="iosrt_fleet">'."\n";
		foreach ($results->models as $mid => $model) {
			$all_mids[] = $mid;
			$html .= $this->makeFleetModelHTML($mid, $model, $modelsfeatures, $priceincludes, $categories, $places, $company, $eLang, $currency);
		}
		$html .= "</div>\n";

		unset($modelsfeatures, $priceincludes, $categories);

		$html .= '<div class="iosrt_invisible" id="iosrt_novehmatch_filters">'.$eLang->get('NOVEH_MATCH_FILTERS').'</div>'."\n";
		$html .= '<div class="iosrt_invisible" id="iosrt_all_models" data-mids="'.implode(';', $all_mids).'"></div>'."\n";

		$html .= $this->makeFooter($eLang, $company);

		if ($this->hasgallery) {
			$html .= "<script>\n";
			$html .= ' var iosRTSwiper = new Swiper (\'.iosrt_swiper\', {'."\n";
			$html .= 'loop: true, pagination: { el: \'.swiper-pagination\' }, navigation: { nextEl: \'.swiper-button-next\', prevEl: \'.swiper-button-prev\'}'."\n";
			$html .= "})\n";
			$html .= "</script>\n";
		}
		return $html;
	}


	/***********************/
	/* DISPLAY FLEET MODEL */
	/***********************/
	private function fleetModel() {
		$eLang = $this->lang;

		$this->setHead();

		$company = $this->helper->basicRequest('company', true);
		if ($company === false) {
			$this->errormsg = $this->helper->getError();
			$msg = ($this->errormsg != '') ? $this->errormsg : 'Could not load company basic data!';
			return '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('LOADFAIL_ERRMSGFOL').'<br />'.$this->errormsg."</p></div>\n";
		}

		$mid = (int)$this->getFromRequest('mid', 0);
		$cache_hash = trim($this->getFromRequest('ch', ''));
		if ($mid < 1) {
			$html = '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$eLang->get('VEH_NOT_FOUND')."</p></div>\n";
			$html .= '<div class="iosrt_vlspace"><a href="'.$this->makeFormURL().'" class="iosrt_fleet_back">'.$eLang->get('BACK_FLEET_PAGE')."</a></div>\n";
			return $html;
		}

		$fleetpage = '';
		if (trim($this->options['fleetpage']) != '') {
			if ($this->options['platform'] == 'elxis') {
				if (stripos($this->options['fleetpage'], 'http') === false) {//Elxis URI
					$fleetpage = eFactory::getElxis()->makeURL($this->options['fleetpage']);
				} else {
					$fleetpage = $this->options['fleetpage'];
				}
			} else {
				$fleetpage = $this->options['fleetpage'];
			}
			if (!filter_var($fleetpage, FILTER_VALIDATE_URL)) { $fleetpage = ''; }
		}
		if ($fleetpage == '') { $fleetpage = $this->makeFormURL(); }

		$results = $this->helper->fleetModelRequest($mid, $cache_hash);
		if ($results === false) {
			$this->errormsg = $this->helper->getError();
			$errormsg = ($this->errormsg == '') ? $eLang->get('VEH_NOT_FOUND') : $this->errormsg;
			$html = '<div class="iosrt_error"><h4>'.$eLang->get('ERROR').'</h4><p>'.$errormsg."</p></div>\n";
			$html .= '<div class="iosrt_vlspace"><a href="'.$fleetpage.'" class="iosrt_fleet_back">'.$eLang->get('BACK_FLEET_PAGE')."</a></div>\n";
			return $html;
		}

		$places = $this->helper->basicRequest('places', true);

		require_once($this->path.'/currency.class.php');
		$currency = new iosrtApiCurrency($results->currency, $eLang->get('THOUSANDS_SEP'), $eLang->get('DECIMALS_SEP'), $eLang->get('CURRENCYFRONT'));

		if ($this->options['vtype'] == 'car') {
			$title = $eLang->get('CARS_FLEET');
		} else if ($this->options['vtype'] == 'moto') {
			$title = $eLang->get('MOTORCYCLES');
		} else {
			$title = $eLang->get('FLEET');
		}

		$modelsfeatures = $this->helper->basicRequest('modelsfeatures');
		$priceincludes = $this->helper->basicRequest('priceincludes');
		$categories = $this->helper->basicRequest('categories');

		$paramsdata = $this->makeModelParamsHTML($results->model['modelparams'], $results->model['vtype'], $modelsfeatures, $eLang, 50, true);

		$images = array();
		if ($results->model['image'] != '') { $images[] = $company->imagesurl.'large/'.$results->model['image']; }
		if (count($results->model['images']) > 0) {
			foreach ($results->model['images'] as $img) { $images[] = $company->imagesurl.'large/'.$img; }
		}
		if (!$images) { $images[] = $this->options['clienturl'].'/images/noimage.png'; }

		$html = "\n".'<!-- Start of IOSR Rentals API client integration -->'."\n";

		$html .= '<div class="iosrt_vehicle_page" itemprop="itemOffered" itemscope itemtype="https://schema.org/Car">'."\n";
		$html .= '<div class="iosrt_veh_top">'."\n";

		$propstr = ($results->model['description'] == '') ? ' itemprop="description"' : '';
		$html .= '<h2 class="iosrt_veh_title" itemprop="name">'.$results->model['title']."</h2>\n";
		$html .= '<div class="iosrt_veh_subtitle"'.$propstr.'>'.$this->makeModelSubtitle($results->model['group'], $results->model['vtype'], $results->model['categories'], $categories)."</div>\n";

		$html .= '<div class="iosrt_veh_topin">'."\n";

		$html .= '<div class="iosrt_veh_topside">'."\n";
		if (trim($this->options['formaction']) != '') {
			$vehdata = array(
				'mid' => $results->model['mid'],
				'gid' => $results->model['gid'],
				'vtype' => $results->model['vtype']
			);
			$this->options['formtext1'] = '';
			$this->options['formtext2'] = '';
			$html .= $this->availabilityForm('', true, array(), $vehdata);
		}
		if ($results->model['minprice'] > 0) {
			$price_txt = $currency->shortprice($results->model['minprice'], $this->options['roundprices']);
			$html .= '<div class="iosrt_veh_minprice">'.$eLang->get('FROM').' <span>'.$price_txt.'</span> / '.$eLang->get('DAY').'</div>'."\n";
			unset($price_txt);
		}
		$txt = $this->availableInLocs($results->model['lids'], $places, $results->model['vtype'], $eLang, true);
		if ($txt != '') { $html .= $txt;  }
		$html .= "</div>\n";//iosrt_veh_topside

		$html .= '<div class="iosrt_veh_topmain">'."\n";
		if (count($images) > 1) {
			$this->hasgallery = true;
			$html .= '<div class="iosrt_veh_img">'."\n";
			$html .= '<div class="iosrt_swiper">'."\n";
			$html .= '<div class="swiper-wrapper">'."\n";
			foreach ($images as $k => $img) {
				$idx = $k + 1;
				$html .= '<figure class="swiper-slide"><img src="'.$img.'" alt="'.$results->model['title'].' '.$idx.'" itemprop="image" /></figure>'."\n";
			}
			$html .= "</div>\n";
			$html .= '<div class="swiper-pagination"></div>'."\n";
			$html .= '<div class="swiper-button-prev"></div>'."\n";
			$html .= '<div class="swiper-button-next"></div>'."\n";
			$html .= "</div>\n";
			$html .= "</div>\n";//iosrt_veh_img
		} else {
			$html .= '<figure class="iosrt_veh_img"><img src="'.$images[0].'" alt="'.$results->model['title'].'" itemprop="image" /></figure>'."\n";
		}
		if ($paramsdata['iconitems']) {
			$html .= '<div class="iosrt_veh_features">'."\n";
			$num = 0;
			foreach ($paramsdata['iconitems'] as $item) {
				if ($num == 8) { break; }
				$html .= '<div class="iosrt_veh_feature"><img src="'.$item['icon'].'" alt="'.$item['title'].'" title="'.$item['title'].'" /><div>'.$item['text']."</div>\n</div>\n";
				$num++;
			}
			$html .= "</div>\n";
		}
		$html .= "</div>\n";//iosrt_veh_topmain

		$html .= "</div>\n";//iosrt_veh_topin

		$html .= "</div>\n";//iosrt_veh_top

		if ($results->model['description'] != '') { $html .= '<div class="iosrt_veh_description" itemprop="description">'.stripslashes($results->model['description'])."</div>\n"; }

		$feathtml = '';
		$txt = '';
		if ($paramsdata['iconitems']) {
			foreach ($paramsdata['iconitems'] as $item) { $txt .= '<li>'.$item['title']."</li>\n"; }
		}
		if ($paramsdata['textitems']) {
			foreach ($paramsdata['textitems'] as $item) { $txt .= '<li>'.$item."</li>\n"; }
		}
		if ($txt != '') {
			$feathtml = '<ul class="iosrt_veh_list">'."\n";
			$feathtml .= $txt;
			$feathtml .= "</ul>\n"; 
		}

		$princhtml = '';
		if (trim($results->model['priceincludes']) != '') {
			$incs = explode(',', $results->model['priceincludes']);
			if (is_array($priceincludes) && (count($priceincludes) > 0)) {
				$princhtml .= '<ul class="iosrt_veh_list">'."\n";
				foreach ($incs as $inc) {
					if (isset($priceincludes[$inc])) {
						if ($priceincludes[$inc]['description'] != '') {
							$princhtml .= '<li>'.$priceincludes[$inc]['title'].'<div>'.$priceincludes[$inc]['description']."</div></li>\n";
						} else {
							$princhtml .= '<li>'.$priceincludes[$inc]['title']."</li>\n";
						}
					}
				}
				$princhtml .= "</ul>\n";
			}
		}

		$html .= '<div class="iosrt_veh_mid">'."\n";
		if (($feathtml != '') && ($princhtml != '')) {
			$html .= '<div class="iosrt_veh_feat">'."\n";
			$txt = ($results->model['vtype'] == 'boat') ? $eLang->get('SPECIFICATION') : $eLang->get('MODEL_FEATURES');
			$html .= '<h3 class="iosrt_veh_head">'.$txt."</h3>\n";
			$html .= $feathtml;
			$html .= "</div>\n";//iosrt_veh_feat
			$html .= '<div class="iosrt_veh_princ">'."\n";
			$html .= '<h3 class="iosrt_veh_head">'.$eLang->get('OUR_PRICES_INCLUDE')."</h3>\n";
			$html .= $princhtml;
			$html .= "</div>\n";//iosrt_veh_princ
		} else if ($feathtml != '') {
			$txt = ($results->model['vtype'] == 'boat') ? $eLang->get('SPECIFICATION') : $eLang->get('MODEL_FEATURES');
			$html .= '<h3 class="iosrt_veh_head">'.$txt."</h3>\n";
			$html .= $feathtml;
		} else if ($princhtml != '') {
			$html .= '<h3 class="iosrt_veh_head">'.$eLang->get('OUR_PRICES_INCLUDE')."</h3>\n";
			$html .= $princhtml;
		}
		$html .= "</div>\n";//iosrt_veh_mid

		unset($feathtml, $princhtml);

		if (trim($company->terms) != '') {
			$html .= '<h3 class="iosrt_veh_head">'.$eLang->get('TERMS_CONDS')."</h3>\n";
			$html .= '<div class="iosrt_veh_terms">'.stripslashes($company->terms)."</div>\n";
		}

		$html .= '<div class="iosrt_vlspace"><a href="'.$fleetpage.'" class="iosrt_fleet_back">'.$eLang->get('BACK_FLEET_PAGE')."</a></div>\n";
		$html .= "</div>\n";//iosrt_vehicle_page

		$html .= $this->makeFooter($eLang, $company);

		if ($this->hasgallery) {
			$html .= "<script>\n";
			$html .= ' var iosRTSwiper = new Swiper (\'.iosrt_swiper\', {'."\n";
			$html .= 'loop: true, pagination: { el: \'.swiper-pagination\' }, navigation: { nextEl: \'.swiper-button-next\', prevEl: \'.swiper-button-prev\'}'."\n";
			$html .= "})\n";
			$html .= "</script>\n";
		}

		return $html;
	}


	/*************************/
	/* MAKE FLEET MODEL HTML */
	/*************************/
	private function makeFleetModelHTML($mid, $model, $modelsfeatures, $priceincludes, $categories, $places, $company, $eLang, $currency) {
		$images = array();
		if ($model['image'] != '') { $images[] = $company->imagesurl.'large/'.$model['image']; }
		if (count($model['images']) > 0) {
			foreach ($model['images'] as $img) { $images[] = $company->imagesurl.'large/'.$img; }
		}
		if (!$images) { $images[] = $this->options['clienturl'].'/images/noimage.png'; }

		if ($model['categories'] != '') {
			$modelfilters = 'categories:'.$model['categories'];
			$ctgs = explode(',', $model['categories']);
			$parts = array();
			foreach ($ctgs as $ctg) { $parts[] = 'categories:'.$ctg; }
			$modelfilters = implode(';', $parts);
			$modelfilters .= ';vtype:'.$model['vtype'];
			if ($model['modelparams'] != '') { $modelfilters .= ';'.$model['modelparams']; }
		} else {
			$modelfilters = 'vtype:'.$model['vtype'];
			if ($model['modelparams'] != '') { $modelfilters .= ';'.$model['modelparams']; }
		}

		$html = '<div class="iosrt_fleet_model" id="iosrt_model_'.$mid.'" data-filters="'.$modelfilters.'" itemprop="itemOffered" itemscope itemtype="https://schema.org/Car">'."\n";

		$ribbon_txt = '';
		if ($model['ribbon'] != '') {
			$css_addon = ($model['ribboncolor'] != '') ? ' iosrt_ribbon_'.$model['ribboncolor'] : '';
			$ribbon_txt = '<div class="iosrt_fleet_ribbon"><div class="iosrt_fleet_ribbonin"><div class="iosrt_fleet_ribboncorner'.$css_addon.'"></div><div class="iosrt_fleet_ribbontext'.$css_addon.'">'.$model['ribbon'].'</div></div></div>';
		}

		$cache_hash = md5('fleet'.$this->options['currency'].$this->options['vtype']);

		$fleetpage = '';
		if (trim($this->options['fleetpage']) != '') {
			if ($this->options['platform'] == 'elxis') {
				if (stripos($this->options['fleetpage'], 'http') === false) {//Elxis URI
					$fleetpage = eFactory::getElxis()->makeURL($this->options['fleetpage']);
				} else {
					$fleetpage = $this->options['fleetpage'];
				}
			} else {
				$fleetpage = $this->options['fleetpage'];
			}
			if (!filter_var($fleetpage, FILTER_VALIDATE_URL)) { $fleetpage = ''; }
		}
		if ($fleetpage == '') { $fleetpage = $this->makeFormURL(); }

		if (count($images) > 1) {
			$this->hasgallery = true;
			$html .= '<div class="iosrt_fleet_img">'.$ribbon_txt."\n";
			$html .= '<div class="iosrt_swiper">'."\n";
			$html .= '<div class="swiper-wrapper">'."\n";
			foreach ($images as $img) {
				$html .= '<figure class="swiper-slide"><img src="'.$img.'" alt="vehicle image" itemprop="image" /></figure>'."\n";
			}
			$html .= "</div>\n";
			$html .= '<div class="swiper-pagination"></div>'."\n";
			$html .= '<div class="swiper-button-prev"></div>'."\n";
			$html .= '<div class="swiper-button-next"></div>'."\n";
			$html .= "</div>\n";
			$html .= "</div>\n";//iosrt_fleet_img
		} else {
			$html .= '<figure class="iosrt_fleet_img"><a href="'.$fleetpage.'?mid='.$mid.'&amp;ch='.$cache_hash.'" title="'.$model['title'].'"><img src="'.$images[0].'" alt="'.$model['title'].'" itemprop="image" /></a>'.$ribbon_txt.'</figure>'."\n";
		}

		$html .= '<div class="iosrt_fleet_modelin">'."\n";
		$html .= '<h3 class="iosrt_fleet_h3" itemprop="name">'.$model['title']."</h3>\n";
		$html .= '<div class="iosrt_fleet_subtitle" itemprop="description">'.$this->makeModelSubtitle($model['group'], $model['vtype'], $model['categories'], $categories)."</div>\n";

		if ($model['minprice'] > 0) {
			$price_txt = $currency->shortprice($model['minprice'], $this->options['roundprices']);
			$model_price_incs = array();
			if (trim($model['priceincludes']) != '') {
				$incs = explode(',', $model['priceincludes']);
				if (is_array($priceincludes) && (count($priceincludes) > 0)) {
					foreach ($incs as $inc) {
						if (isset($priceincludes[$inc])) { $model_price_incs[] = $priceincludes[$inc]['title']; }
					}
				}
			}
			if ($model_price_incs) {
				$txt = $eLang->get('OUR_PRICES_INCLUDE').': '.implode(', ', $model_price_incs);
				$html .= '<div class="iosrt_fleet_minprice">'.$eLang->get('FROM').' <span class="iosrt_fleet_price">'.$price_txt.'</span> / '.$eLang->get('DAY').' <span class="iosrt_fleet_info iosrt_wtooltip" title="'.$txt.'">i</span></div>'."\n";
			} else {
				$html .= '<div class="iosrt_fleet_minprice">'.$eLang->get('FROM').' <span class="iosrt_fleet_price">'.$price_txt.'</span> / '.$eLang->get('DAY').'</div>'."\n";
			}
			unset($price_txt, $model_price_incs);
		}

		$html .= '<div class="iosrt_vsspace iosrt_center"><a href="'.$fleetpage.'?mid='.$mid.'&amp;ch='.$cache_hash.'" class="iosrt_fleet_book" title="'.$eLang->get('BOOK_BUTTON').'">'.$eLang->get('BOOK_BUTTON').'</a></div>'."\n";

		$txt = $this->availableInLocs($model['lids'], $places, $model['vtype'], $eLang, true);
		if ($txt != '') { $html .= $txt; }

		$data = $this->makeModelParamsHTML($model['modelparams'], $model['vtype'], $modelsfeatures, $eLang, 5, true);
		if ($data['iconitems']) {
			$html .= '<div class="iosrt_fleet_features">';
			foreach ($data['iconitems'] as $item) {
				$html .= '<div class="iosrt_fleet_feature">'."\n";
				$html .= '<img src="'.$item['icon'].'" alt="'.$item['title'].'" title="'.$item['title'].'" /><div>'.$item['text']."</div>\n";
				$html .= "</div>\n";
			}
			$html .= "</div>\n";
		}

		$html .= "</div>\n";//iosrt_fleet_modelin

		$html .= "</div>\n";//iosrt_fleet_model

		return $html;
	}


	/***************************/
	/* MAKE FLEET PAGE FILTERS */
	/***************************/
	private function fleetFilters($models, $categories, $modelsfeatures, $eLang) {
		if ($this->options['sidecolumn'] == '') { return ''; }//just in case

		$blocks = explode(',', $this->options['sidecolumn']);

		$searchidx = array_search('search', $blocks);
		$vtypeidx = array_search('vtype', $blocks);
		$categoriesidx = array_search('categories', $blocks);
		if (($searchidx !== false) || ($vtypeidx !== false) || ($categoriesidx !== false)) {//vtype must be first and categories second, dont use search
			$tmpblocks = $blocks;
			$blocks = array();
			//if ($searchidx !== false) { $blocks[] = 'search'; }//don't use search
			if ($vtypeidx !== false) { $blocks[] = 'vtype'; }
			if ($categoriesidx !== false) { $blocks[] = 'categories'; }

			foreach ($tmpblocks as $block) {
				if (($block == 'search') || ($block == 'categories') || ($block == 'vtype')) { continue; }
				$blocks[] = $block;
			}
			unset($tmpblocks);
		}
		unset($searchidx, $vtypeidx, $categoriesidx);
		if (!$blocks) { return ''; }

		$filter_categories = array();
		$features = array();//all features and their values
		$vtypes = array();

		foreach ($models as $mid => $model) {
			$gvtype = $model['vtype'];
			if (!in_array($gvtype, $vtypes)) { $vtypes[] = $gvtype; }
			if ($model['categories'] != '') {
				$parts = explode(',', $model['categories']);
				foreach ($parts as $part) {
					if (isset($categories[$gvtype][$part])) { $filter_categories[$part] = $categories[$gvtype][$part]; }
				}
			}

			if (trim($model['modelparams']) != '') {
				$parts = explode(';', $model['modelparams']);
				foreach ($parts as $part) {
					$feat = explode(':', $part);
					if (count($feat) != 2) { continue; }
					$name = $feat[0];
					if (in_array($name, array('tank', 'horsepower', 'enginecc', 'enginestrokes', 'discbrakes'))) { continue; }
					if (!in_array('all', $blocks)) {//"all" keyword is used for all model features
						if (!in_array($name, $blocks)) { continue; }
					} else {
						if (!in_array($name, $blocks)) { $blocks[] = $name; }
					}
					if (!isset($features[$name])) {
						$features[$name] = array();
						$features[$name][] = $feat[1];
					} else {
						if (!in_array($feat[1], $features[$name])) { $features[$name][] = $feat[1]; }
					}
				}
			}
		}

		$vtype = (count($vtypes) !== 1) ? '' : $vtypes[0];
		if (count($vtypes) < 2) {//dont display vtypes filters if only 1 vehicle type
			$k = array_search('vtype', $blocks);
			if ($k !== false) { unset($blocks[$k]); }
		}

		$filters = array();
		if ($vtype == 'car') {
			$filters['general'] = array('title' => $eLang->get('CAR_SPECIFICATIONS'), 'options' => array());
		} else if ($vtype == 'boat') {
			$filters['general'] = array('title' => $eLang->get('SPECIFICATION'), 'options' => array());
		} else {
			$filters['general'] = array('title' => $eLang->get('VEHICLE_SPECIFICATIONS'), 'options' => array());
		}
		if ($features) {
			foreach ($features as $name => $values) {
				$item = false;
				if ($vtype != '') {
					if (isset($modelsfeatures[$vtype])) {
						if (isset($modelsfeatures[$vtype][$name])) { $item = $modelsfeatures[$vtype][$name]; }
					}
				} else {
					foreach ($modelsfeatures as $vtype => $mfeatures) {
						if (isset($mfeatures[$name])) { $item = $mfeatures[$name]; break; }
					}
				}
				if (!$item) { continue; }

				$type = $item['type'];
				if ($type == 'number') {
					if (count($values) > 1) {
						$filters[$name] = array('title' => $item['title'], 'options' => array());
						foreach ($values as $value) {
							$filters[$name]['options'][] = array('title' => $value, 'value' => $value, 'feature' => $name);
						}
					} else {
						$filters['general']['options'][] = array('title' => $item['title'].': '.$values[0], 'value' => $values[0], 'feature' => $name);
					}
				} else if ($type == 'select') {
					if (!$item['options']) { continue; }
					$n = count($values);
					if ($n > 1) {
						$filters[$name] = array('title' => $item['title'], 'options' => array());
					}
					foreach($item['options'] as $opt) {
						foreach ($values as $value) {
							if ($value == $opt['value']) {
								if ($n > 1) {
									$filters[$name]['options'][] = array('title' => $opt['title'], 'value' => $opt['value'], 'feature' => $name);
								} else {
									$filters['general']['options'][] = array('title' => $item['title'].': '.$opt['title'], 'value' => $opt['value'], 'feature' => $name);
								}
							}
						}
					}
				} else if ($type == 'yesno') {
					if ((count($values) == 1) && ($values[0] == 1)) {
						$filters['general']['options'][] = array('title' => $item['title'], 'value' => 1, 'feature' => $name);
					}
				} else {
					continue;//not supported type!
				}
			}
		}

		$html = '';
		$filters_box_opened = false;
		$all_filters = array();
		foreach ($blocks as $block) {
			if ($block == 'all') { continue; }
			if ($block == 'search') { continue; }

			if (!$filters_box_opened) {
				$html .= '<div class="iosrt_fleetfilters_box">'."\n";
				$html .= '<h3 class="iosrt_fleetfilter_boxtitle"><a href="javascript:void(null);" class="iosrt_fleetfilter_lnkdown" onclick="iosRTToggleFleetFilters();" id="iosrt_fleetfilter_lnk">'.$eLang->get('FILTERS')."</a></h3>\n";
				$html .= '<div class="iosrt_invisible" id="iosrt_filters_showhidebox">'."\n";
				$html .= '<div class="iosrt_fleetfilters_inbox">'."\n";
				$filters_box_opened = true;
			}

			if ($block == 'vtype') {
				$html .= '<div class="iosrt_fleetfilters_wrapbox">'."\n";

				$html .= '<h3 class="iosrt_fleetfilter_title">'.$eLang->get('VEHICLE_TYPE')."</h3>\n";
				$html .= '<ul class="iosrt_fleetfilters" id="iosrt_filter_group_vtype">'."\n";
				foreach ($vtypes as $vtype) {
					switch ($vtype) {
						case 'car': $title = $eLang->get('CARS'); break;
						case 'moto': $title = $eLang->get('MOTORCYCLES'); break;
						case 'scooter': $title = $eLang->get('SCOOTERS'); break;
						case 'boat': $title = $eLang->get('BOATS'); break;
						case 'atv': $title = $eLang->get('ATVS'); break;
						case 'buggy': $title = $eLang->get('BUGGIES'); break;
						case 'bicycle': $title = $eLang->get('BICYCLES'); break;
						case 'other': $title = $eLang->get('OTHER_VEHICLES'); break;
						default: $title = ucfirst($vtype); break;
					}
					$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_vtype:'.$vtype.'">'.$title.'<input type="checkbox" value="'.$vtype.'" id="iosrt_filter_vtype:'.$vtype.'" onchange="iosRTFilter(true);" data-orgroup="vtype" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
					$all_filters[] = 'vtype:'.$vtype;
				}
				$html .= "</ul>\n";
				$html .= "</div>\n";
				continue;
			}

			if ($block == 'categories') {
				if (!$filter_categories) { continue; }
				$html .= '<div class="iosrt_fleetfilters_wrapbox">'."\n";
				$html .= '<h3 class="iosrt_fleetfilter_title">'.$eLang->get('CATEGORY')."</h3>\n";
				$html .= '<ul class="iosrt_fleetfilters" id="iosrt_filter_group_categories">'."\n";
				foreach ($filter_categories as $name => $title) {
					$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_categories:'.$name.'">'.$title.'<input type="checkbox" value="'.$name.'" id="iosrt_filter_categories:'.$name.'" onchange="iosRTFilter(true);" data-orgroup="categories" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
					$all_filters[] = 'categories:'.$name;
				}
				$html .= "</ul>\n";
				$html .= "</div>\n";
				continue;
			}

			if (!isset($filters[$block])) { continue; }//filter put in "general" block

			$html .= '<div class="iosrt_fleetfilters_wrapbox">'."\n";
			$html .= '<h3 class="iosrt_fleetfilter_title">'.$filters[$block]['title']."</h3>\n";
			$html .= '<ul class="iosrt_fleetfilters" id="iosrt_filter_group_'.$block.'">'."\n";
			foreach ($filters[$block]['options'] as $opt) {
				$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'">'.$opt['title'].'<input type="checkbox" value="'.$opt['value'].'" id="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'" onchange="iosRTSingleFilter(\''.$opt['feature'].'\', this, true);" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
				$all_filters[] = $opt['feature'].':'.$opt['value'];
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";
		}

		if ($filters['general']['options']) {
			if (!$filters_box_opened) {
				$html .= '<div class="iosrt_filters_box">'."\n";
				$filters_box_opened = true;
			}
			$html .= '<div class="iosrt_fleetfilters_wrapbox">'."\n";
			$html .= '<h3 class="iosrt_fleetfilter_title">'.$filters['general']['title']."</h3>\n";
			$html .= '<ul class="iosrt_fleetfilters" id="iosrt_filter_group_general">'."\n";
			foreach ($filters['general']['options'] as $opt) {
				$html .= '<li><label class="iosrt_filter_label" for="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'">'.$opt['title'].'<input type="checkbox" value="'.$opt['value'].'" id="iosrt_filter_'.$opt['feature'].':'.$opt['value'].'" onchange="iosRTFilter(true);" /><span class="iosrt_filter_checkmark"></span></label></li>'."\n";
				$all_filters[] = $opt['feature'].':'.$opt['value'];
			}
			$html .= "</ul>\n";
			$html .= "</div>\n";
		}

		if ($filters_box_opened) {
			$html .= "</div>\n";//iosrt_fleetfilters_inbox
			$html .= "</div>\n";//iosrt_filters_showhidebox
			$html .= "</div>\n";//iosrt_fleetfilters_box
		}

		$html .= '<div class="iosrt_invisible" id="iosrt_all_filters" data-filters="'.implode(';', $all_filters).'"></div>'."\n";

		return $html;
	}


	/************************/
	/* RC TASK "CLEARCACHE" */
	/************************/
	private function rcClearCache($type) {
		$cache_dir = $this->path.'/cache/'.$this->options['apiid'].'/';
		if (file_exists($cache_dir)) {
			$files = glob($cache_dir.'*.php');
			if ($files) {
				foreach ($files as $file) {
					@unlink($file);
				}
			}
			if (!file_exists($cache_dir.'index.html')) {
				$fp = @fopen($cache_dir.'index.html', 'w');
				if ($fp) {
					fwrite($fp, 'IOS Rentals by Ioannis Sannos ( <a href="https://www.isopensource.com">https://www.isopensource.com</a> )');
					fclose($fp);
				}
			}
		}

		$response = array('success' => 1, 'message' => 'OK');
		$this->responseRC($type, $response);
	}


	/************************/
	/* RC XML/JSON RESPONSE */
	/************************/
	private function responseRC($type, $params) {
		if ($type == 'XML') {
			$response = '<iosrentals>'."\n";
			$response .= "\t".'<software><![CDATA[IOS Rentals]]></software>'."\n";
			$response .= "\t".'<author><![CDATA[Ioannis Sannos]]></author>'."\n";
			$response .= "\t".'<authorurl><![CDATA[https://www.isopensource.com]]></authorurl>'."\n";
			$response .= "\t".'<license><![CDATA[Commercial]]></license>'."\n";
			$response .= "\t".'<time>'.time().'</time>'."\n";
			if ($params) {
				foreach ($params as $k => $v) {
					if (is_array($v)) {
						$response .= "\t<".$k.">"."\n";
						foreach ($v as $ka => $va) {
							if (is_array($va)) {
								$response .= "\t\t<".$ka."><![CDATA[ERROR: MAXIMUM LEVEL REACHED!]]></".$ka.">\n";
							} else if (($va == '') || is_numeric($va)) {
								$response .= "\t\t<".$ka.">".$va."</".$ka.">\n";
							} else {
								$response .= "\t\t<".$ka."><![CDATA[".$va."]]></".$ka.">\n";
							}
						}
						$response .= "\t</".$k.">"."\n";
					} else if (($v == '') || is_numeric($v)) {
						$response .= "\t<".$k.">".$v."</".$k.">\n";
					} else {
						$response .= "\t<".$k."><![CDATA[".$v."]]></".$k.">\n";
					}
				}
			}
			$response .= "</iosrentals>\n";

			$this->pageHeaders('text/xml');
			echo $response;
			exit;
		}

		if (!$params) { $params = array(); }
		$params['software'] = 'IOS Rentals';
		$params['author'] = 'Ioannis Sannos';
		$params['time'] = time();

		$this->pageHeaders('application/json');
		echo json_encode($params);
		exit;
	}


	/*****************************************/
	/* CANCEL/CHANGE RESERVATION (AJAX TASK) */
	/*****************************************/
	private function changeReservation() {
		$eLang = $this->lang;

		$pat = "#([\']|[\!]|[\(]|[\)]|[\;]|[\"]|[\$]|[\#]|[\<]|[\>]|[\*]|[\%]|[\~]|[\`]|[\^]|[\|]|[\{]|[\}]|[\\\])#u";
		$subject = isset($_POST['subject']) ? (int)$_POST['subject'] : 0;
		$rescode = filter_input(INPUT_POST, 'rescode', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH);
		if ($rescode != '') { $rescode = trim(preg_replace($pat, '', $rescode)); }
		$email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		$comments = filter_input(INPUT_POST, 'comments', FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		if ($comments != '') { $comments = strip_tags($comments); }

		$response = array('success' => 0, 'message' => '');
		if ($rescode == '') {
			$response['message'] = sprintf($eLang->get('FIELD_NOEMPTY'), $eLang->get('RESERVATION_CODE'));
			$this->pageHeaders('application/json');
			echo json_encode($response);
			exit;
		}
		if ((trim($email) == '') || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
			$response['message'] = $eLang->get('INV_EMAIL');
			$this->pageHeaders('application/json');
			echo json_encode($response);
			exit;
		}

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',JSON,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'JSON',
			'ts' => $ts,
			'enc' => $enc,
			'subject' => $subject,
			'rescode' => $rescode,
			'email' => $email,
			'comments' => $comments
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/cancel';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/cancel';
		}

		$json = $this->helper->rcpost($taskurl, $params);
		if ($this->helper->getError() != '') {//RC connection failed
			$response['message'] = $this->helper->getError();
			$this->pageHeaders('application/json');
			echo json_encode($response);
			exit;
		}

		$rcresult = @json_decode($json, true);
		if (json_last_error() != JSON_ERROR_NONE) {
			$response['message'] = 'Request failed! Please retry later (1).';
		} else if (!is_array($rcresult)) {
			$response['message'] = 'Request failed! Please retry later (2).';
		} else if (!isset($rcresult['success'])) {
			if (isset($rcresult['message'])) { $response['message'] = $rcresult['message']; }
			if (trim($response['message']) == '') { $response['message'] = 'Request failed! Please retry later (3).'; }
		} else {
			$response['success'] = (int)$rcresult['success'];
			if (isset($rcresult['message'])) { $response['message'] = $rcresult['message']; }
			if ($response['success'] != 1) {
				if (trim($response['message']) == '') { $response['message'] = 'Request failed! Please retry later (4).'; }
			} else {
				if (trim($response['message']) == '') { $response['message'] = 'Your request submitted successfully! We will contact you as soon as possible.'; }
			}
		}

		$encoded = json_encode($response);
		$this->pageHeaders('application/json');
		echo $encoded;
		exit;
	}

}

?>