<?php 
/**
* @version		$Id: helper.class.php 1375 2019-12-30 19:22:46Z IOS $
* @package		Component IOS Reservations
* @author		Ioannis Sannos ( https://www.isopensource.com )
* @copyright	Copyright (c) 2008-2020 Is Open Source (https://www.isopensource.com). All rights reserved.
* @license		Commercial
* @description 	Online booking system
**************************************/

/**
=== COMMERCIAL LICENSE ===

IT IS EXTREMELY FORBIDDEN TO MODIFY THIS FILE
IOS RENTALS IS COPYRIGHTED BY IOANNIS SANNOS
HTTPS://WWW.ISOPENSOURCE.COM
*/

defined('_IOSRTAPICLIENT_') or die ('Direct access to this location is not allowed');


class iosrtApiHelper {

	private $options = array();
	private $clientversion = 20;
	private $path = '';
	private $errormsg = '';


	/*********************/
	/* MAGIC CONSTRUCTOR */
	/*********************/
	public function __construct($options, $path, $clientversion) {
		$this->options = $options;
		$this->path = $path;
		$this->clientversion = (int)$clientversion;
	}


	/**************************/
	/* GET LAST ERROR MESSAGE */
	/**************************/
	public function getError() {
		return $this->errormsg;
	}


	public function rcpost($url, $params=null) {
		if (function_exists('curl_init')) {
			$response = $this->curlpost($url, $params);
		} else {
			$response = $this->httppost($url, $params);
		}
		return $response;
	}


	private function curlpost($url, $params=null) {
		if (isset($_SERVER['HTTP_USER_AGENT']) && ($_SERVER['HTTP_USER_AGENT'] != '')) {
			$agent = $_SERVER['HTTP_USER_AGENT'];
		} else {
			$agent = 'CURL';
		}
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_POST, true);
		if (isset($params)) {
			curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
		} else {
			curl_setopt($ch, CURLOPT_POSTFIELDS, "");
		}
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_FAILONERROR, true);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_USERAGENT, $agent);
		curl_setopt($ch, CURLOPT_REFERER, $this->options['clienturl'].'/ajax.php');
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
		curl_setopt($ch, CURLOPT_TIMEOUT,20);
		$result = curl_exec($ch);
		if (curl_errno($ch) == 0) {
			curl_close($ch);
			return $result;
		} else {
			$this->errormsg = curl_error($ch);
			curl_close($ch);
		return false;
		}
	}


	private function httppost($url, $params=null) {
		$parseurl = parse_url($url);
		$poststr = '';
		if ($params) {
			foreach($params as $key => $val) {
				$poststr .= $key.'='.urlencode($val).'&';
			}
		}
		$requestheader = 'POST '.$parseurl['path']." HTTP/1.1\r\n";
		$requestheader .= 'Host: '.$parseurl['host']."\r\n";
		$requestheader .= "Referer: ".$this->options['clienturl'].'/ajax.php'."\r\n";
		$requestheader .= "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:58.0) Gecko/20100101 Firefox/58.0\r\n";
		$requestheader .= 'Accept: text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,*/*;q=0.6'."\r\n";
		$requestheader .= 'Accept-Language: en-us,en;q=0.5'."\r\n";
		$requestheader .= 'Accept-Encoding: deflate'."\r\n";
		$requestheader .= 'Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7'."\r\n";
		if ($poststr != '') {
			$requestheader .= "Content-Type: application/x-www-form-urlencoded\r\n";
			$requestheader .= "Content-Length: ".strlen($poststr)."\r\n";
			$requestheader .= "\r\n";
			$requestheader .= $poststr;
		}
		$requestheader .= "Connection: Close\r\n\r\n"; 
		if (!isset($parseurl['port'])) {
			$parseurl['port'] = ($parseurl['scheme'] == 'https') ? 443 : 80;
		}

		$fp = @fsockopen($parseurl['host'], $parseurl['port'], $errno , $errstr , 15);
		if (!$fp) { $this->errormsg = 'Could not connect to API server!'; return false; }

		stream_set_timeout($fp, 10);
		fputs($fp, $requestheader);
		$raw = '';
		$start = false;
		$end = false;
		while(!feof($fp)) {
			$x = fgets($fp);
			if (!$start) {
				if (preg_match('#^(\<iosrentals\>)#', trim($x))) { $start = true; }
			}
			if (!$start) { continue; }
			if (!$end) {
				if (preg_match('#^(\<\/iosrentals\>)#', trim($x))) { $end = true; }
			}
			$raw .= $x;
			$info = stream_get_meta_data($fp);
			if ($info['timed_out']) {
				fclose($fp);
				$this->errormsg = 'Connection time out!';
				return false;
			}
			if ($end) { break; }
		}
		fclose($fp);
		
		return $raw;
	}


	/********************************/
	/* GET SEARCHABLE VEHICLE TYPES */
	/********************************/
	public function searchableVehicleTypes($company) {
		$vtypes = explode(',', $company->vtypes);
		if ($this->options['vtype'] == '') {
			$vtypes_to_search = $vtypes;
		} else {
			$vtypes_to_search = array();
			$selected_vtypes = explode(',', $this->options['vtype']);
			foreach ($selected_vtypes as $v) {
				if (in_array($v, $vtypes)) { $vtypes_to_search[] = $v; }
			}
		}
		return $vtypes_to_search;
	}


	/*******************************************************************/
	/* GET BASIC DATA (COMPANY/PLACES/ETC) DATA FROM CACHE OR REMOTELY */
	/*******************************************************************/
	public function basicRequest($type, $evenexpiredcache=false) {
		$cache_file = $this->getCacheFile($type);
		if ($cache_file != '') {
			if ($evenexpiredcache) {//fetch data even with expired cache (for steps other than "0"/availability form)
				include($cache_file);
				return $rtdata;
			}
			if (time() - filemtime($cache_file) < $this->options['cachetime']) {
				include($cache_file);
				return $rtdata;
			}
		}

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/basic';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/basic';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') {
			if ($cache_file != '') {//RC connection failed, we have cached file, use it even expired!
				include($cache_file);
				return $rtdata;
			}
			return false;
		}

		$data = $this->parseBasicData($xml);
		if ($this->errormsg != '') {
			if ($cache_file != '') {//RC connection failed, we have cached file, use it even expired!
				include($cache_file);
				return $rtdata;
			}
			return false;
		}
		unset($xml);

		$this->saveToCache('company', $data['company'], true, true);
		$this->saveToCache('places', $data['places'], true, false);
		$this->saveToCache('priceincludes', $data['priceincludes'], true, false);
		$this->saveToCache('modelsfeatures', $data['modelsfeatures'], true, false);
		$this->saveToCache('categories', $data['categories'], true, false);
		$this->saveToCache('noworkhours', $data['noworkhours'], true, false);
		$this->saveToCache('rentdurations', $data['rentdurations'], false, false);

		return $data[$type];
	}


	/********************************************/
	/* SEARCH / GET AVAILABLE VEHICLES REMOTELY */
	/********************************************/
	public function searchRequest($fdata, $company) {
		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);

		$vtype = $fdata['vtype'];
		if ($this->options['unifiedmoto'] == 1) {
			if (($vtype == 'moto') || ($vtype == 'scooter') || ($vtype == 'atv') || ($vtype == 'buggy')) {
				$vtypes = $this->searchableVehicleTypes($company);
				$svtypes = array();
				if (in_array('moto', $vtypes)) { $svtypes[] = 'moto'; }
				if (in_array('scooter', $vtypes)) { $svtypes[] = 'scooter'; }
				if (in_array('atv', $vtypes)) { $svtypes[] = 'atv'; }
				if (in_array('buggy', $vtypes)) { $svtypes[] = 'buggy'; }
				$vtype = implode(',', $svtypes);
				unset($vtypes, $svtypes);
			}
		}

		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion,
			'pickdate' => $fdata['pickdate'].' '.$fdata['pickhour'].':'.$fdata['pickmin'].':00',
			'picklid' => $fdata['picklid'],
			'dropdate' => $fdata['dropdate'].' '.$fdata['drophour'].':'.$fdata['dropmin'].':00',
			'droplid' => $fdata['droplid'],
			'vtype' => $vtype,
			'mid' => 0,
			'gid' => 0
		);
		if (isset($fdata['mid'])) { $params['mid'] = (int)$fdata['mid']; }//from fleet page
		if (isset($fdata['gid'])) { $params['gid'] = (int)$fdata['gid']; }//from fleet page

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/search';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/search';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') { return false; }//RC connection failed

		$results = $this->parseSearchResults($xml);
		if ($this->errormsg != '') { return false; }
		unset($xml);

		return $results;
	}


	/**********************/
	/* BEGIN BOOK REQUEST */
	/**********************/
	public function beginBookRequest($fdata) {
		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion,
			'pickdate' => $fdata['pickup'],
			'picklid' => $fdata['picklid'],
			'dropdate' => $fdata['dropoff'],
			'droplid' => $fdata['droplid'],
			'mid' => $fdata['mid']
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/beginbook';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/beginbook';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') { return false; }//RC connection failed

		$results = $this->parseBeginBookResults($xml);
		if ($this->errormsg != '') { return false; }
		unset($xml);

		return $results;
	}


	/***************************************************/
	/* PAYMENT NOTIFICATION REQUEST (EG ALPHA SUCCESS) */
	/***************************************************/
	public function payNotifyRequest($pdata, $paymethod) {
		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion,
			'paymethod' => $paymethod
		);
		if ($pdata) {
			foreach ($pdata as $k => $v) { $params[$k] = $v; }
		}

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/paynotify';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/paynotify';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') { return -1; }//RC connection failed

		$result = $this->parsePayNotifyResult($xml);
		return $result;
	}


	/*******************************/
	/* GET FLEET VEHICLES REMOTELY */
	/*******************************/
	public function fleetRequest($vtype) {
		$cache_hash = md5('fleet'.$this->options['currency'].$vtype);
		$cache_file = $this->getCacheFile('fleet_'.$cache_hash);
		if ($cache_file != '') {
			if (time() - filemtime($cache_file) < $this->options['cachetime']) {
				include($cache_file);
				return $rtdata;
			}
		}

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion,
			'vtype' => $vtype
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/fleet';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/fleet';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') {//RC connection failed
			if ($cache_file != '') {//RC connection failed, we have cached file, use it even expired!
				include($cache_file);
				return $rtdata;
			}
			return false;
		}

		$data = $this->parseFleetResults($xml);
		if ($this->errormsg != '') {
			if ($cache_file != '') {//RC connection failed, we have cached file, use it even expired!
				include($cache_file);
				return $rtdata;
			}
			return false;
		}
		unset($xml);

		$this->saveToCache('fleet_'.$cache_hash, $data, true, true, false);

		return $data;
	}


	/************************************************/
	/* GET/REQUEST FLEET MODEL DETAILS (MODEL PAGE) */
	/************************************************/
	public function fleetModelRequest($mid, $cache_hash) {
		$mid = (int)$mid;
		if ($cache_hash != '') {
			$cache_file = $this->getCacheFile('fleet_'.$cache_hash);
			if ($cache_file != '') {
				include($cache_file);
				if (isset($rtdata->models[$mid])) {
					$data = new stdClass;
					$data->currency = $rtdata->currency;
					$data->usercurrency = $rtdata->usercurrency;
					$data->usercurrencyrate = $rtdata->usercurrencyrate;
					$data->model = $rtdata->models[$mid];
					return $data;
				}
			}
		}

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.',XML,'.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => 'XML',
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion,
			'mid' => $mid
		);

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/model';
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/model';
		}

		$xml = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') { return false; }

		$data = $this->parseModelResults($xml);
		if ($this->errormsg != '') { return false; }
		unset($xml);

		return $data;
	}


	/*******************/
	/* GENERIC REQUEST */
	/*******************/
	public function genericRequest($endpoint, $type='XML', $custom_options=array()) {
		if ($type != 'JSON') { $type = 'XML'; }

		$ts = time();
		$enc = sha1($this->options['apiid'].','.$ts.','.$type.','.$this->options['prkey']);
		$params = array(
			'apiid' => $this->options['apiid'],
			'currency' => $this->options['currency'],
			'type' => $type,
			'ts' => $ts,
			'enc' => $enc,
			'clientversion' => $this->clientversion
		);
		if ($custom_options) {
			foreach ($custom_options as $k => $v) {
				$params[$k] = $v;
			}
		}

		if ($this->options['lang'] != '') {
			$taskurl = $this->options['serverurl'].'/inner.php/'.$this->options['lang'].'/reservations/rentals/api/'.$endpoint;
		} else {
			$taskurl = $this->options['serverurl'].'/inner.php/reservations/rentals/api/'.$endpoint;
		}

		$response = $this->rcpost($taskurl, $params);
		if ($this->errormsg != '') { return false; }
		return $response;
	}


	/****************************************************/
	/* SAVE DATA TO CACHE (SIMPLE OBJECT/COMPLEX ARRAY) */
	/****************************************************/
	public function saveToCache($name, $data, $multilingual=true, $is_object=true, $is_book=false) {
		$cache_dir = $is_book ? $this->path.'/cache/bk/' : $this->path.'/cache/'.$this->options['apiid'].'/';
		$cache_file = $cache_dir.$name.'.php';
		if ($multilingual) {
			if ($this->options['lang'] != '') { $cache_file = $cache_dir.$name.'.'.$this->options['lang'].'.php'; }
		}

		if (!file_exists($cache_dir)) {
			if ($ok = @mkdir($cache_dir, 0777, true)) {
				if (!$ok) { return false; }
				$fp = @fopen($cache_dir.'index.html', 'w');
				if ($fp) {
					fwrite($fp, 'IOS Rentals by Ioannis Sannos ( <a href="https://www.isopensource.com">https://www.isopensource.com</a> )');
					fclose($fp);
				}
			}
		}

		$contents = '<?php '."\n\n";
		$contents .= 'defined(\'_IOSRTAPICLIENT_\') or die (\'Direct access to this location is not allowed\');'."\n\n";
		if ($is_object) {
			$contents .= '$rtdata = new stdClass;'."\n";
		} else {
			$contents .= '$rtdata = array('."\n";
		}
		if ($is_object) {
			if ($data) {
				$vars = get_object_vars($data);
				foreach ($vars as $k => $v) {
					if (is_array($v)) {
						$contents .= '$rtdata->'.$k.' = array('."\n";
						foreach ($v as $ka => $va) {
							$idxa = is_numeric($ka) ? $ka : '\''.$ka.'\'';
							if (is_array($va)) {
								$contents .= "\t".$idxa.' => array('."\n";
								foreach ($va as $kb => $vb) {
									$idxb = is_numeric($kb) ? $kb : '\''.$kb.'\'';
									if (is_array($vb)) {
										$contents .= "\t\t".$idxb.' => array('."\n";
										foreach ($vb as $kc => $vc) {
											$idxc = is_numeric($kc) ? $kc : '\''.$kc.'\'';
											$contents .= "\t\t\t".$idxc.' => \''.addslashes($vc).'\','."\n";
										}
										$contents .= "\t\t),\n";
									} else {
										$contents .= "\t\t".$idxb.' => \''.addslashes($vb).'\','."\n";
									}
								}
								$contents .= "\t),\n";
							} else {
								$contents .= "\t".$idxa.' => \''.addslashes($va).'\','."\n";
							}
						}
						$contents .= ");\n";
					} else {
						//if (($v == '') || !is_numeric($v)) {
							$contents .= '$rtdata->'.$k.' = \''.addslashes($v).'\';'."\n";
						//} else {
						//	$contents .= '$rtdata->'.$k.' = '.$v.';'."\n";//problem with numerics starting with "0", eg 05345
						//}
					}
				}
			}
		} else {
			if ($data) {
				foreach ($data as $k => $v) {
					$idx = is_numeric($k) ? $k : '\''.$k.'\'';
					if (is_array($v)) {
						$contents .= "\t".$idx.' => array('."\n";
						foreach ($v as $ka => $va) {
							$idxa = is_numeric($ka) ? $ka : '\''.$ka.'\'';
							if (is_array($va)) {
								$contents .= "\t\t".$idxa.' => array('."\n";
								foreach ($va as $kb => $vb) {
									$idxb = is_numeric($kb) ? $kb : '\''.$kb.'\'';
									if (is_array($vb)) {
										$contents .= "\t\t\t".$idxb.' => array('."\n";
										foreach ($vb as $kc => $vc) {
											$idxc = is_numeric($kc) ? $kc : '\''.$kc.'\'';
											if (is_array($vc)) {
												$contents .= "\t\t\t\t".$idxc.' => array('."\n";
												foreach ($vc as $kd => $vd) {
													$idxd = is_numeric($kd) ? $kd : '\''.$kd.'\'';
													if (is_array($vd)) {
														$contents .= "\t\t\t\t\t".$idxd.' => array('."\n";
														foreach ($vd as $ke => $ve) {
															$idxe = is_numeric($ke) ? $ke : '\''.$ke.'\'';
															if (is_array($ve)) {
																$contents .= "\t\t\t\t\t\t".$idxe.' => array('."\n";
																foreach ($ve as $kf => $vf) {
																	$idxf = is_numeric($kf) ? $kf : '\''.$kf.'\'';
																	if (is_array($vf)) {
																		$contents .= "\t\t\t\t\t\t\t".$idxf.' => array(),'."\n";//MAX LEVEL
																	//} else if (($vf == '') || !is_numeric($vf)) {
																	//	$contents .= "\t\t\t\t\t\t\t".$idxf.' => \''.addslashes($vf).'\','."\n";
																	} else {
																	//	$contents .= "\t\t\t\t\t\t\t".$idxf.' => '.$vf.','."\n";//problem with numerics starting with "0"
																		$contents .= "\t\t\t\t\t\t\t".$idxf.' => \''.addslashes($vf).'\','."\n";
																	}
																}
																$contents .= "\t\t\t\t\t\t),\n";
															//} else if (($ve == '') || !is_numeric($ve)) {
															//	$contents .= "\t\t\t\t\t\t".$idxe.' => \''.addslashes($ve).'\','."\n";
															} else {
															//	$contents .= "\t\t\t\t\t\t".$idxe.' => '.$ve.','."\n";
																$contents .= "\t\t\t\t\t\t".$idxe.' => \''.addslashes($ve).'\','."\n";
															}
														}
														$contents .= "\t\t\t\t\t),\n";
													//} else if (($vd == '') || !is_numeric($vd)) {
													//	$contents .= "\t\t\t\t\t".$idxd.' => \''.addslashes($vd).'\','."\n";
													} else {
													//	$contents .= "\t\t\t\t\t".$idxd.' => '.$vd.','."\n";
														$contents .= "\t\t\t\t\t".$idxd.' => \''.addslashes($vd).'\','."\n";
													}
												}
												$contents .= "\t\t\t\t),\n";
											//} else if (($vc == '') || !is_numeric($vc)) {
											//	$contents .= "\t\t\t\t".$idxc.' => \''.addslashes($vc).'\','."\n";
											} else {
											//	$contents .= "\t\t\t\t".$idxc.' => '.$vc.','."\n";
												$contents .= "\t\t\t\t".$idxc.' => \''.addslashes($vc).'\','."\n";
											}
										}
										$contents .= "\t\t\t),\n";
									//} else if (($vb == '') || !is_numeric($vb)) {
									//	$contents .= "\t\t\t".$idxb.' => \''.addslashes($vb).'\','."\n";
									} else {
									//	$contents .= "\t\t\t".$idxb.' => '.$vb.','."\n";
										$contents .= "\t\t\t".$idxb.' => \''.addslashes($vb).'\','."\n";
									}
								}
								$contents .= "\t\t),\n";
							//} else if (($va == '') || !is_numeric($va)) {
							//	$contents .= "\t\t".$idxa.' => \''.addslashes($va).'\','."\n";
							} else {
							//	$contents .= "\t\t".$idxa.' => '.$va.','."\n";
								$contents .= "\t\t".$idxa.' => \''.addslashes($va).'\','."\n";
							}
						}
						$contents .= "\t),\n";
					//} else if (($v == '') || !is_numeric($v)) {
					//	$contents .= "\t".$idx.' => \''.addslashes($v).'\','."\n";
					} else {
					//	$contents .= "\t".$idx.' => '.$v.','."\n";
						$contents .= "\t".$idx.' => \''.addslashes($v).'\','."\n";
					}
				}
			}
		}

		if (!$is_object) {
			$contents .= ");\n";
		}
		$contents .= "\n".'?>';

		$f = @fopen($cache_file, 'wb');
		if (!$f) { return false; }

		$len = strlen($contents);
		@fwrite($f, $contents, $len);
		fclose($f);
		return true;
	}


	/*******************/
	/* GET CACHED FILE */
	/*******************/
	public function getCacheFile($name, $multilingual=true, $is_book=false) {
		$cache_dir = $is_book ? $this->path.'/cache/bk/' : $this->path.'/cache/'.$this->options['apiid'].'/';
		$cache_file = $cache_dir.$name.'.php';
		if ($multilingual) {
			if ($this->options['lang'] != '') { $cache_file = $cache_dir.$name.'.'.$this->options['lang'].'.php'; }
		}
		if (file_exists($cache_file)) { return $cache_file; }
		return '';
	}


	/*********************************/
	/* PARSE BASIC DATA XML RESPONSE */
	/*********************************/
	public function parseBasicData($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not load data! Please try again later.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$company = new stdClass;
		$company->cid = (int)$xmlDoc->cid;
		$company->title = (string)$xmlDoc->title;
		$company->country = (string)$xmlDoc->country;
		$company->city = (string)$xmlDoc->city;
		$company->address = (string)$xmlDoc->address;
		$company->postalcode = (string)$xmlDoc->postalcode;
		$company->phone = (string)$xmlDoc->phone;
		$company->mobile = (string)$xmlDoc->mobile;
		$company->fax = (string)$xmlDoc->fax;
		$company->email = (string)$xmlDoc->email;
		$company->website = (string)$xmlDoc->website;
		$company->logourl = (string)$xmlDoc->logourl;
		$company->tolerance = (int)$xmlDoc->tolerance;
		$company->extrahours = (int)$xmlDoc->extrahours;
		$company->seasontolerance = (string)$xmlDoc->seasontolerance;
		$company->tax = (string)$xmlDoc->tax;
		$company->taxispc = (int)$xmlDoc->taxispc;
		$company->samelocdisc = (int)$xmlDoc->samelocdisc;
		$company->bothlocsdisc = (int)$xmlDoc->bothlocsdisc;
		$company->minrentdays = (int)$xmlDoc->minrentdays;
		$company->maxrentdays = (int)$xmlDoc->maxrentdays;
		$company->terms = (string)$xmlDoc->terms;
		$company->surfee = (string)$xmlDoc->surfee;
		$company->rcancel = (int)$xmlDoc->rcancel;
		$company->rcanceldays = (int)$xmlDoc->rcanceldays;
		$company->acancelon = (int)$xmlDoc->acancelon;
		$company->acanceloff = (int)$xmlDoc->acanceloff;
		$company->preptime = (int)$xmlDoc->preptime;
		$company->fuelprices = (string)$xmlDoc->fuelprices;
		$company->refuelcost = (string)$xmlDoc->refuelcost;
		$company->nightstart = (string)$xmlDoc->nightstart;
		$company->nightend = (string)$xmlDoc->nightend;
		$company->formelements = (string)$xmlDoc->formelements;
		$company->imagesurl = (string)$xmlDoc->imagesurl;
		$company->fixedtime = isset($xmlDoc->fixedtime) ? (int)$xmlDoc->fixedtime : 0;//v1.6, not used in client
		$company->pickplushours = isset($xmlDoc->pickplushours) ? (int)$xmlDoc->pickplushours : 0;//v1.8

		$company->vtypes = (string)$xmlDoc->vtypes;

		$places = array();
		if (count($xmlDoc->places->children()) > 0) {
			foreach ($xmlDoc->places->children() as $country) {
				$code = (string)$country->code;
				$places[$code] = array();
				$places[$code]['name'] = (string)$country->name;
				$places[$code]['areas'] = array();
				if (count($country->areas->children()) > 0) {
					foreach ($country->areas->children() as $area) {
						$aid = (int)$area->aid;
						$title = (string)$area->title;
						$places[$code]['areas'][$aid] = array('title' => $title, 'locations' => array());
						if (count($area->locations->children()) > 0) {
							foreach ($area->locations->children() as $loc) {
								$lid = (int)$loc->lid;
								$title = (string)$loc->title;
								$latlong = (string)$loc->latlong;
								$places[$code]['areas'][$aid]['locations'][$lid] = array('title' => $title, 'latlong' => $latlong);
							}
						}
					}
				}
			}
		}

		$priceincludes = array();
		if (isset($xmlDoc->priceincludes)) {
			if (count($xmlDoc->priceincludes->children()) > 0) {
				foreach ($xmlDoc->priceincludes->children() as $prow) {
					$id = (string)$prow->id;
					$priceincludes[$id] = array();
					$priceincludes[$id]['vtype'] = (string)$prow->vtype;
					$priceincludes[$id]['title'] = (string)$prow->title;
					$priceincludes[$id]['description'] = (string)$prow->description;
				}
			}
		}

		$modelsfeatures = array();
		if (isset($xmlDoc->modelsfeatures)) {
			if (count($xmlDoc->modelsfeatures->children()) > 0) {
				foreach ($xmlDoc->modelsfeatures->children() as $vtype => $features) {
					if (count($features->children()) > 0) {
						$vtype = (string)$vtype;
						if (!isset($modelsfeatures[$vtype])) { $modelsfeatures[$vtype] = array(); }
						foreach ($features->children() as $feature) {
							$name = (string)$feature->name;
							$str = (string)$feature->attributes;
							$attrs = array();
							if ($str != '') { $attrs = json_decode($str, true); }
							$str = (string)$feature->options;
							$opts = array();
							if ($str != '') { $opts = json_decode($str, true); }

							$modelsfeatures[$vtype][$name] = array();
							$modelsfeatures[$vtype][$name]['type'] = (string)$feature->type;
							$modelsfeatures[$vtype][$name]['title'] = (string)$feature->title;
							$modelsfeatures[$vtype][$name]['description'] = (string)$feature->description;
							$modelsfeatures[$vtype][$name]['attributes'] = $attrs;
							$modelsfeatures[$vtype][$name]['options'] = $opts;
							$modelsfeatures[$vtype][$name]['ordering'] = (int)$feature->ordering;
						}
					}
				}
			}
		}

		$categories = array();
		if (isset($xmlDoc->categories)) {
			if (count($xmlDoc->categories->children()) > 0) {
				foreach ($xmlDoc->categories->children() as $vtype => $ctgs) {
					if (count($ctgs->children()) > 0) {
						$vtype = (string)$vtype;
						if (!isset($categories[$vtype])) { $categories[$vtype] = array(); }
						foreach ($ctgs->children() as $ctg => $data) {
							$name = (string)$ctg;
							$categories[$vtype][$name] = (string)$data;
						}
					}
				}
			}
		}

		$noworkhours = array();
		if (isset($xmlDoc->noworkhours)) {
			if (count($xmlDoc->noworkhours->children()) > 0) {
				foreach ($xmlDoc->noworkhours->children() as $nwhour) {
					$lid = (int)$nwhour->lid;
					if (!isset($noworkhours[$lid])) { $noworkhours[$lid] = array(); }
					$noworkhours[$lid][] = array(
						'title' => (string)$nwhour->title,
						'sdatetime' => (string)$nwhour->sdatetime,
						'edatetime' => (string)$nwhour->edatetime,
						'weekdays' => (string)$nwhour->weekdays
					);
				}
			}
		}

		$rentdurations = array();
		if (isset($xmlDoc->rentdurations)) {
			if (count($xmlDoc->rentdurations->children()) > 0) {
				foreach ($xmlDoc->rentdurations->children() as $rdur) {
					$lid = (int)$rdur->lid;
					if (!isset($rentdurations[$lid])) { $rentdurations[$lid] = array(); }
					$rentdurations[$lid][] = array(
						'sdate' => (string)$rdur->sdate,
						'edate' => (string)$rdur->edate,
						'minrentdays' => (int)$rdur->minrentdays,
						'maxrentdays' => (int)$rdur->maxrentdays
					);
				}
			}
		}

		return array(
			'company' => $company,
			'places' => $places,
			'priceincludes' => $priceincludes,
			'modelsfeatures' => $modelsfeatures,
			'categories' => $categories,
			'noworkhours' => $noworkhours,
			'rentdurations' => $rentdurations
		);
	}


	/************************/
	/* PARSE SEARCH RESULTS */
	/************************/
	public function parseSearchResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not load data! Please try again later.';
			return false;
		}
		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$results = new stdClass;
		$results->pickup = (string)$xmlDoc->pickup;
		$results->dropoff = (string)$xmlDoc->dropoff;
		$results->picklid = (int)$xmlDoc->picklid;
		$results->droplid = (int)$xmlDoc->droplid;
		$results->days = (int)$xmlDoc->days;
		$results->charge_days = (int)$xmlDoc->charge_days;
		$results->hours = (int)$xmlDoc->hours;
		$results->charge_hours = (int)$xmlDoc->charge_hours;
		$results->currency = (string)$xmlDoc->currency;
		$results->groups = array();
		if (count($xmlDoc->groups->children()) > 0) {
			foreach ($xmlDoc->groups->children() as $group) {
				$gid = (int)$group->gid;
				$results->groups[$gid] = array(
					'title' => (string)$group->title,
					'vtype' => (string)$group->vtype,
					'priceincludes' => (string)$group->priceincludes,
					'categories' => (string)$group->categories,
					'availvehicles' => (int)$group->availvehicles,
					'price' => array('dailyprices' => array()),
					'models' => array()
				);

				if (count($group->price->dailyprices->children()) > 0) {
					foreach ($group->price->dailyprices->children() as $dprice) {
						$zdate = (int)$dprice['zdate'];
						$results->groups[$gid]['price']['dailyprices'][$zdate] = (string)$dprice['amount'];
					}
				}

				$results->groups[$gid]['price']['rentalfee'] = (string)$group->price->rentalfee;
				$results->groups[$gid]['price']['extrahoursfee'] = (string)$group->price->extrahoursfee;
				$results->groups[$gid]['price']['afterhourspick'] = (string)$group->price->afterhourspick;
				$results->groups[$gid]['price']['afterhoursdrop'] = (string)$group->price->afterhoursdrop;
				$results->groups[$gid]['price']['pickupfee'] = (string)$group->price->pickupfee;
				$results->groups[$gid]['price']['dropofffee'] = (string)$group->price->dropofffee;
				$results->groups[$gid]['price']['total'] = (string)$group->price->total;
				$results->groups[$gid]['price']['earlybookdays'] = (int)$group->price->discounts->earlybookdays;
				$results->groups[$gid]['price']['earlybookpc'] = (int)$group->price->discounts->earlybookpc;
				$results->groups[$gid]['price']['fullamountdays'] = (int)$group->price->discounts->fullamountdays;
				$results->groups[$gid]['price']['fullamountpc'] = (int)$group->price->discounts->fullamountpc;
				$results->groups[$gid]['price']['fakepc'] = (int)$group->price->discounts->fakepc;
				$results->groups[$gid]['price']['cashpc'] = (int)$group->price->discounts->cashpc;

				if (count($group->models->children()) > 0) {
					foreach ($group->models->children() as $model) {
						$mid = (int)$model->mid;
						$images = array();
						if (isset($model->images)) {//IOSRT v1.4+
							$v = trim($model->images);
							if ($v != '') { $images = explode(',', $v); }
						}
						$lids = array();
						if (isset($model->lids)) {//IOSRT v1.4+
							$v = trim($model->lids);
							if ($v != '') { $lids = explode(',', $v); }
						}

						$results->groups[$gid]['models'][$mid] = array(
							'title' => (string)$model->title,
							'image' => (string)$model->image,
							'description' => (string)$model->description,
							'modelparams' => (string)$model->modelparams,
							'ribbon' => (string)$model->ribbon,
							'ribboncolor' => (string)$model->ribboncolor,
							'images' => $images,
							'lids' => $lids,
							'availvehicles' => (int)$model->availvehicles,
							'similar' => array()
						);

						if (count($model->similars->children()) > 0) {
							foreach ($model->similars->children() as $similar) {
								$smid = (int)$similar->mid;
								$results->groups[$gid]['models'][$mid]['similar'][$smid] = array(
									'title' => (string)$similar->title,
									'image' => (string)$similar->image
								);
							}
						}
					}
				}
			}
		}

		return $results;
	}


	/******************************/
	/* PARSE CHECK COUPON RESULTS */
	/******************************/
	public function parseCheckCouponResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not check coupon! Please try again later.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not check coupon. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not check coupon. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$results = new stdClass;
		$results->currency = (string)$xmlDoc->currency;
		$results->code = (string)$xmlDoc->code;
		$results->ispickdate = (int)$xmlDoc->ispickdate;
		$results->fromdate = (string)$xmlDoc->fromdate;
		$results->todate = (string)$xmlDoc->todate;
		$results->fromdays = (int)$xmlDoc->fromdays;
		$results->todays = (int)$xmlDoc->todays;
		$results->ispc = (int)$xmlDoc->ispc;
		$results->canbeused = (int)$xmlDoc->canbeused;
		$results->timesused = (int)$xmlDoc->timesused;
		$results->amount = (string)$xmlDoc->amount;

		return $results;
	}


	/****************************/
	/* PARSE BEGIN BOOK RESULTS */
	/****************************/
	public function parseBeginBookResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not load data! Please try again later.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$results = new stdClass;
		$results->pickup = (string)$xmlDoc->pickup;
		$results->dropoff = (string)$xmlDoc->dropoff;
		$results->picklid = (int)$xmlDoc->picklid;
		$results->droplid = (int)$xmlDoc->droplid;
		$results->picklocation = (string)$xmlDoc->picklocation;
		$results->picklatlong = (string)$xmlDoc->picklatlong;
		$results->pickaid = (int)$xmlDoc->pickaid;
		$results->pickarea = (string)$xmlDoc->pickarea;
		$results->pickcountry = (string)$xmlDoc->pickcountry;
		$results->pickcountryname = (string)$xmlDoc->pickcountryname;
		$results->droplocation = (string)$xmlDoc->droplocation;
		$results->droplatlong = (string)$xmlDoc->droplatlong;
		$results->dropaid = (int)$xmlDoc->dropaid;
		$results->droparea = (string)$xmlDoc->droparea;
		$results->dropcountry = (string)$xmlDoc->dropcountry;
		$results->currency = (string)$xmlDoc->currency;
		$results->days = (int)$xmlDoc->days;
		$results->charge_days = (int)$xmlDoc->charge_days;
		$results->hours = (int)$xmlDoc->hours;
		$results->charge_hours = (int)$xmlDoc->charge_hours;
		$results->rentalfee = (string)$xmlDoc->rentalfee;
		$results->extrahoursfee = (string)$xmlDoc->extrahoursfee;
		$results->afterhourspick = (string)$xmlDoc->afterhourspick;
		$results->afterhoursdrop = (string)$xmlDoc->afterhoursdrop;
		$results->pickupfee = (string)$xmlDoc->pickupfee;
		$results->dropofffee = (string)$xmlDoc->dropofffee;
		$results->earlybookpc = (int)$xmlDoc->earlybookpc;
		$results->earlybookdays = (int)$xmlDoc->earlybookdays;
		$results->fullamountpc = (int)$xmlDoc->fullamountpc;
		$results->fullamountdays = (int)$xmlDoc->fullamountdays;
		$results->preselextras = (string)$xmlDoc->preselextras;//v1.3+, pre-selected extras total amount
		$results->cashpc = (int)$xmlDoc->cashpc;
		$results->vehicle = array();
		$results->vehicle['vtype'] = (string)$xmlDoc->vehicle->vtype;
		$results->vehicle['mid'] = (int)$xmlDoc->vehicle->mid;
		$results->vehicle['model'] = (string)$xmlDoc->vehicle->model;
		$results->vehicle['image'] = (string)$xmlDoc->vehicle->image;
		$results->vehicle['description'] = (string)$xmlDoc->vehicle->description;
		$results->vehicle['modelparams'] = (string)$xmlDoc->vehicle->modelparams;
		$results->vehicle['modelavailvehicles'] = (int)$xmlDoc->vehicle->modelavailvehicles;
		$results->vehicle['gid'] = (int)$xmlDoc->vehicle->gid;
		$results->vehicle['group'] = (string)$xmlDoc->vehicle->group;
		$results->vehicle['preptime'] = (int)$xmlDoc->vehicle->preptime;
		$results->vehicle['priceincludes'] = (string)$xmlDoc->vehicle->priceincludes;
		$results->vehicle['categories'] = (string)$xmlDoc->vehicle->categories;
		$results->extras = array();
		if (count($xmlDoc->extras->children()) > 0) {
			foreach ($xmlDoc->extras->children() as $extra) {
				$xid = (int)$extra->xid;
				$xgid = isset($extra->xgid) ? (int)$extra->xgid : 0;//IOS Rentals 1.8
				$priceduration = isset($extra->priceduration) ? trim($extra->priceduration) : '';//IOS Rentals 1.8
				$required = isset($extra->required) ? (int)$extra->required : 0;//IOS Rentals 2.0

				$results->extras[$xid] = array(
					'title' => (string)$extra->title,
					'description' => (string)$extra->description,
					'price' => (string)$extra->price,
					'priceperday' => (int)$extra->priceperday,
					'priceformula' => (string)$extra->priceformula,
					'maxitems' => (int)$extra->maxitems,
					'freeitems' => (int)$extra->freeitems,
					'preselitems' => (int)$extra->preselitems,
					'seasons' => (string)$extra->seasons,
					'identifier' => (string)$extra->identifier,
					'maxprice' => (string)$extra->maxprice,
					'xgid' => $xgid,
					'priceduration' => $priceduration,
					'required' => $required
				);
			}
		}

		$results->extrasgroups = array();
		if (isset($xmlDoc->extrasgroups)) {//IOS Rentals 1.8
			if (count($xmlDoc->extrasgroups->children()) > 0) {
				foreach ($xmlDoc->extrasgroups->children() as $exgrp) {
					$xgid = (int)$exgrp->xgid;
					$results->extrasgroups[$xgid] = array(
						'title' => (string)$exgrp->title,
						'singleselect' => (int)$exgrp->singleselect
					);
				}
			}
		}

		$results->paymods = array();
		if (count($xmlDoc->paymods->children()) > 0) {
			foreach ($xmlDoc->paymods->children() as $paymod) {
				$pid = (int)$paymod->pid;
				$results->paymods[$pid] = array(
					'name' => (string)$paymod->name,
					'title' => (string)$paymod->title,
					'description' => (string)$paymod->description,
					'image' => (string)$paymod->image,
					'isonline' => (int)$paymod->isonline
				);
				if ($paymod->creditcards) { $results->paymods[$pid]['creditcards'] = (string)$paymod->creditcards; }
			}
		}

		$results->bookelements = array();
		if (count($xmlDoc->bookelements->children()) > 0) {
			foreach ($xmlDoc->bookelements->children() as $item) {
				$attrs = $item->attributes();
				if (!isset($attrs['name'])) { continue; }
				if (!isset($attrs['show'])) { continue; }
				if (!isset($attrs['type'])) { continue; }
				$name = trim($attrs['name']);
				$type =  trim($attrs['type']);
				$show = (int)$attrs['show'];
				if ($type == '') { $type = 'text'; }
				$title = (string)$item->title;
				$tip = (string)$item->tip;
				$options = (string)$item->options;
				if (($name == '') || ($show == 0) || ($show == 3) || (trim($title) == '')) { continue; }
				$results->bookelements[$name] = array('type' => $type, 'show' => $show, 'title' => $title, 'tip' => $tip, 'options' => $options);
			}
		}
		//Just in case: ofirstname, olastname, oemail are always required!
		if (!isset($results->bookelements['ofirstname'])) {
			$results->bookelements['ofirstname'] = array('type' => 'text', 'show' => 2, 'title' => 'Firstname', 'tip' => '', 'options' => '');
		}
		if (!isset($results->bookelements['olastname'])) {
			$results->bookelements['olastname'] = array('type' => 'text', 'show' => 2, 'title' => 'Lastname', 'tip' => '', 'options' => '');
		}
		if (!isset($results->bookelements['oemail'])) {
			$results->bookelements['oemail'] = array('type' => 'text', 'show' => 2, 'title' => 'E-mail', 'tip' => '', 'options' => '');
		}

		return $results;
	}


	/**********************/
	/* PARSE BOOK RESULTS */
	/**********************/
	public function parseBookResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not process rental! Please try again.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not process rental. Invalid response. Please try again.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not process rental. Invalid response. Please try again.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$msg = '';
				if (isset($xmlDoc->message)) { $msg = (string)$xmlDoc->message; } 
				if (trim($msg) == '') { $msg = 'Saving booking failed. Please try again.'; }
				$this->errormsg = $msg;
				return false;
			}
		}

		$results = array();//array because we will save it into cache
		$results['currency'] = (string)$xmlDoc->currency;
		$results['usercurrency'] = (string)$xmlDoc->usercurrency;
		$results['usercurrencyrate'] = (string)$xmlDoc->usercurrencyrate;
		$results['rescode'] = (string)$xmlDoc->rescode;
		$results['status'] = (int)$xmlDoc->status;
		$results['pickdate'] = (string)$xmlDoc->pickdate;
		$results['dropdate'] = (string)$xmlDoc->dropdate;
		$results['numdays'] = (int)$xmlDoc->numdays;
		$results['extrahours'] = (int)$xmlDoc->extrahours;
		$results['picklid'] = (int)$xmlDoc->picklid;
		$results['droplid'] = (int)$xmlDoc->droplid;
		$results['vtype'] = (string)$xmlDoc->vtype;
		$results['picklocation'] = (string)$xmlDoc->picklocation;
		$results['pickarea'] = (string)$xmlDoc->pickarea;
		$results['pickcountryname'] = (string)$xmlDoc->pickcountryname;
		$results['droplocation'] = (string)$xmlDoc->droplocation;
		$results['droparea'] = (string)$xmlDoc->droparea;
		$results['dropcountryname'] = (string)$xmlDoc->dropcountryname;
		$results['ofirstname'] = (string)$xmlDoc->ofirstname;
		$results['olastname'] = (string)$xmlDoc->olastname;
		$results['oemail'] = (string)$xmlDoc->oemail;
		$results['totalprice'] = (string)$xmlDoc->totalprice;
		$results['feeamount'] = (string)$xmlDoc->feeamount;
		$results['pid'] = (int)$xmlDoc->pid;
		$results['paymod'] = (string)$xmlDoc->paymod;
		$results['vehiclemodel'] = (string)$xmlDoc->vehiclemodel;
		$results['vehiclegroup'] = (string)$xmlDoc->vehiclegroup;
		$results['vehicleimage'] = (string)$xmlDoc->vehicleimage;
		$results['vehiclepriceincludes'] = (string)$xmlDoc->vehiclepriceincludes;
		$results['vehiclecategories'] = (string)$xmlDoc->vehiclecategories;
		$results['formurl'] = '';
		$results['paymoduledata'] = array();
		$results['paymoduledata']['page'] = (int)$xmlDoc->paymoduledata->page;
		$results['paymoduledata']['pid'] = (int)$xmlDoc->paymoduledata->pid;
		$results['paymoduledata']['paymod'] = (string)$xmlDoc->paymoduledata->paymod;
		$results['paymoduledata']['name'] = (string)$xmlDoc->paymoduledata->name;
		$results['paymoduledata']['title'] = (string)$xmlDoc->paymoduledata->title;
		$results['paymoduledata']['description'] = (string)$xmlDoc->paymoduledata->description;
		$results['paymoduledata']['image'] = (string)$xmlDoc->paymoduledata->image;
		$results['paymoduledata']['isonline'] = (int)$xmlDoc->paymoduledata->isonline;
		$results['paymoduledata']['formaction'] = (string)$xmlDoc->paymoduledata->formaction;
		$results['paymoduledata']['formitems'] = array();
		$results['paymoduledata']['listtop'] = (string)$xmlDoc->paymoduledata->listtop;
		$results['paymoduledata']['listbottom'] = (string)$xmlDoc->paymoduledata->listbottom;
		$results['paymoduledata']['listitems'] = array();

		//$formurl = $this->makeFormURL();//THIS IS WRONG AS WE ARE IN AN AJAX REQUEST (links to $this->options['clienturl'] instead)
		if (isset($_POST['ubase'])) {
			$ubase = trim(base64_decode($_POST['ubase']));
			if ($ubase != '') {
				$n = strpos($ubase, '?');
				if ($n !== false) { $ubase = substr($ubase, 0, $n); }
				if (filter_var($ubase, FILTER_VALIDATE_URL)) { $results['formurl'] = $ubase; }
			}
		} else if (isset($_SERVER['HTTP_REFERER'])) {
			$ubase = trim($_SERVER['HTTP_REFERER']);
			if ($ubase != '') {
				$n = strpos($ubase, '?');
				if ($n !== false) { $ubase = substr($ubase, 0, $n); }
				if (filter_var($ubase, FILTER_VALIDATE_URL)) { $results['formurl'] = $ubase; }
			}
		}
		if ($results['formurl'] == '') { $results['paymoduledata']['page'] = 0; }//something is wrong..., no online processor

		$success = $results['formurl'].'?step=3&rescode='.$results['rescode'];
		$paysuccess = $results['formurl'].'?step=4&rescode='.$results['rescode'];
		$payfail = $results['formurl'].'?step=5&rescode='.$results['rescode'];
		if (count($xmlDoc->paymoduledata->formitems->children()) > 0) {
			foreach ($xmlDoc->paymoduledata->formitems->children() as $formitem) {
				$v = (string)$formitem->value;
				if ($v == '{FORMURL}') {
					$v = $results['formurl'];
				} else if ($v == '{SUCCESSURL}') {
					$v = $success;
				} else if ($v == '{PAYSUCCESSURL}') {
					$v = $paysuccess;
				} else if ($v == '{PAYFAILURL}') {
					$v = $payfail;
				}
				$results['paymoduledata']['formitems'][] = array(
					'iname' => (string)$formitem->name,
					'ivalue' => $v
				);
			}
		}

		if (count($xmlDoc->paymoduledata->listitems->children()) > 0) {
			foreach ($xmlDoc->paymoduledata->listitems->children() as $listitem) {
				$results['paymoduledata']['listitems'][] = array(
					'ititle' => (string)$listitem->title,
					'itext' => (string)$listitem->text
				);
			}
		}

		return $results;
	}


	/***********************/
	/* PARSE FLEET RESULTS */
	/***********************/
	public function parseFleetResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not load data! Please try again later.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$results = new stdClass;
		$results->currency = (string)$xmlDoc->currency;
		$results->usercurrency = (string)$xmlDoc->usercurrency;
		$results->usercurrencyrate = (int)$xmlDoc->usercurrencyrate;
		$results->models = array();

		if (count($xmlDoc->fleet->children()) > 0) {
			foreach ($xmlDoc->fleet->children() as $model) {
				$mid = (int)$model->mid;
				$images = array();
				if (isset($model->images)) {//IOSRT v1.4+
					$v = trim($model->images);
					if ($v != '') { $images = explode(',', $v); }
				}
				$lids = array();
				if (isset($model->lids)) {//IOSRT v1.4+
					$v = trim($model->lids);
					if ($v != '') { $lids = explode(',', $v); }
				}
				$results->models[$mid] = array(
					'mid' => $mid,
					'gid' => (int)$model->gid,
					'title' => (string)$model->title,
					'group' => (string)$model->group,
					'vtype' => (string)$model->vtype,
					'image' => (string)$model->image,
					'description' => (string)$model->description,
					'modelparams' => (string)$model->modelparams,
					'ribbon' => (string)$model->ribbon,
					'ribboncolor' => (string)$model->ribboncolor,
					'images' => $images,
					'lids' => $lids,
					'priceincludes' => (string)$model->priceincludes,
					'categories' => (string)$model->categories,
					'minprice' => (string)$model->minprice
				);
			}
		}

		return $results;
	}


	/*************************************/
	/* PARSE PAYMENT NOTIFICATION RESULT */
	/*************************************/
	public function parsePayNotifyResult($xml) {
		if (!$xml || ($xml == '')) { return -1; }
		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) { return -1; }
		if ($xmlDoc->getName() != 'iosrentals') { return -1; }

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return 0;
			}
		}

		$result = (int)$xmlDoc->notifysuccess;

		return $result;
	}


	/******************************************/
	/* PARSE FLEET MODEL RESULTS (MODEL PAGE) */
	/******************************************/
	public function parseModelResults($xml) {
		if (!$xml || ($xml == '')) {
			$this->errormsg = 'Could not load data! Please try again later.';
			return false;
		}

		libxml_use_internal_errors(true);
		$xmlDoc = simplexml_load_string($xml, 'SimpleXMLElement');
		if (!$xmlDoc) {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}
		if ($xmlDoc->getName() != 'iosrentals') {
			$this->errormsg = 'Could not load data. Invalid response.';
			return false;
		}

		if (isset($xmlDoc->error)) {
			if (intval($xmlDoc->error) == 1) {
				$txt = '';
				if ($this->options['showerrorcodes'] == 1) {
					if (isset($xmlDoc->status)) {
						if (trim($xmlDoc->status) != '') { $txt .= trim($xmlDoc->status).': '; }
					}
				}
				if (isset($xmlDoc->message)) { $txt .= (string)$xmlDoc->message; } else { $txt .= 'Request failed.'; }
				$this->errormsg = $txt;
				return false;
			}
		}

		$images = array();
		if (isset($xmlDoc->model->images)) {//IOSRT v1.4+
			$v = trim($xmlDoc->model->images);
			if ($v != '') { $images = explode(',', $v); }
		}
		$lids = array();
		if (isset($xmlDoc->model->lids)) {//IOSRT v1.4+
			$v = trim($xmlDoc->model->lids);
			if ($v != '') { $lids = explode(',', $v); }
		}

		$results = new stdClass;
		$results->currency = (string)$xmlDoc->currency;
		$results->usercurrency = (string)$xmlDoc->usercurrency;
		$results->usercurrencyrate = (int)$xmlDoc->usercurrencyrate;
		$results->model = array(
			'mid' => (int)$xmlDoc->model->mid,
			'gid' => (int)$xmlDoc->model->gid,
			'title' => (string)$xmlDoc->model->title,
			'group' => (string)$xmlDoc->model->group,
			'vtype' => (string)$xmlDoc->model->vtype,
			'image' => (string)$xmlDoc->model->image,
			'description' => (string)$xmlDoc->model->description,
			'modelparams' => (string)$xmlDoc->model->modelparams,
			'ribbon' => (string)$xmlDoc->model->ribbon,
			'ribboncolor' => (string)$xmlDoc->model->ribboncolor,
			'images' => $images,
			'lids' => $lids,
			'priceincludes' => (string)$xmlDoc->model->priceincludes,
			'categories' => (string)$xmlDoc->model->categories,
			'minprice' => (string)$xmlDoc->model->minprice
		);

		return $results;
	}


	/**********************/
	/* CLEAR OLD BOOKINGS */
	/**********************/
	public function clearOldBookings() {
		$cache_dir = $this->path.'/cache/bk/';
		$files = glob($cache_dir.'*.php');
		if (!$files) { return; }
		$ts = time();
		foreach ($files as $file) {
			$tsf = filemtime($file);
			if ($ts - $tsf > 432000) {//5 days
				unlink($file);
			}
		}
	}


	/**************************/
	/* ADD INPUT TEXT ELEMENT */
	/**************************/
	public function addText($name, $value, $label, $attrs=array(), $full=true) {
		$type = 'text';
		$tip = '';
		$labelclass = 'iosrt_label';
		$sideclass = 'iosrt_labelside';
		$attributes = array();
		$attributes['id'] = 'iosrt_'.$name;
		$attributes['placeholder'] = $label;
		$attributes['title'] = $label;
		$attributes['class'] = 'iosrt_text';
		if ($attrs) {
			foreach ($attrs as $k => $v) {
				if ($k == 'type') { $type = $v; continue; }
				if ($k == 'tip') { $tip = $v; continue; }
				if ($k == 'labelclass') { $labelclass = $v; continue; }
				if ($k == 'sideclass') { $sideclass = $v; continue; }
				$attributes[$k] = $v;
			}
		}

		$html = '';
		if ($full) { $html .= '<div class="iosrt_formrow">'."\n"; }
		$html .= '<label class="'. $labelclass.'" for="'.$attributes['id'].'">'.$label."</label>\n";
		if ($full) { $html .= '<div class="'.$sideclass.'">'; }
		$html .= '<input type="'.$type.'" name="'.$name.'" value="'.$value.'" ';
		foreach ($attributes as $k => $v) {
			if ($v == '') { continue; }
			$html .= $k.'="'.$v.'" ';
		}
		$html .= '/>';
		if ($tip != '') { $html .= '<div class="iosrt_tip">'.$tip."</div>\n"; }
		if ($full) { $html .= "</div>\n</div>\n"; }

		return $html;
	}


	/******************************/
	/* ADD DROP DOWN SELECT FIELD */
	/******************************/
	public function addSelect($name, $label='', $selected=null, $options=array(), $attrs=array(), $full=true) {
		$attributes = array('id' => 'iosrt_'.$name, 'class' => 'iosrt_select');
		$tip = '';
		$labelclass = 'iosrt_label';
		$sideclass = 'iosrt_labelside';
		if ($attrs) {
			foreach ($attrs as $k => $v) {
				if ($k == 'tip') { $tip = $v; continue; }
				if ($k == 'labelclass') { $labelclass = $v; continue; }
				if ($k == 'sideclass') { $sideclass = $v; continue; }
				$attributes[$k] = $v;
			}
		}

		$html = '';
		if ($full) { $html .= '<div class="iosrt_formrow">'."\n"; }
		$html .= '<label class="'. $labelclass.'" for="'.$attributes['id'].'">'.$label."</label>\n";
		if ($full) { $html .= '<div class="'.$sideclass.'">'; }
		$html .= '<select name="'.$name.'"';
		foreach ($attributes as $k => $v) {
			if ($v == '') { continue; }
			if ($k == 'name') { continue; }
			$html .= ' '.$k.'="'.$v.'"';
		}
		$html .= ">\n";
		if (is_array($options)) {
			if (count($options) > 0) {
				foreach ($options as $option) {
					$sel = ($option[0] == $selected) ? ' selected="selected"' : '';
					$html .= '<option value="'.$option[0].'"'.$sel.'>'.$option[1]."</option>\n";
				}
			}
		}
		$html .= "</select>\n";
		if ($tip != '') { $html .= '<div class="iosrt_tip">'.$tip."</div>\n"; }
		if ($full) { $html .= "</div>\n</div>\n"; }

		return $html;
	}


}

?>