<?php 
/**
* @version		$Id: iosrt-wp.php 1487 2020-08-18 16:08:23Z IOS $
* @package		Component IOS Reservations
* @author		Ioannis Sannos ( https://www.isopensource.com )
* @copyright	Copyright (c) 2008-2020 Is Open Source (https://www.isopensource.com). All rights reserved.
* @license		Commercial
* @description 	Online booking system
**************************************/

if (!defined('ABSPATH')) { die('Direct access to this location is not allow!' ); }


class iosRentalsWP {

	private $plugindir = '';
	private $pluginurl = '';
	private $apiidok = true;
	private $client = null;
	private $version = '2.0.0';


	/********************/
	/* MAGIC CONTRUCTOR */
	/********************/
	public function __construct($plugindir, $pluginurl, $version) {
		$this->plugindir = $plugindir;
		$this->pluginurl = $pluginurl;
		$this->version = $version;
	}


	/*************/
	/* ADD HOOKS */
	/*************/
	public function makeAll() {
		add_action('admin_menu', array($this, 'makeAdminMenu'));
		add_action('admin_enqueue_scripts', array($this, 'addAdminCSSJS'));
		add_action('wp_head', array($this, 'prepareClient'));
		add_shortcode('iosrtapi', array($this, 'parseShortCode'));// [iosrtapi apiid="X"]
	}


	/********************************/
	/* PREPARE CLIENT AND MAKE HEAD */
	/********************************/
	public function prepareClient() {
		if (is_admin()) { return; }
		if (!is_single() && !is_page()) { return; }
		$apiid = '';

		global $post;
		if (has_shortcode($post->post_content, 'iosrtapi')) {
			$regex = "#\[iosrtapi\s*(.*?)\]#s";
			preg_match_all($regex, $post->post_content, $matches, PREG_PATTERN_ORDER);
			if ($matches) {
				if (isset($matches[1][0])) {
					$str = trim($matches[1][0]);
					$str = str_replace('&quot;', '', $str);
					$str = str_replace('"', '', $str);
					$apiid = trim(str_replace('apiid=', '', $str));
				}
			}
		}

		if ($apiid == '') { $this->apiidok = false; return; }

		if (!defined('_IOSRTAPICLIENT_')) { define('_IOSRTAPICLIENT_', 1); }
		$cfgoptions = array();
		$mldata = $this->wpLanguage($apiid);
		if ($mldata['gui'] != '') { $cfgoptions['guilang'] = $mldata['gui']; }
		if ($mldata['server'] != '') { $cfgoptions['lang'] = $mldata['server']; }
		if ($mldata['action'] != '') { $cfgoptions['formaction'] = $mldata['action']; }
		if ($mldata['fleetpage'] != '') { $cfgoptions['fleetpage'] = $mldata['fleetpage']; }
		unset($mldata);

		require_once($this->plugindir.'/client/client.class.php');
		$this->client = new iosrtApiClient($apiid, $cfgoptions);
		if ($this->client->getError() != '') { $this->apiidok = false; return; }

		define('_IOSRTAPIID_', $apiid);

		$font = $this->client->getOption('loadfont', '');
		if ($font != '') {
			echo '<link rel="stylesheet" id="iosrtapi-font-css" href="'.$this->pluginurl.'/client/css/'.$font.'.css?ver='.$this->version.'" type="text/css" media="all" />'."\n";
		}

		$cssfile = 'client.css';
		if ($this->client->getLang('DIR') == 'rtl') { $cssfile = 'client-rtl.css'; }
		echo '<link rel="stylesheet" id="iosrtapi-client-css" href="'.$this->pluginurl.'/client/css/'.$cssfile.'?ver='.$this->version.'" type="text/css" media="all" />'."\n";
		echo '<link rel="stylesheet" id="iosrtapi-flatpickr-css" href="'.$this->pluginurl.'/client/css/flatpickr.min.css?ver='.$this->version.'" type="text/css" media="all" />'."\n";
		echo '<link rel="stylesheet" id="iosrtapi-swiper-css" href="'.$this->pluginurl.'/client/css/swiper.min.css?ver='.$this->version.'" type="text/css" media="all" />'."\n";

		$step = 0;
		if (isset($_POST['step'])) {
			$step = (int)$_POST['step'];
		} else if (isset($_GET['step'])) {
			$step = (int)$_GET['step'];
		} else {
			$step = 0;
		}
		if (($step == 3) || ($step == 4)) {//thanks page, success / payment success
			echo '<link rel="stylesheet" id="iosrtapi-print-css" href="'.$this->pluginurl.'/client/css/print.css?ver='.$this->version.'" type="text/css" media="all" />'."\n";
		}
		echo '<script src="'.$this->pluginurl.'/client/js/tippy.min.js?ver='.$this->version.'"></script>'."\n";
		echo '<script src="'.$this->pluginurl.'/client/js/flatpickr.min.js?ver='.$this->version.'"></script>'."\n";
		echo '<script src="'.$this->pluginurl.'/client/js/swiper.min.js?ver='.$this->version.'"></script>'."\n";
		$lng = strtolower($this->client->getGUILang());
		if (file_exists($this->plugindir.'/client/js/flatpickri18n/'.$lng.'.js')) {
			echo '<script src="'.$this->pluginurl.'/client/js/flatpickri18n/'.$lng.'.js?ver='.$this->version.'"></script>'."\n";
		}
		echo '<script src="'.$this->pluginurl.'/client/js/client.js?ver='.$this->version.'"></script>'."\n";

		$calendar = $this->client->calendarData();
		echo '<script>'."\n";
		if ($calendar['disranges']) {
			echo 'var iosRTCalDisRanges = '.json_encode($calendar['disranges']).';'."\n";
		} else {
			echo 'var iosRTCalDisRanges = {};'."\n";
		}
		if ($calendar['nwranges']) {
			echo 'var iosRTCalHourRanges = '.json_encode($calendar['nwranges']).';'."\n";
		} else {
			echo 'var iosRTCalHourRanges = {};'."\n";
		}
		if ($calendar['rentdranges']) {
			echo 'var iosRTCalRentdRanges = '.json_encode($calendar['rentdranges']).';'."\n";
		} else {
			echo 'var iosRTCalRentdRanges = {};'."\n";
		}
		echo 'var iosRTCalminrentdays = '.$calendar['minrentdays'].';'."\n";
		echo 'var iosRTCalmaxrentdays = '.$calendar['maxrentdays'].';'."\n";
		echo 'var iosRTCalpickupdayofweek = '.$calendar['pickupdayofweek'].';'."\n";
		echo 'var iosRTCaldropoffdayofweek = '.$calendar['dropoffdayofweek'].';'."\n";
		echo 'var iosRThidedrop = '.$calendar['hidedrop'].';'."\n";
		echo 'var iosRTpickplushours = '.$calendar['pickplushours'].';'."\n";
		echo "</script>\n";
	}


	/*******************************************/
	/* WORDPRESS LANGUAGE (WPML) AND IOSRT API */
	/*******************************************/
	private function wpLanguage($apiid) {
		$mldata = array('gui' => '', 'server' => '', 'action' => '', 'fleetpage' => '');
		//if (!function_exists('icl_object_id')) { return $mldata; }//WORDPRESS WPML
		if (!defined('ICL_LANGUAGE_CODE')) { return $mldata; }
		if (!file_exists($this->plugindir.'/client/apiconfig.php')) { return $mldata; }
		include($this->plugindir.'/client/apiconfig.php');
		if (!isset($apiconfig) || !is_array($apiconfig) || (count($apiconfig) == 0)) { return $mldata; }

		$guilang = 'ZZZZZZZZZZZZ';
		$lang = 'ZZZZZZZZZZZZ';
		$formaction = '';
		$fleetpage = '';
		foreach ($apiconfig as $cfg) {
			if ($cfg['apiid'] != $apiid) { continue; }
			$guilang = $cfg['guilang'];
			$lang = $cfg['lang'];
			$formaction = $cfg['formaction'];
			$fleetpage = $cfg['fleetpage'];
			break;
		}

		if ($guilang != '') { return $mldata; }
		
		$curlang = ICL_LANGUAGE_CODE;//apply_filters('wpml_current_language', null);//or ICL_LANGUAGE_CODE or get_bloginfo("language");
		$curlang_low = strtolower($curlang);
		$testlang = '';
		if (strpos($curlang_low, '_') !== false) {
			$p = explode('_', $curlang_low);
			if ((count($p) == 2) && (strlen($p[0]) == 2)) { $testlang = $p[0]; }
		} else if (strpos($curlang_low, '-') !== false) {
			$p = explode('-', $curlang_low);
			if ((count($p) == 2) && (strlen($p[0]) == 2)) { $testlang = $p[0]; }
		} else if (strlen($curlang_low) == 2) {
			$testlang = $curlang_low;
		}
		if (($testlang != '') && file_exists($this->plugindir.'/client/language/'.$testlang.'.php')) {
			$mldata['gui'] = $testlang;
			if ($lang == '') { $mldata['server'] = $testlang; }
			if ($formaction != '') {
				if (stripos($formaction, '(ML/)') !== false) {
					$wpml_options = get_option('icl_sitepress_settings');
					$default_lang = strtolower($wpml_options['default_language']);
					if ($curlang_low == $default_lang) {
						$mldata['action'] = str_replace('(ML/)', '', $formaction);
					} else {
						$mldata['action'] = str_replace('(ML/)', $curlang.'/', $formaction);
					}
				}
			}
			if ($fleetpage != '') {
				if (stripos($fleetpage, '(ML/)') !== false) {
					$wpml_options = get_option('icl_sitepress_settings');
					$default_lang = strtolower($wpml_options['default_language']);
					if ($curlang_low == $default_lang) {
						$mldata['fleetpage'] = str_replace('(ML/)', '', $fleetpage);
					} else {
						$mldata['fleetpage'] = str_replace('(ML/)', $curlang.'/', $fleetpage);
					}
				}
			}
		}
		return $mldata;
	}


	/*******************/
	/* PARSE SHORTCODE */
	/*******************/
	public function parseShortCode($atts) {//[iosrtapi apiid="X"]
		if (!$this->apiidok) { return ''; }

		$a = shortcode_atts(array('apiid' => ''), $atts);
		$apiid = trim($a['apiid']);
		if ($apiid == '') { return ''; }
		if (!is_single() && !is_page()) { return ''; }

		if (!$this->client) {//try to re-load
			if (!defined('_IOSRTAPIID_')) { return ''; }
			if (!defined('_IOSRTAPICLIENT_')) { define('_IOSRTAPICLIENT_', 1); }
			require_once($this->plugindir.'/client/client.class.php');
			$cfgoptions = array();
			$this->client = new iosrtApiClient(_IOSRTAPIID_, $cfgoptions);
		}
		$html = $this->client->init();

		return $html;
	}


	/*******************/
	/* MAKE ADMIN MENU */
	/*******************/
	public function makeAdminMenu() {
		add_menu_page(__('IOSRT APIs','iosrt-menu'), __('IOSRT APIs','iosrt-menu'), 'manage_options', 'iosrt-list-apis', array($this, 'iosrtConfigAPIs'), 'dashicons-admin-generic', 65);
	}


	/********************************/
	/* ADD CSS/JS FOR ADMIN SECTION */
	/********************************/
	public function addAdminCSSJS() {
        wp_enqueue_style('iosrt_wp_admin_css', plugins_url('wp/iosrt-wp.css', __FILE__));
        wp_enqueue_script('iosrt_wp_admin_js', plugins_url('wp/iosrt-wp.js', __FILE__));
	}


	/*****************************/
	/* DISPLAY API SETTINGS FORM */
	/*****************************/
	public function iosrtConfigAPIs() {
		if (!current_user_can('manage_options')) {
			wp_die( __('You do not have sufficient permissions to access this page.') );
			return;//just in case...
		}

		$settings_saved = -1;
		$errormsg = '';
		if (isset($_POST['iosrtapisbt'])) {
			$newconfigs = array();

			for ($i = 0; $i < 4; $i++) {
				$idx = 'serverurl'.$i;
				$serverurl = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'apiid'.$i;
				$apiid = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'prkey'.$i;
				$prkey = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if (($serverurl == '') || ($apiid == '') || ($prkey == '')) { continue; }
				if (!filter_var($serverurl, FILTER_VALIDATE_URL)) {
					$errormsg = 'Provided server URL is invalid!';
					break;
				}

				$idx = 'guilang'.$i;
				$guilang = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'lang'.$i;
				$lang = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'currency'.$i;
				$currency = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'cachetime'.$i;
				$cachetime = isset($_POST[$idx]) ? (int)$_POST[$idx] : 14400;
				if ($cachetime < 1800) { $cachetime = 14400; }
				$idx = 'loadfont'.$i;
				$loadfont = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'modelfeaticons'.$i;
				$modelfeaticons = isset($_POST[$idx]) ? (int)$_POST[$idx] : 6;
				$idx = 'modelprice'.$i;
				$modelprice = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'applyvat'.$i;
				$applyvat = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'vatnote'.$i;
				$vatnote = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'rentalduration'.$i;
				$rentalduration = isset($_POST[$idx]) ? (int)$_POST[$idx] : 0;
				$idx = 'imagewidth'.$i;
				$imagewidth = isset($_POST[$idx]) ? (int)$_POST[$idx] : 240;
				$idx = 'pickupday'.$i;
				$pickupday = isset($_POST[$idx]) ? (int)$_POST[$idx] : -1;
				$idx = 'dropoffday'.$i;
				$dropoffday = isset($_POST[$idx]) ? (int)$_POST[$idx] : -1;
				$idx = 'pickuptime'.$i;
				$pickuptime = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'dropofftime'.$i;
				$dropofftime = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'hidedrop'.$i;
				$hidedrop = isset($_POST[$idx]) ? (int)$_POST[$idx] : 0;
				$idx = 'titlepos'.$i;
				$titlepos = isset($_POST[$idx]) ? (int)$_POST[$idx] : 0;
				$idx = 'modellocs'.$i;
				$modellocs = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'roundprices'.$i;
				$roundprices = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'lids'.$i;
				$lids = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'showerrorcodes'.$i;
				$showerrorcodes = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'formaction'.$i;
				$formaction = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if ($formaction != '') {
					if (!filter_var($formaction, FILTER_VALIDATE_URL)) {//Note: use "(ML/)" keyword for multi-lingual urls/WPML. Example: http://www.example.com/(ML/)testme/
						$errormsg = 'Provided form action URL is invalid!';
						break;
					}
				}
				$idx = 'fleetpage'.$i;
				$fleetpage = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if ($fleetpage != '') {
					if (!filter_var($fleetpage, FILTER_VALIDATE_URL)) {//Note: use "(ML/)" keyword for multi-lingual urls/WPML. Example: http://www.example.com/(ML/)testme/
						$errormsg = 'Provided fleet page URL is invalid!';
						break;
					}
				}
				$idx = 'unifiedmoto'.$i;
				$unifiedmoto = isset($_POST[$idx]) ? (int)$_POST[$idx] : 0;
				$idx = 'dropsameaspick'.$i;
				$dropsameaspick = isset($_POST[$idx]) ? (int)$_POST[$idx] : 0;
				$idx = 'sidecolumn'.$i;
				$sidecolumn = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				$idx = 'formtext1'.$i;
				$formtext1 = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if ($formtext1 == '') {
					$idx = 'formtextc1'.$i;
					$formtext1 = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				}
				$idx = 'formtext2'.$i;
				$formtext2 = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if ($formtext2 == '') {
					$idx = 'formtextc2'.$i;
					$formtext2 = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				}
				$idx = 'showcoupon'.$i;
				$showcoupon = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'showinvoice'.$i;
				$showinvoice = isset($_POST[$idx]) ? (int)$_POST[$idx] : 1;
				$idx = 'vtype'.$i;
				$vtype = isset($_POST[$idx]) ? trim($_POST[$idx]) : '';
				if ($vtype != '') {
					$vtypes = array('car', 'moto', 'scooter', 'atv', 'buggy', 'bicycle', 'boat', 'other');
					$parts = explode(',', $vtype);
					$arr = array();
					foreach ($parts as $part) {
						$v = trim(strtolower($part));
						if ($v == '') { continue; }
						if (!in_array($v, $vtypes)) { continue; }
						$arr[] = $v;
					}
					$vtype = ($arr) ? implode(',', $arr) : '';
				}

				$newconfigs[] = array(
					'serverurl' => $serverurl,
					'clienturl' => $this->pluginurl.'/client',
					'platform' => 'wordpress',
					'sethead' => 1,
					'apiid' => $apiid,
					'prkey' => $prkey,
					'guilang' => $guilang,
					'lang' => $lang,
					'currency' => $currency,
					'cachetime' => $cachetime,
					'loadfont' => $loadfont,
					'vtype' => $vtype,
					'modelfeaticons' => $modelfeaticons,
					'modelprice' => $modelprice,
					'applyvat' => $applyvat,
					'vatnote' => $vatnote,
					'showerrorcodes' => $showerrorcodes,
					'formaction' => $formaction,
					'fleetpage' => $fleetpage,
					'unifiedmoto' => $unifiedmoto,
					'dropsameaspick' => $dropsameaspick,
					'sidecolumn' => $sidecolumn,
					'formtext1' => $formtext1,
					'formtext2' => $formtext2,
					'showcoupon' => $showcoupon,
					'showinvoice' => $showinvoice,
					'rentalduration' => $rentalduration,
					'imagewidth' => $imagewidth,
					'pickupday' => $pickupday,
					'dropoffday' => $dropoffday,
					'pickuptime' => $pickuptime,
					'dropofftime' => $dropofftime,
					'hidedrop' => $hidedrop,
					'titlepos' => $titlepos,
					'modellocs' => $modellocs,
					'lids' => $lids,
					'roundprices' => $roundprices
				);
	    	}

	    	if ($errormsg == '') {
				$settings_saved = $this->saveAPIConfigs($newconfigs);
				if (isset($_POST['delcache'])) {
					if ($_POST['delcache'] == 1) { $this->deleteCache(); }
				}
	    	}
	    	unset($newconfigs);
	    }

	    if ($errormsg != '') {
	    	echo '<div class="error"><p><strong>'.$errormsg.'</strong></p></div>'."\n";
	    } else if ($settings_saved == 0) {
	    	echo '<div class="error"><p><strong>Saving settings failed!</strong></p></div>'."\n";
	    } else if ($settings_saved == 1) {
	    	echo '<div class="updated"><p><strong>Settings saved successfully!</strong></p></div>'."\n";
	    }

    	$guilangs = $this->getGUILanguages();

		echo '<div class="wrap">'."\n";
		echo '<h1 class="iosrt_wp_h1">IOSRT API Settings <span>v'.$this->version."</span></h1>\n";
		echo '<p class="description">You can provide settings for up to 3 APIs. To integrate an API inside an article type inside an article\'s or page ';
		echo '<strong>[iosrtapi apiid="X"]</strong> where <strong>X</strong> the API ID. Note: Different APIs must belong to the same rental company.</p>'."\n";

		echo '<form name="iosrt_apifm" method="post" action="">'."\n";
	
		echo '<div class="iosrt_wp_tab">'."\n";
		echo '<a href="javascript:void(null);" id="iosrt_wp_tablnk0" class="iosrt_wp_tablink iosrt_wp_tabactive" onclick="iosRTWPOpenTab(0)">API 1</a>'."\n";
		echo '<a href="javascript:void(null);" id="iosrt_wp_tablnk1" class="iosrt_wp_tablink" onclick="iosRTWPOpenTab(1)">API 2</a>'."\n";
		echo '<a href="javascript:void(null);" id="iosrt_wp_tablnk2" class="iosrt_wp_tablink" onclick="iosRTWPOpenTab(2)">API 3</a>'."\n";
		echo '<a href="javascript:void(null);" id="iosrt_wp_tablnk3" class="iosrt_wp_tablink" onclick="iosRTWPOpenTab(3)">Other options</a>'."\n";
		echo "</div>\n";

		$apiconfig = $this->getAPIconfig(0);
		echo '<div id="iosrt_wp_tab0" class="iosrt_wp_tabcontent">'."\n";
		$this->makeAPIEditFields(0, $guilangs, $apiconfig);
		echo "</div>\n";

		$apiconfig = $this->getAPIconfig(1);
		echo '<div id="iosrt_wp_tab1" class="iosrt_wp_tabinvcontent">'."\n";
		$this->makeAPIEditFields(1, $guilangs, $apiconfig);
		echo "</div>\n";

		$apiconfig = $this->getAPIconfig(2);
		echo '<div id="iosrt_wp_tab2" class="iosrt_wp_tabinvcontent">'."\n";
		$this->makeAPIEditFields(2, $guilangs, $apiconfig);
		echo "</div>\n";

		echo '<div id="iosrt_wp_tab3" class="iosrt_wp_tabinvcontent">'."\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_delcache">Empty cache?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="checkbox" name="delcache" id="iosrt_delcache" value="1" /> '.__('Yes');
		echo '<div class="iosrt_wp_tip">Check this box to delete existing cached files on save.</div>'."\n";
		echo "</div>\n</div>\n";

		echo "</div>\n";//iosrt_wp_tab3

		echo '<input type="hidden" name="iosrtapisbt" value="1" />'."\n";
		submit_button();
		echo "</form>\n";
		echo "</div>\n";
	}


	/************************************/
	/* MAKE API EDIT FIELDS (0 INDEXED) */
	/************************************/
	private function makeAPIEditFields($apiidx, $guilangs, $apiconfig) {
		$num = $apiidx + 1;

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_apiid'.$apiidx.'">API ID ('.$num.")</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="apiid'.$apiidx.'" id="iosrt_apiid'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="64" value="'.$apiconfig['apiid'].'" placeholder="API ID" />';
		if ($apiconfig['apiid'] != '') {
			echo '<div class="iosrt_wp_tip">Integration code: <em>[iosrtapi apiid=&quot;'.$apiconfig['apiid'].'&quot;]</em></div>'."\n";
		} else {
			echo '<div class="iosrt_wp_tip">Integration code: <em>[iosrtapi apiid=&quot;API ID HERE&quot;]</em></div>'."\n";
		}
		echo "</div></div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_prkey'.$apiidx.'">Private key'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="prkey'.$apiidx.'" id="iosrt_prkey'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="64" value="'.$apiconfig['prkey'].'" placeholder="Private key" />';
		echo "</div></div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_serverurl'.$apiidx.'">Server URL'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="serverurl'.$apiidx.'" id="iosrt_serverurl'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="150" value="'.$apiconfig['serverurl'].'" placeholder="Server URL" />';
		echo '<div class="iosrt_wp_tip">Full URL to the API server without trailing slash. Example: <em>https://something.com</em></div>'."\n";
		echo "</div></div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_formaction'.$apiidx.'">Form action URL'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="formaction'.$apiidx.'" id="iosrt_formaction'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="120" value="'.$apiconfig['formaction'].'" placeholder="Form action URL" />';
		echo '<div class="iosrt_wp_tip">If you wish to display search results to a different page than the form page provide here this page full URL address. If you leave this option empty the results will be displayed in the same page. To provide multi-lingual URL for WPML use the <em>(ML/)</em> keyword. Example: <em>https://www.example.com/(ML/)something/</em></div>'."\n";
		echo "</div></div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_fleetpage'.$apiidx.'">Fleet page URL'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="fleetpage'.$apiidx.'" id="iosrt_fleetpage'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="120" value="'.$apiconfig['fleetpage'].'" placeholder="Fleet page URL" />';
		echo '<div class="iosrt_wp_tip">If you wish to display fleet model in a different page than the fleet page provide here this page full URL address. If you leave this option empty the fleet model will be displayed in the same page. To provide multi-lingual URL for WPML use the <em>(ML/)</em> keyword. Example: <em>https://www.example.com/(ML/)something/</em></div>'."\n";
		echo "</div></div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_guilang'.$apiidx.'">Interface language'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="guilang'.$apiidx.'" id="iosrt_guilang'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['guilang'] == '') ? ' selected="selected"' : '';
		echo '<option value=""'.$sel.'>Auto detect / WPML</option>'."\n";
		if ($guilangs) {
			foreach ($guilangs as $lng) {
				$sel = ($apiconfig['guilang'] == $lng) ? ' selected="selected"' : '';
				echo '<option value="'.$lng.'"'.$sel.'>'.strtoupper($lng)."</option>\n";
			}
		}
		echo "</select>\n";
		echo "</div>\n</div>\n";

		$server_langs = array('ar', 'bg', 'de', 'el', 'en', 'es', 'fa', 'fr', 'hr', 'it', 'nl', 'pt', 'rs', 'ru', 'sr', 'tr', 'uk', 'zh', 'zt');
		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_lang'.$apiidx.'">Server language'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="lang'.$apiidx.'" id="iosrt_lang'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['lang'] == '') ? ' selected="selected"' : '';
		echo '<option value=""'.$sel.'>Server default / WPML</option>'."\n";
		if ($server_langs) {
			foreach ($server_langs as $lng) {
				$sel = ($apiconfig['lang'] == $lng) ? ' selected="selected"' : '';
				echo '<option value="'.$lng.'"'.$sel.'>'.strtoupper($lng)."</option>\n";
			}
		}
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">If you select a language you must be sure the language is available on server!</div>'."\n";
		echo "</div>\n</div>\n";

		$currencies = array('AUD', 'BGN', 'BRL', 'CAD', 'CHF', 'CNY', 'CZK', 'DKK', 'EUR', 'GBP', 'HKD', 'HRK', 'HUF', 'IDR', 'INR', 'JPY', 'KRW', 'MXN', 'MYR', 'NOK', 'NZD', 'PLN', 'PHP', 'RON', 'RUB', 'SEK', 'SGD', 'THB', 'TRY', 'USD', 'ZAR');
		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_currency'.$apiidx.'">Currency'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="currency'.$apiidx.'" id="iosrt_currency'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['currency'] == '') ? ' selected="selected"' : '';
		echo '<option value=""'.$sel.'>Server default</option>'."\n";
		if ($currencies) {
			foreach ($currencies as $cur) {
				$sel = ($apiconfig['currency'] == $cur) ? ' selected="selected"' : '';
				echo '<option value="'.$cur.'"'.$sel.'>'.$cur."</option>\n";
			}
		}
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_cachetime'.$apiidx.'">Cache time'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="cachetime'.$apiidx.'" id="iosrt_cachetime'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['cachetime'] == 1800) ? ' selected="selected"' : '';
		echo '<option value="1800"'.$sel.'>30 minutes</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 3600) ? ' selected="selected"' : '';
		echo '<option value="3600"'.$sel.'>1 hour</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 7200) ? ' selected="selected"' : '';
		echo '<option value="7200"'.$sel.'>2 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 10800) ? ' selected="selected"' : '';
		echo '<option value="10800"'.$sel.'>3 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 14400) ? ' selected="selected"' : '';
		echo '<option value="14400"'.$sel.'>4 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 21600) ? ' selected="selected"' : '';
		echo '<option value="21600"'.$sel.'>6 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 28800) ? ' selected="selected"' : '';
		echo '<option value="28800"'.$sel.'>8 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 43200) ? ' selected="selected"' : '';
		echo '<option value="43200"'.$sel.'>12 hours</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 86400) ? ' selected="selected"' : '';
		echo '<option value="86400"'.$sel.'>1 day</option>'."\n";
		$sel = ($apiconfig['cachetime'] == 172800) ? ' selected="selected"' : '';
		echo '<option value="172800"'.$sel.'>2 days</option>'."\n";
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_loadfont'.$apiidx.'">Font to load'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="loadfont'.$apiidx.'" id="iosrt_loadfont'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="40" value="'.$apiconfig['loadfont'].'" placeholder="Font to load" />';
		echo '<div class="iosrt_wp_tip">A font to load (by default <em>roboto</em>). If your template already loads the roboto font then provide an empty value here in order not to load the font twice.</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_vtype'.$apiidx.'">Vehicle type(s)'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="vtype'.$apiidx.'" id="iosrt_vtype'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="80" value="'.$apiconfig['vtype'].'" placeholder="Vehicle type(s)" />';
		echo '<div class="iosrt_wp_tip">Comma separated list (no spaces) of vehicle types to search for. Values: <em>car, moto, scooter, atv, buggy, bicycle, boat, other</em>. If empty all available vehicle types will be loaded (those enabled in your server account).</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_modelfeaticons'.$apiidx.'">Model features icons'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="modelfeaticons'.$apiidx.'" id="iosrt_modelfeaticons'.$apiidx.'" class="iosrt_wp_select">'."\n";
		for ($i=0; $i < 21; $i++) {
			$sel = ($apiconfig['modelfeaticons'] == $i) ? ' selected="selected"' : '';
			echo '<option value="'.$i.'"'.$sel.'>'.$i.'</option>'."\n";
		}
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">How many icons to display for model features.</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_imagewidth'.$apiidx.'">Model image width'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="imagewidth'.$apiidx.'" id="iosrt_imagewidth'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['imagewidth'] == 240) ? ' selected="selected"' : '';
		echo '<option value="240"'.$sel.'>240</option>'."\n";
		$sel = ($apiconfig['imagewidth'] == 320) ? ' selected="selected"' : '';
		echo '<option value="320"'.$sel.'>320</option>'."\n";
		$sel = ($apiconfig['imagewidth'] == 400) ? ' selected="selected"' : '';
		echo '<option value="400"'.$sel.'>400</option>'."\n";
		$sel = ($apiconfig['imagewidth'] == 480) ? ' selected="selected"' : '';
		echo '<option value="480"'.$sel.'>480</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Rental duration shown in booking page</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_modelprice'.$apiidx.'">Shown prices'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="modelprice'.$apiidx.'" id="iosrt_modelprice'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['modelprice'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>Rental fee</option>'."\n";
		$sel = ($apiconfig['modelprice'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Total price (inc. pickup fee, etc)</option>'."\n";
		$sel = ($apiconfig['modelprice'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>Rental fee and rest charges</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">What to display as price for a vehicle in search results page?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_applyvat'.$apiidx.'">Apply VAT to prices?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="applyvat'.$apiidx.'" id="iosrt_applyvat'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['applyvat'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['applyvat'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Apply VAT to prices shown in search results page? Works only if company has set a VAT.</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_vatnote'.$apiidx.'">VAT note'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="vatnote'.$apiidx.'" id="iosrt_vatnote'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['vatnote'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['vatnote'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display a note below shown prices in search results page regarding VAT (included, not included, VAT percentage).</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_rentalduration'.$apiidx.'">Rental duration'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="rentalduration'.$apiidx.'" id="iosrt_rentalduration'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['rentalduration'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>Actual days and hours + Charge days and hours</option>'."\n";
		$sel = ($apiconfig['rentalduration'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Actual days and hours</option>'."\n";
		$sel = ($apiconfig['rentalduration'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>Charge days and hours</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Rental duration shown in booking page</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_showerrorcodes'.$apiidx.'">Show error codes?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="showerrorcodes'.$apiidx.'" id="iosrt_showerrorcodes'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['showerrorcodes'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').' (recommended)</option>'."\n";
		$sel = ($apiconfig['showerrorcodes'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').' (only for debug)</option>'."\n";
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_unifiedmoto'.$apiidx.'">Unified motos?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="unifiedmoto'.$apiidx.'" id="iosrt_unifiedmoto'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['unifiedmoto'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['unifiedmoto'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display Motorcycles, Scooters, ATVs and Buggies as a single option (Motorcycles)</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_dropsameaspick'.$apiidx.'">Same as pick-up'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="dropsameaspick'.$apiidx.'" id="iosrt_dropsameaspick'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['dropsameaspick'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['dropsameaspick'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display a <em>Same as pick-up location</em> link below drop-off selection?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_sidecolumn'.$apiidx.'">Side column'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="sidecolumn'.$apiidx.'" id="iosrt_sidecolumn'.$apiidx.'" dir="ltr" class="iosrt_wp_text" maxlength="180" value="'.$apiconfig['sidecolumn'].'" placeholder="Side column" />';
		echo '<div class="iosrt_wp_tip">If you want to display a side column in search results page type here a comma separated list of model features you want to display filters for. 
		Special options: <strong>search</strong> (displays search form), <strong>categories</strong> (displays filters for categories), <strong>vtype</strong> (displays filters for vehicle types) and <strong>all</strong> (filters for all model features). 
		See all model features in IOSRT <em>modelfeatures.xml</em> file. Example 1: <em>search,vtype,categories,gearbox,fuel,riders,baggages</em> Example 2: <em>search,vtype,categories,all</em></div>'."\n";
		echo "</div>\n</div>\n";

		$formtexts = array(
			'FORMTEXT_PDAFTER20_CONTACT' => 'Pick-ups/drop-offs after 20:00 contact',
			'FORMTEXT_PDAFTER21_CONTACT' => 'Pick-ups/drop-offs after 21:00 contact',
			'FORMTEXT_PDAFTER22_CONTACT' => 'Pick-ups/drop-offs after 22:00 contact',
			'FORMTEXT_PDAFTER23_CONTACT' => 'Pick-ups/drop-offs after 23:00 contact',
			'FORMTEXT_PDAFTER00_CONTACT' => 'Pick-ups/drop-offs after 00:00 contact',
			'FORMTEXT_PDAFTER01_CONTACT' => 'Pick-ups/drop-offs after 01:00 contact',
			'FORMTEXT_PDAFTER02_CONTACT' => 'Pick-ups/drop-offs after 02:00 contact',
			'FORMTEXT_MORE07_CONTACT' => 'Rentals more than 7 days contact',
			'FORMTEXT_MORE10_CONTACT' => 'Rentals more than 10 days contact',
			'FORMTEXT_MORE12_CONTACT' => 'Rentals more than 12 days contact',
			'FORMTEXT_MORE15_CONTACT' => 'Rentals more than 15 days contact',
			'FORMTEXT_MORE20_CONTACT' => 'Rentals more than 20 days contact',
			'FORMTEXT_MORE25_CONTACT' => 'Rentals more than 25 days contact',
			'FORMTEXT_MORE30_CONTACT' => 'Rentals more than 30 days contact',
			'FORMTEXT_CUSTOM1' => 'Custom text 1 (language files)',
			'FORMTEXT_CUSTOM2' => 'Custom text 2 (language files)'
		);

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_formtext1'.$apiidx.'">Form text 1'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="formtext1'.$apiidx.'" id="iosrt_formtext1'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = '';
		$other_v = '';
		if ($apiconfig['formtext1'] != '') {
			$ft = $apiconfig['formtext1'];
			if (!isset($formtexts[$ft])) { $sel = ' selected="selected"'; $other_v = $apiconfig['formtext1']; }
		} else {
			$sel = ' selected="selected"';
		}
		echo '<option value=""'.$sel.'>- None / Other -</option>'."\n";
		foreach ($formtexts as $k => $t) {
			$sel = ($apiconfig['formtext1'] == $k) ? ' selected="selected"' : '';
			echo '<option value="'.$k.'"'.$sel.'>'.$t.'</option>'."\n";
		}
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_formtextc1'.$apiidx.'">Other text 1'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="formtextc1'.$apiidx.'" id="iosrt_formtextc1'.$apiidx.'" class="iosrt_wp_text" maxlength="240" value="'.$other_v.'" placeholder="Other text 1" />';
		echo '<div class="iosrt_wp_tip">Alternatively to Form text 1. Not multi-lingual!</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_formtext2'.$apiidx.'">Form text 2'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="formtext2'.$apiidx.'" id="iosrt_formtext2'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = '';
		$other_v = '';
		if ($apiconfig['formtext2'] != '') {
			$ft = $apiconfig['formtext2'];
			if (!isset($formtexts[$ft])) { $sel = ' selected="selected"'; $other_v = $apiconfig['formtext2']; }
		} else {
			$sel = ' selected="selected"';
		}
		echo '<option value=""'.$sel.'>- None / Other -</option>'."\n";
		foreach ($formtexts as $k => $t) {
			$sel = ($apiconfig['formtext2'] == $k) ? ' selected="selected"' : '';
			echo '<option value="'.$k.'"'.$sel.'>'.$t.'</option>'."\n";
		}
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_formtextc2'.$apiidx.'">Other text 2'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="formtextc2'.$apiidx.'" id="iosrt_formtextc2'.$apiidx.'" class="iosrt_wp_text" maxlength="240" value="'.$other_v.'" placeholder="Other text 2" />';
		echo '<div class="iosrt_wp_tip">Alternatively to Form text 2. Not multi-lingual!</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_showcoupon'.$apiidx.'">Show coupon?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="showcoupon'.$apiidx.'" id="iosrt_showcoupon'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['showcoupon'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['showcoupon'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display coupon input form in booking page?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_showinvoice'.$apiidx.'">Show invoice?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="showinvoice'.$apiidx.'" id="iosrt_showinvoice'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['showinvoice'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['showinvoice'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display invoice selection option in booking page?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_pickupday'.$apiidx.'">Pick-up day'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="pickupday'.$apiidx.'" id="iosrt_pickupday'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['pickupday'] == -1) ? ' selected="selected"' : '';
		echo '<option value="-1"'.$sel.'>-- Any --</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>Sunday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Monday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>Tuesday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 3) ? ' selected="selected"' : '';
		echo '<option value="3"'.$sel.'>Wednesday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 4) ? ' selected="selected"' : '';
		echo '<option value="4"'.$sel.'>Thursday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 5) ? ' selected="selected"' : '';
		echo '<option value="5"'.$sel.'>Friday</option>'."\n";
		$sel = ($apiconfig['pickupday'] == 6) ? ' selected="selected"' : '';
		echo '<option value="6"'.$sel.'>Saturday</option>'."\n";
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_dropoffday'.$apiidx.'">Drop-off day'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="dropoffday'.$apiidx.'" id="iosrt_dropoffday'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['dropoffday'] == -1) ? ' selected="selected"' : '';
		echo '<option value="-1"'.$sel.'>-- Any --</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>Sunday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Monday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>Tuesday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 3) ? ' selected="selected"' : '';
		echo '<option value="3"'.$sel.'>Wednesday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 4) ? ' selected="selected"' : '';
		echo '<option value="4"'.$sel.'>Thursday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 5) ? ' selected="selected"' : '';
		echo '<option value="5"'.$sel.'>Friday</option>'."\n";
		$sel = ($apiconfig['dropoffday'] == 6) ? ' selected="selected"' : '';
		echo '<option value="6"'.$sel.'>Saturday</option>'."\n";
		echo "</select>\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_pickuptime'.$apiidx.'">Pick-up time'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="pickuptime'.$apiidx.'" id="iosrt_pickuptime'.$apiidx.'" class="iosrt_wp_text" maxlength="5" value="'.$apiconfig['pickuptime'].'" placeholder="Pick-up time" dir="ltr" />';
		echo '<div class="iosrt_wp_tip">For a fixed pick-up time write the time. Empty allows user to select time. Example: 09:30</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_dropofftime'.$apiidx.'">Drop-off time'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="dropofftime'.$apiidx.'" id="iosrt_dropofftime'.$apiidx.'" class="iosrt_wp_text" maxlength="5" value="'.$apiconfig['dropofftime'].'" placeholder="Drop-off time" dir="ltr" />';
		echo '<div class="iosrt_wp_tip">For a fixed drop-off time write the time. Empty allows user to select time. Example: 15:00</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_hidedrop'.$apiidx.'">Hide drop-off location?'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="hidedrop'.$apiidx.'" id="iosrt_hidedrop'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['hidedrop'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['hidedrop'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">If hidden, drop-off location becomes the same as pick-up location</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_titlepos'.$apiidx.'">Title position'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="titlepos'.$apiidx.'" id="iosrt_titlepos'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['titlepos'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>Top</option>'."\n";
		$sel = ($apiconfig['titlepos'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Side</option>'."\n";
		$sel = ($apiconfig['titlepos'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>Bottom</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Model title position in search results page</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_modellocs'.$apiidx.'">Available locations'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="modellocs'.$apiidx.'" id="iosrt_modellocs'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['modellocs'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['modellocs'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>Yes, if not available in all locations</option>'."\n";
		$sel = ($apiconfig['modellocs'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>'.__('Yes').'</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Display the locations a model is available at?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_lids'.$apiidx.'">Location IDs'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<input type="text" name="lids'.$apiidx.'" id="iosrt_lids'.$apiidx.'" class="iosrt_wp_text" value="'.$apiconfig['lids'].'" placeholder="Drop-off time" dir="ltr" />';
		echo '<div class="iosrt_wp_tip">If you wish to display only a few locations in search form then write here the location IDs you want to display comma separated. Example: 2,8,15. Leave empty for all locations.</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<div class="iosrt_wp_row">'."\n";
		echo '<label class="iosrt_wp_label" for="iosrt_roundprices'.$apiidx.'">Round prices'."</label>\n";
		echo '<div class="iosrt_wp_labelside">'."\n";
		echo '<select name="roundprices'.$apiidx.'" id="iosrt_roundprices'.$apiidx.'" class="iosrt_wp_select">'."\n";
		$sel = ($apiconfig['roundprices'] == 0) ? ' selected="selected"' : '';
		echo '<option value="0"'.$sel.'>'.__('No').'</option>'."\n";
		$sel = ($apiconfig['roundprices'] == 1) ? ' selected="selected"' : '';
		echo '<option value="1"'.$sel.'>'.__('Yes').', if amount is integer</option>'."\n";
		$sel = ($apiconfig['roundprices'] == 2) ? ' selected="selected"' : '';
		echo '<option value="2"'.$sel.'>'.__('Yes').', without decimals</option>'."\n";
		$sel = ($apiconfig['roundprices'] == 3) ? ' selected="selected"' : '';
		echo '<option value="3"'.$sel.'>'.__('Yes').', with decimals</option>'."\n";
		echo "</select>\n";
		echo '<div class="iosrt_wp_tip">Round prices to nearest integer on search results, fleet and model pages?</div>'."\n";
		echo "</div>\n</div>\n";

		echo '<input type="hidden" name="sethead'.$apiidx.'" value="1" />'."\n";
		echo '<input type="hidden" name="platform'.$apiidx.'" value="wordpress" />'."\n";
	}


	/*******************************/
	/* GET AVAILABLE GUI LANGUAGES */
	/*******************************/
	private function getGUILanguages() {
		$dir = $this->plugindir.'/client/language/';
		$languages = array();
		if (!is_dir($dir)) { return $languages; }
		if (!$dh = opendir($dir)) { return $languages; }
		while (($file = readdir($dh)) !== false) {
			if (($file == '.') || ($file == '..') || !is_file($dir.$file)) { continue; }
			if (!preg_match('@(\.php)$@i', $file)) { continue; }
			$lng = strtolower(str_replace('.php', '', $file));
			if (strlen($lng) == 2) { $languages[] = $lng; }
		}
		closedir($dh);
		return $languages;
	}


	/*********************************/
	/* GET API CONFIGURATION OPTIONS */
	/*********************************/
	private function getAPIconfig($num=0) {
		$defaultcfg = array(
			'serverurl' => '',
			'clienturl' => $this->pluginurl.'/client',
			'platform' => 'wordpress',
			'sethead' => 1,
			'apiid' => '',
			'prkey' => '',
			'guilang' => '',
			'lang' => '',
			'currency' => '',
			'cachetime' => 14400,
			'loadfont' => 'roboto',
			'vtype' => '',
			'modelfeaticons' => 6,
			'modelprice' => 1,
			'applyvat' => 1,
			'vatnote' => 1,
			'showerrorcodes' => 0,
			'formaction' => '',
			'fleetpage' => '',
			'unifiedmoto' => 0,
			'dropsameaspick' => 0,
			'sidecolumn' => '',
			'formtext1' => '',
			'formtext2' => '',
			'showcoupon' => 1,
			'showinvoice' => 1,
			'rentalduration' => 0,
			'imagewidth' => 240,
			'pickupday' => -1,
			'dropoffday' => -1,
			'pickuptime' => '',
			'dropofftime' => '',
			'hidedrop' => 0,
			'titlepos' => 0,
			'modellocs' => 1,
			'roundprices' => 1,
			'lids' => ''
		);

		if (!file_exists($this->plugindir.'/client/apiconfig.php')) { return $defaultcfg; }
		if (!defined('_IOSRTAPICLIENT_')) { define('_IOSRTAPICLIENT_', 1); }
		include($this->plugindir.'/client/apiconfig.php');
		if (!isset($apiconfig) || !is_array($apiconfig) || (count($apiconfig) == 0)) { return $defaultcfg; }
		if (!isset($apiconfig[$num])) { return $defaultcfg; }
		//compatibility with previous than 1.0.8 versions
		if (!isset($apiconfig[$num]['showcoupon'])) { $apiconfig[$num]['showcoupon'] = 1; }
		if (!isset($apiconfig[$num]['showinvoice'])) { $apiconfig[$num]['showinvoice'] = 1; }
		if (!isset($apiconfig[$num]['imagewidth'])) { $apiconfig[$num]['imagewidth'] = 240; }//v1.4
		if (!isset($apiconfig[$num]['pickupday'])) { $apiconfig[$num]['pickupday'] = -1; }//v1.4
		if (!isset($apiconfig[$num]['dropoffday'])) { $apiconfig[$num]['dropoffday'] = -1; }//v1.4
		if (!isset($apiconfig[$num]['pickuptime'])) { $apiconfig[$num]['pickuptime'] = ''; }//v1.4
		if (!isset($apiconfig[$num]['dropofftime'])) { $apiconfig[$num]['dropofftime'] = ''; }//v1.4
		if (!isset($apiconfig[$num]['hidedrop'])) { $apiconfig[$num]['hidedrop'] = 0; }//v1.4
		if (!isset($apiconfig[$num]['titlepos'])) { $apiconfig[$num]['titlepos'] = 0; }//v1.4
		if (!isset($apiconfig[$num]['modellocs'])) { $apiconfig[$num]['modellocs'] = 1; }//v1.4
		if (!isset($apiconfig[$num]['lids'])) { $apiconfig[$num]['lids'] = ''; }//v1.4
		if (!isset($apiconfig[$num]['fleetpage'])) { $apiconfig[$num]['fleetpage'] = ''; }//v1.4
		if (!isset($apiconfig[$num]['roundprices'])) { $apiconfig[$num]['roundprices'] = 1; }//v1.4

		return $apiconfig[$num];
	}


	/***************************/
	/* SAVE API CONFIGURATIONS */
	/***************************/
	private function saveAPIConfigs($newconfigs) {
		$data = '<?php '."\n\n";
		$data .= 'defined(\'_IOSRTAPICLIENT_\') or die (\'Direct access to this location is not allowed\');'."\n\n";
		$data .= '$apiconfig = array();'."\n\n";
		if ($newconfigs) {
			foreach ($newconfigs as $cfg) {
				$data .= '$apiconfig[] = array('."\n";
				foreach ($cfg as $k => $v) {
					$data .= "\t".'\''.$k.'\' => ';
					if (is_numeric($v)) {
						$data .= $v.",\n";
					} else {
						$data .= '\''.addslashes($v).'\','."\n";
					}
				}
				$data .= ");\n\n";
			}
		}
		$data .= '?>';

		$handle = @fopen($this->plugindir.'/client/apiconfig.php', 'w');
		if (!$handle) { return false; }
		if (!fwrite($handle, $data)) {
			fclose($handle);
			return false;
		}
		fclose($handle);
		return true;
	}


	/***********************/
	/* DELETE CLIENT CACHE */
	/***********************/
	private function deleteCache() {
		$cache_dir = $this->plugindir.'/client/cache/';
		$files = glob($cache_dir.'*.php');
		if (!$files) { return; }
		foreach ($files as $file) { @unlink($file); }
	}

}

?>