<?php
/**
 * Plugin Name: CSSIgniter Updater
 * Plugin URI: https://www.cssigniter.com/plugins/cssigniter-updater/
 * Description: Enables dashboard notifications and updates for CSSIgniter themes and plugins.
 * Version: 1.2
 * Author: CSSIgniter
 * Author URI: https://www.cssigniter.com/
 * Network: true
 */

class CSSIgniter_Updater {
	private $debug = false; // Only set to true temporarily, when developing against a local copy of the CSSIgniter Updates system.

	private $api_version = '1.1';

	private $endpoint_url = 'https://www.cssigniter.com/wp-admin/admin-ajax.php';
	private $check_every  = 24; // In hours.

	private $options = array();
	private $hmac    = false;

	protected $json_url = array(
		'themes'  => 'https://www.cssigniter.com/theme_versions.json',
		'plugins' => 'https://www.cssigniter.com/plugin_versions.json',
	);

	public function __construct() {}

	public function initialize() {
		add_action( 'after_setup_theme', array( $this, 'unhook_theme_notifications' ), 20 );
		add_action( 'init', array( $this, 'init' ) );
		add_action( 'init', array( $this, 'unhook_plugin_notifications' ), 20 );
	}

	public function init() {
		if ( $this->debug ) {
			// Only enable these for local development.
			add_filter( 'http_request_args', array( $this, 'disable_sslverify' ), 10, 2 );
			add_filter( 'http_request_host_is_external', '__return_true' );
		}

		load_plugin_textdomain( 'cssigniter-updater', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );

		$this->options        = $this->settings_validate( is_multisite() ? get_site_option( 'cssigniter_updater_settings' ) : get_option( 'cssigniter_updater_settings' ) );
		$this->options['url'] = network_site_url();
		$this->hmac           = $this->hash_hmac( $this->options['username'] . $this->options['url'] );
		$this->check_every    = apply_filters( 'cssigniter_updater_check_every_days', 24 ) * HOUR_IN_SECONDS;

		add_filter( 'plugins_api', array( $this, 'plugin_info' ), 20, 3 );

		add_action( is_multisite() ? 'network_admin_menu' : 'admin_menu', array( $this, 'updater_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'network_admin_edit_cssigniter_updater_save_network_settings', array( $this, 'save_network_settings' ) );

		add_filter( 'pre_set_site_transient_update_themes', array( $this, 'check_themes' ), 20 );
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_plugins' ), 20 );

		if ( ( is_network_admin() || is_admin() ) && isset( $_REQUEST['page'] ) && 'cssigniter-updater-settings' === $_REQUEST['page'] && isset( $_REQUEST['settings-updated'] ) && 'true' === $_REQUEST['settings-updated'] ) {
			add_action( 'admin_notices', array( $this, 'notice_plugin_connection' ) );
			add_action( 'network_admin_notices', array( $this, 'notice_plugin_connection' ) );
		}
	}

	public function unhook_theme_notifications() {
		// This takes care of the panel themes, that have their own, broken, notifications.
		remove_action( 'pre_set_site_transient_update_themes', 'ci_theme_update_check_admin_handler' );
	}

	public function unhook_plugin_notifications() {
		// This takes care of our premium plugins.
		global $wp_filter;
		$plugin_classes = array(
			'AudioIgniter_Updater',
			'EventIgniter_Updater',
		);
		if ( isset( $wp_filter['pre_set_site_transient_update_plugins'] ) && ! empty( $wp_filter['pre_set_site_transient_update_plugins']->callbacks[10] ) ) {
			foreach ( $wp_filter['pre_set_site_transient_update_plugins']->callbacks[10] as $key => $callback ) {
				if ( isset( $callback['function'] ) && is_array( $callback['function'] ) && isset( $callback['function'][0] ) && is_object( $callback['function'][0] ) ) {
					if ( in_array( get_class( $callback['function'][0] ), $plugin_classes, true ) ) {
						$instance = $callback['function'][0];
						remove_action( 'pre_set_site_transient_update_plugins', array( $instance, 'update_check_admin_handler' ) );
					}
				}
			}
		}
	}

	public function plugin_info( $info, $action, $args ) {

		if ( 'plugin_information' !== $action ) {
			return false;
		}

		// Do nothing if it is not our plugin.
		if ( ! array_key_exists( $args->slug, $this->latest_plugin_versions() ) ) {
			return $info;
		}

		$transient_name = 'cssigniter_updater_saved_plugin_info_' . $args->slug;

		$saved_info = get_site_transient( $transient_name );

		if ( false === $saved_info ) {
			$url = add_query_arg( array(
				'action' => 'updates_get_plugin_info',
				'plugin' => $args->slug,
			), $this->endpoint_url );

			$response = wp_safe_remote_get( $url );

			$success = false;

			if ( 200 === wp_remote_retrieve_response_code( $response ) ) {
				$json = json_decode( wp_remote_retrieve_body( $response ), true );
				if ( isset( $json['success'] ) && true === $json['success'] ) {
					$success = true;

					$plugin_info = $json['data'];
				}
			}

			if ( ! $success ) {
				return $info;
			}

			$saved_info = new stdClass();

			$saved_info->name           = $plugin_info['name'];
			$saved_info->slug           = $plugin_info['slug'];
			$saved_info->version        = $plugin_info['version'];
			$saved_info->tested         = $plugin_info['tested'];
			$saved_info->requires       = $plugin_info['requires'];
			$saved_info->author         = $plugin_info['author'];
			$saved_info->author_profile = $plugin_info['author_profile'];
			$saved_info->download_link  = $plugin_info['download_link'];
			$saved_info->last_updated   = $plugin_info['last_updated'];
			$saved_info->sections       = $plugin_info['sections'];

			// We want to serve fresh information, so let's just cache for an hour, just to avoid hammering the server.
			set_site_transient( $transient_name, $saved_info, HOUR_IN_SECONDS );
		}

		$info = $saved_info;

		return $info;
	}

	public function check_themes( $transient ) {

		// Sample transient output: https://www.dropbox.com/s/ppixxxftdkmbtwz/Screenshot%202018-09-19%2020.05.24.png?dl=0

		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$query_themes = $transient->checked;

		$translations = wp_get_installed_translations( 'themes' );
		$locales      = array_unique( array_values( get_available_languages() ) );

		if ( ! empty( $this->options['username'] ) && ! empty( $this->options['url'] ) && ! empty( $this->options['key'] ) ) {
			$response = get_site_transient( 'cssigniter_updater_saved_theme_updates' );

			// Make sure we invalidate API version 1.0 data that may be cached.
			if ( false !== $response && ! isset( $response['themes'] ) ) {
				$response = false;
			}

			if ( false === $response ) {
				$response = array();

				$url = add_query_arg( array(
					'action'      => 'updates_get_theme_updates',
					'hmac'        => $this->hmac,
					'username'    => $this->options['username'],
					'url'         => $this->options['url'],
					'api_version' => $this->api_version,
				), $this->endpoint_url );

				$api_response = wp_safe_remote_post( $url, array(
					'body' => array(
						'themes'       => $query_themes,
						'translations' => $translations,
						'locale'       => $locales,
					),
				) );

				if ( 200 === wp_remote_retrieve_response_code( $api_response ) ) {
					$json = json_decode( wp_remote_retrieve_body( $api_response ), true );
					if ( isset( $json['success'] ) && true === $json['success'] ) {
						$response = $json['data'];
					}
				}

				set_site_transient( 'cssigniter_updater_saved_theme_updates', $response, $this->check_every );
			}
		}

		foreach ( $query_themes as $theme => $old_version ) {
			if ( isset( $response['response'][ $theme ] ) ) {
				if ( isset( $transient->no_update[ $theme ] ) ) {
					unset( $transient->no_update[ $theme ] );
				}

				$entry       = $response['response'][ $theme ];
				$package_url = '';

				if ( ! empty( $entry['package_base_url'] ) ) {
					$package_url = add_query_arg( array(
						'action'   => 'updates_get_theme_zip',
						'hmac'     => $this->hmac,
						'username' => $this->options['username'],
						'url'      => $this->options['url'],
						'theme'    => $theme,
						'version'  => $entry['new_version'],
					), $entry['package_base_url'] );

					$entry['package'] = $package_url;
				}

				$transient->response[ $theme ] = $entry;

			} elseif ( isset( $response['no_update'][ $theme ] ) ) {
				if ( isset( $transient->response[ $theme ] ) ) {
					unset( $transient->response[ $theme ] );
				}

				$transient->no_update[ $theme ] = $response['no_update'][ $theme ];
			}
		}

		if ( isset( $response['translations'] ) ) {
			foreach ( $response['translations'] as $translation ) {
				if ( ! empty( $translation['package_base_url'] ) ) {
					$package_url = add_query_arg( array(
						'action'   => 'updates_get_translation_zip',
						'hmac'     => $this->hmac,
						'username' => $this->options['username'],
						'url'      => $this->options['url'],
						'type'     => $translation['type'],
						'slug'     => $translation['slug'],
						'version'  => $translation['version'],
						'locale'   => $translation['language'],
					), $translation['package_base_url'] );

					$translation['package'] = $package_url;
				}

				$transient->translations[] = $translation;
			}
		}

		return $transient;
	}

	public function check_plugins( $transient ) {

		// Sample transient output: https://www.dropbox.com/s/ggrs42wx4ohu8dn/Screenshot%202018-09-19%2020.39.33.png?dl=0

		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$query_plugins = $transient->checked;

		$translations = wp_get_installed_translations( 'plugins' );
		$locales      = array_unique( array_values( get_available_languages() ) );

		if ( ! empty( $this->options['username'] ) && ! empty( $this->options['url'] ) && ! empty( $this->options['key'] ) ) {
			$response = get_site_transient( 'cssigniter_updater_saved_plugin_updates' );

			// Make sure we invalidate API version 1.0 data that may be cached.
			if ( false !== $response && ! isset( $response['plugins'] ) ) {
				$response = false;
			}

			if ( false === $response ) {
				$response = array();

				$url = add_query_arg( array(
					'action'      => 'updates_get_plugin_updates',
					'hmac'        => $this->hmac,
					'username'    => $this->options['username'],
					'url'         => $this->options['url'],
					'api_version' => $this->api_version,
				), $this->endpoint_url );

				$api_response = wp_safe_remote_post( $url, array(
					'body' => array(
						'plugins'      => $query_plugins,
						'translations' => $translations,
						'locale'       => $locales,
					),
				) );

				if ( 200 === wp_remote_retrieve_response_code( $api_response ) ) {
					$json = json_decode( wp_remote_retrieve_body( $api_response ), true );
					if ( isset( $json['success'] ) && true === $json['success'] ) {
						$response = $json['data'];
					}
				}

				set_site_transient( 'cssigniter_updater_saved_plugin_updates', $response, $this->check_every );
			}
		}

		foreach ( $query_plugins as $plugin_path => $old_version ) {
			$parts = explode( '/', $plugin_path );
			$slug  = $parts[0];

			if ( isset( $response['response'][ $plugin_path ] ) ) {
				if ( isset( $transient->no_update[ $plugin_path ] ) ) {
					unset( $transient->no_update[ $plugin_path ] );
				}

				$entry         = $response['response'][ $plugin_path ];
				$entry['slug'] = $slug;
				$package_url   = '';

				if ( ! empty( $entry['package_base_url'] ) ) {
					$package_url = add_query_arg( array(
						'action'   => 'updates_get_plugin_zip',
						'hmac'     => $this->hmac,
						'username' => $this->options['username'],
						'url'      => $this->options['url'],
						'plugin'   => $slug,
						'version'  => $entry['new_version'],
					), $entry['package_base_url'] );

					$entry['package'] = $package_url;
				}

				$transient->response[ $plugin_path ] = (object) $entry;

			} elseif ( isset( $response['no_update'][ $plugin_path ] ) ) {
				if ( isset( $transient->response[ $plugin_path ] ) ) {
					unset( $transient->response[ $plugin_path ] );
				}

				$transient->no_update[ $plugin_path ] = (object) $response['no_update'][ $plugin_path ];
			}
		}

		if ( isset( $response['translations'] ) ) {
			foreach ( $response['translations'] as $translation ) {
				if ( ! empty( $translation['package_base_url'] ) ) {
					$package_url = add_query_arg( array(
						'action'   => 'updates_get_translation_zip',
						'hmac'     => $this->hmac,
						'username' => $this->options['username'],
						'url'      => $this->options['url'],
						'type'     => $translation['type'],
						'slug'     => $translation['slug'],
						'version'  => $translation['version'],
						'locale'   => $translation['language'],
					), $translation['package_base_url'] );

					$translation['package'] = $package_url;
				}

				$transient->translations[] = $translation;
			}
		}

		return $transient;
	}

	public function register_settings() {
		register_setting( 'cssigniter-updater-settings', 'cssigniter_updater_settings', array( $this, 'settings_validate' ) );
	}

	public function save_network_settings() {
		$settings = $this->settings_validate( isset( $_POST['cssigniter_updater_settings'] ) ? $_POST['cssigniter_updater_settings'] : array() );
		if ( is_multisite() ) {
			update_site_option( 'cssigniter_updater_settings', $settings );
		} else {
			update_option( 'cssigniter_updater_settings', $settings );
		}

		wp_safe_redirect( add_query_arg( array(
			'settings-updated' => 'true',
			'page'             => 'cssigniter-updater-settings',
		), network_admin_url( 'settings.php' ) ) );
		exit;
	}

	public function settings_validate( $settings ) {
		$new_settings = array();
		$settings     = (array) $settings;

		$settings = wp_parse_args( $settings, array(
			'username' => '',
			'key'      => '',
		) );

		$new_settings['username'] = sanitize_user( trim( $settings['username'] ) );
		$new_settings['key']      = sanitize_key( trim( $settings['key'] ) );

		return $new_settings;
	}

	public function updater_menu() {
		add_submenu_page( is_multisite() ? 'settings.php' : 'options-general.php',
			esc_html__( 'CSSIgniter Updater', 'cssigniter-updater' ),
			esc_html__( 'CSSIgniter Updater', 'cssigniter-updater' ),
			is_multisite() ? 'manage_network_options' : 'manage_options',
			'cssigniter-updater-settings',
			array( $this, 'plugin_options' )
		);
	}

	public function plugin_options() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'cssigniter-updater' ) );
		}

		$action = is_multisite() ? 'edit.php?action=cssigniter_updater_save_network_settings' : 'options.php';
		?>
		<div class="wrap">
			<h2><?php esc_html_e( 'CSSIgniter Updater - Settings', 'cssigniter-updater' ); ?></h2>
			<form method="post" action="<?php echo esc_attr( $action ); ?>">
				<?php settings_fields( 'cssigniter-updater-settings' ); ?>

				<table class="form-table">
					<tr valign="top">
						<th scope="row"><?php esc_html_e( 'Instructions', 'cssigniter-updater' ); ?></th>
						<td>
							<fieldset>
								<p class="description">
									<?php
										echo wp_kses(
											/* translators: %s is a url. */
											sprintf( __( 'Log into your CSSIgniter account, find the <strong>Utilities</strong> tab in the <strong>Downloads</strong> page, register your website, and provide the information below. For more detailed setup instructions, read the <a href="%s" target="_blank">related KB article</a>.', 'cssigniter-updater' ),
												'https://www.cssigniter.com/kb/how-to-set-up-automatic-theme-and-plugin-updates/'
											),
											array(
												'strong' => array(),
												'a'      => array(
													'href' => true,
													'target' => true,
												),
											)
										);
									?>
								</p>
							</fieldset>
						</td>
					</tr>

					<tr valign="top">
						<th scope="row"><label for="username"><?php esc_html_e( 'Site URL', 'cssigniter-updater' ); ?></label></th>
						<td>
							<fieldset>
								<input id="username" value="<?php echo esc_url( network_site_url() ); ?>" type="text" autocomplete="off" readonly="readonly" class="widefat">
								<p class="description"><?php echo wp_kses( __( 'Provide this when registering your website. The registered URL needs to be <strong>identical</strong> to this.', 'cssigniter-updater' ), array( 'strong' => array() ) ); ?></p>
							</fieldset>
						</td>
					</tr>

					<tr valign="top">
						<th scope="row"><label for="username"><?php esc_html_e( 'Username', 'cssigniter-updater' ); ?></label></th>
						<td>
							<fieldset>
								<input id="username" name="cssigniter_updater_settings[username]" value="<?php echo esc_attr( $this->options['username'] ); ?>" type="text" autocomplete="off" class="widefat">
								<p class="description"><?php echo wp_kses( __( 'This must be the <strong>username</strong> of your CSSIgniter account, not the email.', 'cssigniter-updater' ), array( 'strong' => array() ) ); ?></p>
							</fieldset>
						</td>
					</tr>

					<tr valign="top">
						<th scope="row"><label for="username"><?php esc_html_e( 'Key', 'cssigniter-updater' ); ?></label></th>
						<td>
							<fieldset>
								<input id="username" name="cssigniter_updater_settings[key]" value="<?php echo esc_attr( $this->options['key'] ); ?>" type="password" autocomplete="off" class="widefat">
								<p class="description"><?php echo wp_kses( __( 'Enter the <strong>key</strong> that was generated for your installation. This is <strong>not the same</strong> as your CSSIgniter <strong>password</strong>.', 'cssigniter-updater' ), array( 'strong' => array() ) ); ?></p>
							</fieldset>
						</td>
					</tr>
				</table>

				<p class="submit">
					<input type="submit" class="button-primary" name="cssigniter-updater-save" value="<?php esc_html_e( 'Save Changes', 'cssigniter-updater' ); ?>"/>
				</p>
			</form>
		<?php
	}


	public function notice_plugin_connection() {
		$message = '';
		$success = false;
		$classes = '';

		if ( empty( $this->options['username'] ) || empty( $this->options['url'] ) || empty( $this->options['key'] ) ) {
			$message = __( 'One or more fields are empty. You will <strong>not receive</strong> automatic CSSIgniter theme, plugin and language file updates.', 'cssigniter-updater' );
			$success = false;
		} else {

			$url = add_query_arg( array(
				'action'   => 'updates_check_connection',
				'hmac'     => $this->hash_hmac( $this->options['username'] . $this->options['url'] ),
				'username' => $this->options['username'],
				'url'      => $this->options['url'],
			), $this->endpoint_url );

			$response = wp_safe_remote_get( $url );

			if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
				$message = __( 'An error occurred while contacting CSSIgniter. Please try again later.', 'cssigniter-updater' );
				$success = false;
			} else {
				$json = json_decode( wp_remote_retrieve_body( $response ), true );
				if ( isset( $json['success'] ) ) {
					if ( true === $json['success'] ) {
						$message = __( 'You are all set to receive automatic CSSIgniter updates.', 'cssigniter-updater' );
						$success = true;

						$this->invalidate_transients();
					} else {
						$message = __( 'Invalid username, URL, or key. You will <strong>not receive</strong> automatic CSSIgniter theme, plugin and language file updates.', 'cssigniter-updater' );
						$success = false;
					}
				}
			}
		}

		if ( $success ) {
			$classes = 'notice notice-success';
		} else {
			$classes = 'notice notice-error';
		}


		if ( empty( $message ) ) {
			return;
		}

		?>
		<div class="<?php echo esc_attr( $classes ); ?>">
			<p>
				<?php echo wp_kses( $message, array(
					'strong' => array(),
				) ); ?>
			</p>
		</div>
		<?php
	}

	public function hash_hmac( $string ) {
		$hash = '';

		if ( ! empty( $this->options['key'] ) ) {
			$hash = hash_hmac( 'md5', $string, $this->options['key'] );
		}

		return $hash;
	}

	public function latest_theme_versions() {
		$versions = get_site_transient( 'cssigniter_updater_theme_versions_json' );

		if ( false === $versions ) {
			$request = wp_safe_remote_get( $this->json_url['themes'] );

			if ( is_wp_error( $request ) || ( isset( $request['response']['code'] ) && 200 !== $request['response']['code'] ) ) {
				return false;
			}

			$body = wp_remote_retrieve_body( $request );

			$versions = json_decode( $body, true );

			set_site_transient( 'cssigniter_updater_theme_versions_json', $versions, $this->check_every );
		}

		return $versions;
	}

	public function latest_plugin_versions() {
		$versions = get_site_transient( 'cssigniter_updater_plugin_versions_json' );

		if ( false === $versions ) {

			$request = wp_safe_remote_get( $this->json_url['plugins'] );

			if ( is_wp_error( $request ) || ( isset( $request['response']['code'] ) && 200 !== $request['response']['code'] ) ) {
				return false;
			}

			$body = wp_remote_retrieve_body( $request );

			$versions = json_decode( $body, true );

			set_site_transient( 'cssigniter_updater_plugin_versions_json', $versions, $this->check_every );
		}

		return $versions;
	}

	private function invalidate_transients() {
		delete_site_transient( 'cssigniter_updater_saved_theme_updates' );
		delete_site_transient( 'cssigniter_updater_saved_plugin_updates' );
		delete_site_transient( 'cssigniter_updater_theme_versions_json' );
		delete_site_transient( 'cssigniter_updater_plugin_versions_json' );
		delete_site_transient( 'update_plugins' );
		delete_site_transient( 'update_themes' );
	}

	public function disable_sslverify( $r, $url ) {
		$r['sslverify'] = false;
		$r['timeout']   = 30;
		return $r;
	}

}

add_action( 'plugins_loaded', 'cssigniter_updater_init' );

function cssigniter_updater_init() {
	$cssigniter_updater = new CSSIgniter_Updater();
	$cssigniter_updater->initialize();
}
