<?php
function zermatt_openweathermap_customize_register( $wp_customize ) {
	//
	// Weather
	//
	$wp_customize->add_section( 'theme_openweathermap', array(
		'title'       => esc_html_x( 'Weather Defaults', 'customizer section title', 'zermatt' ),
		'description' => wp_kses( __( 'The following settings set the defaults that the theme weather will use.', 'zermatt' ), zermatt_get_allowed_tags( 'guide' ) ),
	) );

	$wp_customize->add_setting( 'theme_openweathermap_api_key', array(
		'default'           => '',
		'sanitize_callback' => 'sanitize_text_field',
	) );
	$wp_customize->add_control( 'theme_openweathermap_api_key', array(
		'type'        => 'text',
		'section'     => 'theme_openweathermap',
		'label'       => esc_html__( 'API Key', 'zermatt' ),
		/* translators: %s is a URL to the OpenWeatherMap.org AppID page. */
		'description' => wp_kses( sprintf( __( 'In order to use the weather feature, you need an free API key from <a href="%s" target="_blank">OpenWeatherMap.org</a>', 'zermatt' ), 'http://openweathermap.org/appid' ), zermatt_get_allowed_tags( 'guide' ) ),
	) );

	$wp_customize->add_setting( 'theme_openweathermap_location_id', array(
		'default'           => zermatt_openweathermap_default_location_id(),
		'sanitize_callback' => 'sanitize_text_field',
	) );
	$wp_customize->add_control( 'theme_openweathermap_location_id', array(
		'type'        => 'text',
		'section'     => 'theme_openweathermap',
		'label'       => esc_html__( 'Location ID', 'zermatt' ),
		/* translators: %1$s is the URL to openweathermap.com. %2$s is an non-linked example URL. %3$s is a location ID. */
		'description' => wp_kses( sprintf( __( 'Enter the location ID number of your city. The location ID number can be found by visiting <a href="%1$s" target="_blank">OpenWeatherMap.org</a> and searching for your city. Once you are on your city\'s page, the location ID is the last numeric part of the URL. For example, the URL for London, UK is <code>%2$s</code> and the location ID is <code>%3$s</code>.', 'zermatt' ), 'http://openweathermap.org/', 'http://openweathermap.org/city/2643743', '2643743' ), zermatt_get_allowed_tags( 'guide' ) ),
	) );

	$wp_customize->add_setting( 'theme_openweathermap_units', array(
		'default'           => zermatt_openweathermap_default_units(),
		'sanitize_callback' => 'zermatt_sanitize_openweathermap_units',
	) );
	$wp_customize->add_control( 'theme_openweathermap_units', array(
		'type'    => 'select',
		'section' => 'theme_openweathermap',
		'label'   => esc_html__( 'Units', 'zermatt' ),
		'choices' => zermatt_openweathermap_get_units_choices(),
	) );
}
add_action( 'customize_register', 'zermatt_openweathermap_customize_register' );


/**
 * Returns a default location ID for OpenWeatherMap.
 *
 * @return string
 */
function zermatt_openweathermap_default_location_id() {
	return apply_filters( 'zermatt_openweathermap_default_location_id', '2657928' );
}

/**
 * Returns a default units system for OpenWeatherMap.
 *
 * @return string
 */
function zermatt_openweathermap_default_units() {
	return apply_filters( 'zermatt_openweathermap_default_units', 'metric' );
}

/**
 * Returns the valid unit systems for OpenWeatherMap.
 *
 * @return array
 */
function zermatt_openweathermap_get_units_choices() {
	return apply_filters( 'zermatt_openweathermap_units_choices', array(
		'metric'   => esc_html__( 'Celsius', 'zermatt' ),
		'imperial' => esc_html__( 'Fahrenheit', 'zermatt' ),
		'standard' => esc_html__( 'Kelvin', 'zermatt' ),
	) );
}

/**
 * Sanitizes a units' system for OpenWeatherMap.
 *
 * @param string $value
 *
 * @return string
 */
function zermatt_sanitize_openweathermap_units( $value ) {
	$choices = zermatt_openweathermap_get_units_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return zermatt_openweathermap_default_units();
}

/**
 * Returns a temperature symbol, depending on the units' system.
 *
 * @param string $units
 *
 * @return string
 */
function zermatt_openweathermap_get_temperature_unit_symbol( $units ) {
	$k = esc_html_x( 'K', 'Temperature unit symbol (Kelvin)', 'zermatt' );
	$f = esc_html_x( 'F', 'Temperature unit symbol (Fahrenheit)', 'zermatt' );
	$c = esc_html_x( 'C', 'Temperature unit symbol (Celsius)', 'zermatt' );

	$unit = 'C';
	switch ( $units ) {
		case 'standard':
			$unit = $k;
			break;
		case 'imperial':
			$unit = $f;
			break;
		case 'metric':
		default:
			$unit = $c;
			break;
	}

	return apply_filters( 'zermatt_openweathermap_temperature_unit_symbol', $unit, $units );
}

add_action( 'wp_ajax_zermatt_get_weather_conditions', 'zermatt_ajax_openweathermap_weather_conditions' );
add_action( 'wp_ajax_nopriv_zermatt_get_weather_conditions', 'zermatt_ajax_openweathermap_weather_conditions' );
function zermatt_ajax_openweathermap_weather_conditions() {
	$valid_nonce = check_ajax_referer( 'weather-check', 'weather_nonce', false );
	if ( false === $valid_nonce ) {
		$response = array(
			'error'  => true,
			'errors' => array( 'Invalid nonce' ),
			'data'   => new stdClass(),
		);

		wp_send_json( $response );
	}

	if ( ! isset( $_GET['location_id'] ) ) {
		$response = array(
			'error'  => true,
			'errors' => array( 'Missing location id.' ),
			'data'   => new stdClass(),
		);

		wp_send_json( $response );
	}

	$units       = false;
	$location_id = wp_kses( wp_unslash( $_GET['location_id'] ), 'strip' );

	if ( isset( $_GET['units'] ) ) { // WPCS: Input var okay.
		$units = zermatt_sanitize_openweathermap_units( wp_kses( wp_unslash( $_GET['units'] ), 'strip' ) );
	}

	$response = zermatt_get_weather_conditions( $location_id, $units, get_theme_mod( 'theme_openweathermap_api_key' ) );

	wp_send_json( $response );
}

/**
 * @param string      $location_id
 * @param string|bool $units
 * @param string|bool $api_key
 * @param object|null $weather_obj Use the specified request object as returned by zermatt_openweathermap_get_current_weather().
 *
 * @return array
 */
function zermatt_get_weather_conditions( $location_id, $units = false, $api_key = false, $weather_obj = null ) {

	if ( null === $weather_obj ) {
		$api_key = $api_key ? trim( $api_key ) : '';
		$units   = $units ? $units : get_theme_mod( 'theme_openweathermap_units', zermatt_openweathermap_default_units() );

		if ( empty( $api_key ) || empty( $location_id ) ) {
			$response = array(
				'error'  => true,
				'errors' => array( 'Missing API key or location.' ),
				'data'   => new stdClass(),
			);

			return $response;
		}

		$weather = zermatt_openweathermap_get_current_weather( $api_key, $location_id, $units );
	} else {
		$weather = $weather_obj;
	}


	if ( is_wp_error( $weather ) ) {
		$response = array(
			'error'  => true,
			'errors' => $weather->get_error_messages(),
			'data'   => new stdClass(),
		);
	} elseif ( ! empty( $weather ) && isset( $weather['response']['code'] ) && 200 === intval( $weather['response']['code'] ) ) {
		$response = array(
			'error'  => false,
			'errors' => array(),
			'data'   => json_decode( $weather['body'] ),
		);
	} else {
		$response = array(
			'error'  => true,
			'errors' => array( 'OpenWeatherData.org Error' ),
			'data'   => new stdClass(),
		);
	}

	return $response;
}

/**
 * Low level query to the openweathermap.org API for the current weather conditions.
 * Usually, you'll want to use zermatt_get_weather_conditions() instead.
 *
 * @param string      $api_key
 * @param string      $location_id
 * @param string|bool $units        May be 'metric' (Celsius), 'imperial' (Fahrenheit), or 'standard' (Kelvin).
 * @param bool        $bypass_cache Bypass all caches. Use for debugging only.
 *
 * @return WP_Error|array The response or WP_Error on failure.
 */
function zermatt_openweathermap_get_current_weather( $api_key, $location_id, $units = 'metric', $bypass_cache = false ) {
	$query_hash      = zermatt_openweathermap_get_query_hash( $api_key, $location_id, $units );
	$trans_name      = zermatt_openweathermap_get_transient_name( $query_hash );
	$cache_time      = apply_filters( 'zermatt_openweathermap_current_weather_query_cache_time', 20 * MINUTE_IN_SECONDS );
	$retry_time      = apply_filters( 'zermatt_openweathermap_current_weather_query_retry_time', 2 * MINUTE_IN_SECONDS ); // Retry after failure.
	$request_timeout = apply_filters( 'zermatt_openweathermap_current_weather_query_timeout', 30 );

	// This transient will never expire. Holds the last known good response for fallback.
	$noexp_name = zermatt_openweathermap_get_nonexpiring_transient_name( $query_hash );

	$api_url = 'https://api.openweathermap.org/data/2.5/weather';

	$response = get_transient( $trans_name );
	if ( false === $response || $bypass_cache ) {

		$api_params = array(
			'id'    => $location_id,
			'appid' => $api_key,
		);

		// If 'standard' then the 'units' parameter shouldn't be passed.
		if ( in_array( $units, array( 'metric', 'imperial' ), true ) ) {
			$api_params['units'] = $units;
		}

		$url = add_query_arg( $api_params, $api_url );
		$url = apply_filters( 'zermatt_openweathermap_current_weather_request_url', $url, $api_params, $api_url );

		$response = wp_safe_remote_get( $url, array(
			'timeout' => $request_timeout,
		) );


		if ( ! $bypass_cache ) {
			$noexp = (array) get_transient( $noexp_name );

			if ( ! is_wp_error( $response ) && ! empty( $response ) && isset( $response['response']['code'] ) && intval( $response['response']['code'] ) === 200 ) {
				$json = json_decode( $response['body'], true );

				if ( ! is_null( $json ) && ! empty( $json['weather'] ) && ! empty( $json['main'] ) ) {
					$noexp['last_good_response']  = $response;
					$noexp['last_good_timestamp'] = time();
				} else {
					$cache_time = $retry_time;
				}
			}

			if ( is_wp_error( $response ) ) {
				$noexp['last_fail_message']   = $response->get_error_messages();
				$noexp['last_fail_timestamp'] = time();

				$cache_time = $retry_time;

				if ( ! empty( $noexp['last_good_response'] ) ) {
					$response = $noexp['last_good_response'];
				}
			}

			// Cache indefinitely, both for a fallback to $trans_name, as well as debugging.
			set_transient( $noexp_name, $noexp, 0 );

			set_transient( $trans_name, $response, $cache_time );
		}
	}

	return $response;
}

function zermatt_openweathermap_get_query_hash( $api_key, $location_id, $units ) {
	$hash = md5( $api_key . $location_id . $units . 'current_weather' );

	return apply_filters( 'zermatt_openweathermap_query_hash', $hash, $api_key, $location_id, $units );
}

function zermatt_openweathermap_get_transient_name( $query_hash ) {
	$base_name = 'zermatt_openweathermap_current_weather_%s';
	$name      = sprintf( $base_name, $query_hash );

	return apply_filters( 'zermatt_openweathermap_transient_name', $name, $base_name, $query_hash );
}

function zermatt_openweathermap_get_nonexpiring_transient_name( $query_hash ) {
	$base_name = 'zermatt_openweathermap_current_weather_noexp_%s';
	$name      = sprintf( $base_name, $query_hash );

	return apply_filters( 'zermatt_openweathermap_nonexpiring_transient_name', $name, $base_name, $query_hash );
}